/***************************************************************************/
/* 		This code is part of WWW graber called pavuk		   */
/*		Copyright (c) 1997,1998,1999 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>

#ifdef HAVE_SYS_MODE_H
#include <sys/mode.h>
#endif

#include "config.h"
#include "tools.h"
#include "url.h"
#include "times.h"
#include "cookie.h"

static int cookies_num = 0;
static cookie_entry *cookies = NULL;
static cookie_entry *last_cookie = NULL;
static int cookie_changed = 0;
static time_t cookie_updated = 0L;

void cookie_deep_free(centry)
cookie_entry *centry;
{
	_free(centry->domain);
	_free(centry->path);
	_free(centry->name);
	_free(centry->value);
	_free(centry);
}

void cookie_free_all(centry)
cookie_entry *centry;
{
	cookie_entry *pce;

	for (pce = centry; pce ; pce = pce->next)
		cookie_deep_free(pce);
}

cookie_entry *cookie_get_last_entry(centry , num)
cookie_entry *centry;
int *num;
{
	if (num) *num = 0;

	if (!centry) return NULL;

	if (num) (*num) ++;
	while(centry->next)
	{
		centry = centry->next;
		if (num) (*num) ++;
	}

	return centry;
}

void cookie_remove_oldest_entry(centry , lastentry , num)
cookie_entry **centry;
cookie_entry **lastentry;
int *num;
{
	cookie_entry *last = NULL;

	if (*lastentry) last = *lastentry;
	else last = cookie_get_last_entry(*centry , NULL);

	if (last && last->prev)
	{
		if (*centry == last) (*centry) = last->next;
		if (last->prev)
			last->prev->next = last->next;
		if (last->next)
			last->next->prev = last->prev;
		if (*lastentry) (*lastentry) = last->prev;
		cookie_deep_free(last);
		if (num) (*num)--;
	}	
}

static int cookie_match(urlp , cookiep)
url *urlp;
cookie_entry *cookiep;
{
	int retv = TRUE;
	int cl , hl;

	retv &= (urlp->type == URLT_HTTP) || (urlp->type == URLT_HTTPS);

	if (cookiep->secure)
		retv &= (urlp->type == URLT_HTTPS);

	retv &= !strncmp(urlp->p.http.document , cookiep->path , strlen(cookiep->path));

	cl = strlen(cookiep->domain);
	hl = strlen(urlp->p.http.host);

	retv &= !strcasecmp(cookiep->domain , (cl > hl) ? urlp->p.http.host :
		(urlp->p.http.host + hl - cl));
	
	return retv;
}

static int cookie_eq(c1 , c2)
cookie_entry *c1;
cookie_entry *c2;
{
	return (!strcmp(c1->name , c2->name) && !strcmp(c1->domain , c2->domain) &&
		!strcmp(c1->path , c2->path));
}

static cookie_entry *cookie_parse_ns(line)
char *line;
{
	cookie_entry *centry;
	char *p;
	int i;
	char *ln;


	if (!line) return NULL;

	ln = new_string(line);

	centry = _malloc(sizeof(cookie_entry));

	centry->loaded = TRUE;
	centry->next = NULL;
	centry->prev = NULL;
	centry->path = NULL;
	centry->domain = NULL;
	centry->name = NULL;
	centry->value = NULL;

	p = strtokc(ln , '\t');

	for (i = 0 ; i < 7 ; i++)
	{
		if (!p)
		{
			cookie_deep_free(centry);
			_free(ln);
			return NULL;
		}
	
		switch(i)
		{
			case 0:
				centry->domain = new_string(p);
				break;
			case 1:
				centry->flag = (bool)strcasecmp(p , "FALSE");
				break;
			case 2:
				centry->path = new_string(p);
				break;
			case 3:
				centry->secure = (bool)strcasecmp(p , "FALSE");
				break;
			case 4:
				centry->expires = _atoi(p);
				break;
			case 5:
				centry->name = new_string(p);
				break;
			case 6:
				centry->value = new_string(p);
				break;
		}
		p = strtokc(NULL , '\t');
	}

	_free(ln);
	return centry;
}

static cookie_entry *cookie_read_file(fd)
bufio *fd;
{
	char line[5000];
	cookie_entry *centry;
	cookie_entry *retv = NULL;
	cookie_entry *prev = NULL;

	while(bufio_readln(fd , line , sizeof(line)) > 0)
	{
		if (line[0] == '#') continue;

		strip_nl(line);

		if (*line)
		{

			if ((centry = cookie_parse_ns(line)))
			{
				if (!retv)
					retv = centry;
				else
				{
					prev->next = centry;
					centry->prev = prev;
				}
				prev = centry;
			}
			else
				xprintf(1 , gettext("Unable to parse : %s\n") , line);
		}
	}

	return retv;
}



cookie_entry *cookie_read_ns(filename)
char *filename;
{
	bufio *fd;
	cookie_entry *retv = NULL;

	LOCK_COOKIES
	cookie_free_all(cookies);

	if (!(fd = bufio_open(filename , O_BINARY | O_RDONLY)))
	{
		xperror(filename);
		UNLOCK_COOKIES
		return NULL;
	}

	if (_flock(bufio_getfd(fd) , filename ,  O_BINARY | O_RDONLY , FALSE))
	{
		xperror(filename);
		UNLOCK_COOKIES
		return NULL;
	}

	retv = cookie_read_file(fd);

	cookie_updated = time(NULL);

	_funlock(bufio_getfd(fd));
	bufio_close(fd);

	cookies = retv;
	last_cookie = cookie_get_last_entry(cookies , &cookies_num);
	UNLOCK_COOKIES
	return retv;
}

static int cookie_write_file(fd)
bufio *fd;
{
	char line[5000];
	cookie_entry *centry;
	time_t t = time(NULL);

	ftruncate(bufio_getfd(fd) , 0);
	lseek(bufio_getfd(fd) , 0 , SEEK_SET);
	bufio_reset(fd);

	for (centry = cookies ; centry ; centry = centry->next)
	{
		if (!centry->expires || centry->expires > t)
		{
			sprintf(line , "%s\t%s\t%s\t%s\t%ld\t%s\t%s\n" , 
				centry->domain , 
				centry->flag ? "TRUE" : "FALSE" ,
				centry->path ,
				centry->secure ? "TRUE" : "FALSE" ,
				centry->expires , centry->name ,
				centry->value);
			bufio_write(fd , line , strlen(line));
		}
	}

	return 0;
}

int cookie_write_ns(filename)
char *filename;
{
	bufio *fd;

	LOCK_COOKIES
	if (!(fd = bufio_copen(filename , O_BINARY | O_WRONLY | O_CREAT , 
		S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)))
	{
		xperror(filename);
		UNLOCK_COOKIES
		return -1;
	}

	if (_flock(bufio_getfd(fd) , filename , O_BINARY | O_WRONLY | O_CREAT , FALSE))
	{
		xperror(filename);
		UNLOCK_COOKIES
		return -1;
	}

	cookie_write_file(fd);

	_funlock(bufio_getfd(fd));
	bufio_close(fd);

	UNLOCK_COOKIES
	return 0;
}

static void cookie_sync(centry)
cookie_entry *centry;
{
	cookie_entry *cmem;
	cookie_entry *cfile;
	cookie_entry **uentry;
	cookie_entry *p;
	bool found;
	time_t t = time(NULL);

	for (cfile = centry ; cfile ; cfile = p)
	{
		found = FALSE;
		p = cfile->next;

		for (cmem = cookies , uentry = &cookies ;
			cmem ; uentry = &cmem->next , cmem = cmem->next)
		{
			if (cookie_eq(cfile , cmem))
			{
				found = TRUE;
				if ((cmem->expires && cmem->expires < t  && !cmem->loaded) 
					|| (cmem->loaded && !strcmp(cmem->value , cfile->value)))
				{
					cookie_deep_free(cfile);
				}
				else
				{
					*uentry = cfile;
					cfile->next = cmem->next;
					cfile->prev = cmem->prev;
					if (cmem->next) cmem->next->prev = cfile;
					cookie_deep_free(cmem);
				}
				break;

			}
		}

		if (!found)
		{
			cfile->next = cookies;
			cfile->prev = NULL;
			if (cookies) cookies->prev = cfile;
			cookies = cfile;
			cookies_num ++;
			if (cookies_num &&
				cookies_num > cfg.cookies_max)
			{
				cookie_remove_oldest_entry(&cookies , &last_cookie, &cookies_num);
			}
		}
	}
}

int cookie_update_ns(force)
int force;
{
	struct stat estat;
	cookie_entry *centry = NULL;
	bufio *fd;

	if (!cookie_changed) return 0;

	if (!force && cookie_changed < 10) return 0;

	if (!cfg.cookie_file) return -1;

	LOCK_COOKIES
	xprintf(1 , gettext("Updating cookie file\n"));

	if (!(fd = bufio_copen(cfg.cookie_file , O_BINARY | O_RDWR | O_CREAT , 
		S_IRUSR | S_IRGRP | S_IROTH | S_IWUSR)))
	{
		xperror(cfg.cookie_file);
		UNLOCK_COOKIES
		return -1;
	}

	if (_flock(bufio_getfd(fd) , cfg.cookie_file , O_BINARY | O_WRONLY | O_CREAT , FALSE))
	{
		xperror(cfg.cookie_file);
		bufio_close(fd);
		UNLOCK_COOKIES
		return -1;
	}

	if (fstat(bufio_getfd(fd) , &estat))
	{
		xperror(cfg.cookie_file);		
		_funlock(bufio_getfd(fd));
		bufio_close(fd);
		UNLOCK_COOKIES
		return -1;
	}

	if (estat.st_mtime > cookie_updated)
	{
		xprintf(1 , gettext("Cookie file has changed - > synchronizing\n"));
		centry = cookie_read_file(fd);
		cookie_sync(centry);
	}

	cookie_write_file(fd);

	cookie_changed = 0;
	cookie_updated = time(NULL);

	_funlock(bufio_getfd(fd));
	bufio_close(fd);

	UNLOCK_COOKIES
	return 0;
}


char *cookie_get_field(urlp)
url *urlp;
{
	int sel = 0;
	char *retv;
	cookie_entry *centry;
	time_t t = time(NULL);

	LOCK_COOKIES
	retv = new_string("Cookie:");

	for (centry = cookies ; centry ; centry = centry->next)
	{
		if ((!centry->expires || centry->expires > t) &&
			cookie_match(urlp , centry))
		{
			retv = _realloc(retv , strlen(retv) +
						strlen(centry->name) +
						strlen(centry->value) +
						6);

			if (!sel) strcat(retv , " ");
			else strcat(retv , "; ");

			strcat(retv , centry->name);
			strcat(retv , "=");
			strcat(retv , centry->value);
			
			sel ++;
		}
	}

	if (sel)
		strcat(retv , "\r\n");
	else
		_free(retv);

	UNLOCK_COOKIES
	return retv;
}


void cookie_insert_field(field , urlp)
char *field;
url *urlp;
{
	int slen,plen;
	char *p,*p1;
	char *pom = new_string(field);
	cookie_entry *centry;
	cookie_entry *pcentry;
	cookie_entry *uentry;
	bool found = FALSE;
	char **pp;

	LOCK_COOKIES
	centry = _malloc(sizeof(cookie_entry));

	centry->next = NULL;
	centry->prev = NULL;
	centry->loaded = FALSE;
	centry->domain = NULL;
	centry->flag = TRUE;
	centry->path = NULL;
	centry->expires = 0;
	centry->secure = FALSE;
	centry->name = NULL;
	centry->value = NULL;

	p = strtokc(pom , ';');
	while (p)
	{
		while(isspace(*p)) p++;

		if (!strncasecmp(p , "expires=" , 8))
		{
			centry->expires = scntime(p+8);
		}
		else if (!strncasecmp(p , "path=" , 5))
		{
			centry->path = new_string(p+5);
		}
		else if (!strncasecmp(p , "domain=" , 7))
		{
			centry->domain = new_string(p+7);
		}
		else if (!strcasecmp(p , "secure"))
		{
			centry->secure = TRUE;
		}
		else
		{
			p1 = strchr(p , '=');
			if (p1)
			{
				centry->name = new_n_string(p , (p1 - p));
				centry->value = new_string(p1 + 1);
			}
		}

		p = strtokc(NULL , ';');
	}

	_free(pom);

	if (!centry->value || !centry->name)
	{
		cookie_deep_free(centry);
		centry = NULL;
	}
	
	if (centry)
	{
		if (!centry->path)
			centry->path = new_string(url_get_path(urlp));

		if (!centry->domain)
			centry->domain = new_string(url_get_site(urlp));
		lowerstr(centry->domain);

		/*** check if cookie is set for source domain ***/
		if (cfg.cookie_check_domain)
		{
			p = url_get_site(urlp);
			plen = strlen(p);
			slen = strlen(centry->domain);
			if (plen < slen ||
			    strcmp(centry->domain , p + (plen - slen)))
			{
				xprintf(1 , gettext("Server %s is trying to set cookie for %s domain\n") , p , centry->domain);
				cookie_deep_free(centry);
				UNLOCK_COOKIES
				return;
			}
		}

		for (pp = cfg.cookies_disabled_domains ; pp && *pp ; pp++)
		{
			if (!strcasecmp(*pp , centry->domain))
			{
				xprintf(1 , gettext("Removing cookie from disabled domain %s\n") , centry->domain);
				cookie_deep_free(centry);
				UNLOCK_COOKIES
				return;				
			}
		}

		for (pcentry = cookies , uentry = NULL ;
			pcentry ; uentry = pcentry , pcentry = pcentry->next)
		{
			if (cookie_eq(centry , pcentry))
			{
				found = TRUE;
				if (centry->expires && centry->expires < time(NULL))
				{
					pcentry->expires = centry->expires;
					cookie_deep_free(centry);
					cookie_changed ++;
				}
				else
				{
					if (centry->expires != pcentry->expires ||
						strcmp(centry->value , pcentry->value))
					{
						if (uentry)
							uentry->next = centry;
						else
							cookies = centry;
						centry->next = pcentry->next;
						centry->prev = uentry;
						if (centry->next)
							centry->next->prev = centry;
						if (last_cookie == pcentry) 
							last_cookie = centry;
						cookie_deep_free(pcentry);
						cookie_changed ++;
						break;
					}
				}
			}
		}
		if (!found)
		{
			if (centry->expires && centry->expires < time(NULL))
				cookie_deep_free(centry);
			else
			{
				centry->next = cookies;
				if (cookies) cookies->prev = centry;
				if (!last_cookie) last_cookie = centry;
				cookies = centry;
				cookie_changed ++;
				cookies_num ++;
				if (cfg.cookies_max &&
					cookies_num > cfg.cookies_max)
				{
					cookie_remove_oldest_entry(&cookies , &last_cookie, &cookies_num);
				}
			}
		}
	}
	UNLOCK_COOKIES
}

