import datetime
import logging

from storm.monitoring.sensor.api import sensor
from storm.monitoring.sensor.api import service_sensor
from storm.monitoring.sensor.api import metrics
from storm.monitoring.sensor.api import services
from storm.monitoring.sensor.api import measure
from storm.monitoring.sensor.service.procmem import procmem_check
from storm.monitoring.sensor.api import measures

import commands

class ProcMemSensor(service_sensor.ServiceSensor):

    def __init__(self, hostname, service_type):
        self.logger = logging.getLogger("storm_sensor.procmem_sensor")
        self.logger.info("creating an instance of ProcMemSensor")

        self.hostname = hostname
        self.metric_type = metrics.Metrics().proc_mem
        if service_type not in services.Services().get_services():
            msg = 'The specified service type %s does not exist' % str(val)
            raise services.ServicesError(msg)
        self.service_type = service_type
        self.service_name = services.Services().get_service_names_for_service_type(self.service_type)
        self.timestamp = ''
        self.measures = measures.Measures()

    def get_hostname(self):
        """Return hostname information"""
        self.logger.info("getting hostname")
        return self.hostname

    def __get_pid__(self):
        """Return True or False"""
        cmd = '/sbin/service ' + self.service_name + ' status'
        output=commands.getstatusoutput(cmd)
        if output[0] == 0:
            if 'is running' in output[1] or 'is RUNNING' in output[1] or 'OK' in output[1]:
                return output[1].split(')')[0].split('pid ')[1].split()[0]
        msg = 'The specified service type %s is not running. ' \
            % str(self.service_name)
        msg += 'Please verify that the service is installed.'
        raise services.ServicesError(msg)

    def run(self):
        self.logger.info("doing run")
        """Run sensor and save data"""
        self.timestamp = datetime.datetime.now()

        if not self.is_service_running():
            msg = 'The specified service type %s is not running. ' \
                % str(self.service_name)
            msg += 'Please verify that the service is installed.'
            raise services.ServicesError(msg)

        output = procmem_check.PsRss(int(self.__get_pid__()))
        self.measures.add_measure(output.get_rss_mem())
            
        self.get_measures()

    def get_measures(self):
        """Return measures"""
        self.logger.info("getting measures")
        return self.measures

    def get_timestamp(self):
        """Return timestamp"""
        self.logger.info("getting timestamp")
        return self.timestamp

    def get_metric_type(self):
        """Return metric type"""
        self.logger.info("getting metric type")
        return self.metric_type

    def get_storm_service_types(self):
        """Return storm service types"""
        self.logger.info("getting storm service type")
        return self.service_type

    def is_storm_service(self):
        """Return True or False"""
        self.logger.info("checking storm service")
        return services.Services().is_service(self.service_types)

    def is_service_running(self):
        """Return True or False"""
        self.logger.info("checking service running")
        cmd = '/sbin/service ' + self.service_name + ' status'
        output=commands.getstatusoutput(cmd)
        if output[0] == 0:
            if 'is running...' in output[1] or 'is RUNNING' in output[1] or 'OK' in output[1]:
                return True
            else:
                return False
        return False
