import ldap, ldap.schema, logging, re, time
from ldap.filter import filter_format
from arcnagios.ldaputils import str2dn
from arcnagios import arcinfosys, arcutils, vomsutils
from arcnagios.utils import counted_noun, Statdict
from arcnagios.nagutils import NagiosPlugin, \
	ServiceWARNING, ServiceCRITICAL, ServiceUNKNOWN, \
	OK, WARNING, CRITICAL, UNKNOWN, status_by_name
from arcnagios.confargparse import UsageError
from arcnagios.arcinfosys import MdsServiceLdap, egiis_reg_status_to_string, \
	NorduGridCluster, NorduGridQueue, \
	glue_class_map, GlueCE, GlueCluster, GlueSubCluster
from arcnagios import glue2
from arcnagios import ldapschema
from arcnagios.ldaputils import LDAPObject, LDAPValidationError, \
				is_proper_subdn, is_immediate_subdn

# Backward compatibility.
if not hasattr(__builtins__, 'any'):
    from arcnagios.compat import any

class RelationTrackerEntry(object):
    def __init__(self):
	self.seen_in = set([])	# List of DNs.
	self.referred_by = {}	# From GLUE2ForeignKey to list of DNs.

class RelationTracker(object):
    def __init__(self):
	self._d = {}

    def _ensure(self, a, k):
	if not (a, k) in self._d:
	    self._d[(a, k)] = RelationTrackerEntry()
	return self._d[(a, k)]

    def seen_pk(self, a, k):
	return (a, k) in self._d and self._d[(a, k)].seen_in != []

    def add_pk(self, a, k, dn):
	self._ensure(a, k).seen_in.add(dn)

    def add_ref(self, a, k, fk, dn):
	ent = self._ensure(a, k)
	if not fk in ent.referred_by:
	    ent.referred_by[fk] = []
	ent.referred_by[fk].append(dn)

    def iteritems(self):
	return self._d.iteritems()

def first_or_none(x):
    if isinstance(x, list):
	return x != [] and x[0]
    else:
	return x

class Glue_Validator(object):

    def __init__(self, subject, shortname, log):
	self.subject = subject
	self.shortname = shortname
	self.error_count = 0
	self.log = log

    def compare_attribute(self, attrname, arcval, mapping = None, soft = False):
	def normalize(x):
	    # Work around missing single-valuedness restrictions.
	    if isinstance(x, list):
		if   len(x) == 0: return None
		elif len(x) == 1: return x[0]
		else:             return set(x)
	    else:
		return x
	glueval = normalize(getattr(self.subject, attrname))
	arcval = normalize(arcval)
	if arcval and not mapping is None:
	    if not arcval in mapping:
		# Some of values for nordugrid-queue-status like "inactive,
		# gridftpd is down" seems more informative than part of a
		# strict enumeration.
		self.log.warning('[%s].%s: Not comparing %r due to unknown '
				 'mapping.'%(self.shortname, attrname, arcval))
		return
	    arcval = mapping[arcval]
	if glueval != arcval:
	    if soft:
		self.log.warning('[%s].%s: %s != %s'
				 %(self.shortname, attrname, glueval, arcval))
	    else:
		self.log.error('[%s].%s: %s != %s'
			       %(self.shortname, attrname, glueval, arcval))
		self.error_count += 1
	else:
	    self.log.debug('Checked %s: %s'%(attrname, glueval))

class GlueCE_Validator(Glue_Validator):

    def compare_with(self, cluster, queue):
	ck = self.compare_attribute
	#ck('GlueCEUniqueID', '%s?queue=%s'%(cluster.contactstring, queue.name))
	ck('GlueCEName', queue.name)
	# GlueCEInformationServiceURL
	ck('GlueCEInfoLRMSType', cluster.lrms_type)
	ck('GlueCEInfoLRMSVersion', cluster.lrms_version)
	ck('GlueCEInfoHostName', cluster.name)
	# GlueCEInfoGatekeeperPort - irrelevant for ARC
	# GlueCEInfoJobmanager - irrelevant for ARC
	ck('GlueCEInfoContactString',
	   '%s?queue=%s'%(cluster.contactstring, queue.name))
	#ck('GlueCEInfoTotalCPUs', cluster.totalcpus) - deprecated in Glue
	# GlueCEInfoApplicationDir - not in ARC
	# GlueCEInfoDataDir - not in ARC
	# GlueCEDefaultSE - not in ARC
	ck('GlueCEStateStatus', queue.status,
	   mapping = {'active': 'Production', 'inactive': 'Closed'})
	# ck('GlueCEStateRunningJobs', queue.running)

	# Skip running state since we may have fetched the LDAP entries at
	# different times.
	#nwait = queue.gridqueued + queue.localqueued + queue.prelrmsqueued
	#ck('GlueCEStateWaitingJobs', nwait)
	#ck('GlueCEStateTotaljobs', nwait + queue.running)

	# GlueCEStateEstimatedResponseTime - not in ARC
	# GlueCEStateWorstResponseTime - not in ARC
	# GlueCEStateFreeJobSlots - skip running state
	# GlueCEStateFreeCPUs     - skip running state
	# GlueCEPolicyMaxWallTime - not yet in ARC
	ck('GlueCEPolicyMaxCPUTime', queue.maxcputime)
	if not queue.maxqueuable is None: # FIXME: Likely a translation issue.
	    ck('GlueCEPolicyMaxTotalJobs', queue.maxqueuable)
	ck('GlueCEPolicyMaxRunningJobs', queue.maxrunning)
	# GlueCEPolicyPriority - not in ARC
	# GlueCEPolicyAssignedJobSlots - ambiguous mapping
	if getattr(cluster, 'acl', None): # FIXME: Likely a translation issue.
	    ck('GlueCEAccessControlBaseRule', cluster.acl)

class GlueCluster_Validator(Glue_Validator):

    def compare_with(self, cluster):
	ck = self.compare_attribute
	ck('GlueClusterUniqueID', cluster.name)
	ck('GlueClusterName', cluster.aliasname, soft = True)
	# GlueClusterTmpDir - not in ARC
	# GlueClusterWNTmpDir - not in ARC

class GlueSubCluster_Validator(Glue_Validator):

    def compare_with(self, cluster, corq):
	ck = self.compare_attribute
	ck('GlueSubClusterUniqueID', corq.name)
	ck('GlueSubClusterName', corq.name)
	ck('GlueSubClusterPhysicalCPUs', corq.totalcpus)
	# GlueSubClusterLogicalCPUs - not in ARC
	# GlueSubClusterLocation* - not in ARC
	# GlueHostOperatingSystem* - not in ARC
	# GlueHostProcessor* - not in ARC
	ck('GlueHostRAMSize', corq.nodememory)
	# GlueHostVirtualSize - not in ARC
	ck('GlueHostNetworkAdapterOutboundIP', 'outbound' in cluster.nodeaccess)
	ck('GlueHostNetworkAdapterInboundIP', 'inbound' in cluster.nodeaccess)
	ck('GlueHostArchitecturePlatformType', corq.architecture)
	# GlueHostBenchmarkS[IF]00 - Are only the 2000 benchmarks mapped?
	ck('GlueHostApplicationSoftwareRunTimeEnvironment',
	   cluster.runtimeenvironment)

class ARCInfosysProbe(NagiosPlugin, vomsutils.NagiosPluginVomsMixin):

    main_config_section = 'arcinfosys'

    def __init__(self):
	NagiosPlugin.__init__(self, default_port = 2135)

	ap = self.argparser
	ap.add_argument('--ldap-uri', dest = 'ldap_uri',
		help = 'LDAP URI of the infosystem to query.')
	ap.add_argument('--ldap-basedn', dest = 'ldap_basedn',
		help = 'Base DN to query if non-standard.')
	ap.add_argument('-t', '--timeout', dest = 'timeout', default = 300,
		help = 'Overall timeout.')
	asp = ap.add_subparsers(dest = 'metric_name')

	ap = asp.add_parser('glue2')
	ap.add_argument('--glue2-schema',
		default = '/etc/ldap/schema/GLUE20.schema',
		help = 'Path of GLUE 2.0 LDAP schema. '
		       'Default is /etc/ldap/schema/GLUE20.schema.')
	ap.add_argument('--if-dependent-schema',
		type = status_by_name, default = WARNING,
		metavar = 'NAGIOS-STATUS',
		help = 'Nagios status to report if LDAP schema had to be '
		       'fetched from the tested server.')
	ap.add_argument('--warn-if-missing',
		default = 'GLUE2AdminDomain,GLUE2Service,GLUE2Endpoint',
		metavar = 'OBJECTCLASSES',
		help = 'Report warning if there are no entries of the given '
		       'comma-separated list of LDAP objectclasses. '
		       'Default: GLUE2AdminDomain,GLUE2Service,GLUE2Endpoint')
	ap.add_argument('--critical-if-missing', default = '',
		metavar = 'OBJECTCLASSES',
		help = 'Report critical if there are no entries of the given '
		       'comma-separated list of LDAP objectclasses. '
		       'Empty by default.')
	ap.add_argument('--hierarchical-foreign-keys',
		metavar = 'FOREIGN-KEY-NAMES', default = '',
		help = 'A comma-separating list of foreign key attribute types '
		       'which should be reflected in the DIT.')
	ap.add_argument('--hierarchical-aggregates',
		default = False, action = 'store_true',
		help = 'Require that all foreign keys which represent '
		       'aggregation or composition are reflected in the '
		       'DIT.')

	ap = asp.add_parser('aris')
	ap.add_argument('--if-no-clusters', dest = 'if_no_clusters',
		type = status_by_name, default = WARNING,
		metavar = 'NAGIOS-STATUS',
		help = 'Nagios status to report if no cluster is found.')
	ap.add_argument('--if-no-queues', dest = 'if_no_queues',
		type = status_by_name, default = WARNING,
		metavar = 'NAGIOS-STATUS',
		help = 'Nagios status to report if a cluster has no queues.')
	ap.add_argument('--cluster', dest = 'clusters', action = 'append',
		default = [],
		metavar = 'CLUSTER-NAME',
		help = 'Pass one or more times to check specific clusters.')
	ap.add_argument('--cluster-test', dest = 'cluster_tests',
		action = 'append', default = [],
		metavar = 'TESTNAME',
		help = 'Enable a custom test to run against nordugrid-cluster '
		       'entries.')
	ap.add_argument('--queue-test', dest = 'queue_tests',
		action = 'append', default = [],
		metavar = 'TESTNAME',
		help = 'Enable a custom test to run against nordugrid-queue '
		       'entries.')
	ap.add_argument('--enable-glue',
		action = 'store_true', default = False,
		help = 'Enable loading and schema-checks of the Glue schema '
		       'entries if present.')
	ap.add_argument('--compare-glue',
		action = 'store_true', default = False,
		help = 'Enable comparison of Glue entries with ARC. '
		       'Only a limited set of attributes are compared. '
		       'Implies --enable-glue.')
	ap.add_argument('--check-contact',
		action = 'store_true', default = False,
		help = 'Try to list the nordugrid-cluster-contactstring URLs. '
		       'This requires a proxy certificate.')

	ap = asp.add_parser('egiis')
	ap.add_argument('--index-name', dest = 'index_name',
		help = 'The name of the information index to query.')

    def parse_args(self, args):
	NagiosPlugin.parse_args(self, args)
	if self.opts.metric_name == 'egiis':
	    if not self.opts.ldap_basedn and not self.opts.index_name:
		raise UsageError(
			'Either --ldap-basedn or --index-name is required.')
	if not self.opts.ldap_uri:
	    if not self.opts.host:
		raise UsageError('Either --ldap-uri or -H must be specified.')
	    self.opts.ldap_uri = 'ldap://%s:%d'%(self.opts.host, self.opts.port)
	if self.opts.metric_name == 'aris':
	    if self.opts.compare_glue:
		self.opts.enable_glue = True

    def check(self):
	self.log.debug('Using LDAP URI %s.'%self.opts.ldap_uri)
	self.time_limit = time.time() + self.opts.timeout
	return getattr(self, 'check_' + self.opts.metric_name)()

    @property
    def time_left(self):
	return self.time_limit - time.time()

    def search_s(self, basedn, scope, *args, **kwargs):
	"""Customized LDAP search with timeout and Nagios error reporting."""

	self.log.debug('Searching %s.'%basedn)
	if self.time_left <= 0:
	    raise ServiceCRITICAL('Timeout before LDAP search.')
	try:
	    return self.lconn.search_st(basedn, scope, timeout = self.time_left,
					*args, **kwargs)
	except ldap.TIMEOUT:
	    raise ServiceCRITICAL('Timeout during LDAP search.')
	except ldap.NO_SUCH_OBJECT:
	    return []
	except ldap.LDAPError, xc:
	    self.log.error('LDAP details: basedn = %s, scope = %d',
			   basedn, scope)
	    self.log.error('LDAP error: %s'%xc)
	    raise ServiceCRITICAL('LDAP Search failed.')

    def fetch_subschema(self):
	"""Fetch the subschema for the given connection and return the
	corresponding `ldap.schema.SubSchema` object."""

	# TODO: Consider caching the subschema for a period of time, since it
	# can be big.

	self.log.debug('Fetching subschema.')
	sr = self.search_s('cn=subschema', ldap.SCOPE_BASE,
			   attrlist = ['+', '*'])
	if len(sr) != 1:
	    raise ServiceCRITICAL('Could not fetch subschema.')
	subschema_subentry = sr[0][1]
	return ldap.schema.SubSchema(subschema_subentry)

    def maybe_dump_obj(self, obj, name):
	# Dump the entry if debugging is enabled.
	if self.log.getEffectiveLevel() >= logging.DEBUG:
	    self.log.debug('Dump of %s:'%name)
	    for k, v in vars(obj).iteritems():
		if isinstance(v, list) and len(v) > 4:
		    v = v[0:4] + ['...']
		self.log.debug('  %s: %r'%(k, v))

    def custom_verify_regex(self, section, obj):
	variable = self.config.get(section, 'variable')
	values = getattr(obj, variable, [])
	if not isinstance(values, list):
	    values = [values]
	for (code, pfx) in [(CRITICAL, 'critical'), (WARNING, 'warning')]:
	    if self.config.has_option(section, pfx + '.pattern'):
		sp = self.config.get(section, pfx + '.pattern')
		p = re.compile(sp)
		if not any(re.match(p, value) for value in values):
		    if self.config.has_option(section, pfx + '.message'):
			msg = self.config.get(section, pfx + '.message')
		    else:
			msg = '%s did not match %s'%(variable, sp)
		    return code, msg
	return OK, None

    def custom_verify_limit(self, section, obj):
	expr = self.config.get(section, 'value')
	x = eval(expr, vars(obj))
	for (code, pfx) in [(CRITICAL, 'critical'), (WARNING, 'warning')]:
	    msg = None
	    if self.config.has_option(section, pfx + '.message'):
		msg = self.config.get(section, pfx + '.message')
	    if self.config.has_option(section, pfx + '.min'):
		x_min = self.config.getfloat(section, pfx + '.min')
		if x < x_min:
		    return code, (msg or '%s = %s is below %s limit %s.'
					 % (expr, x, pfx, x_min))
	    if self.config.has_option(section, pfx + '.max'):
		x_max = self.config.getfloat(section, pfx + '.max')
		if x > x_max:
		    return code, (msg or '%s = %s is above %s limit %s'
					 % (expr, x, pfx, x_max))
	return OK, None

    def custom_verify_obj(self, obj, tests):
	# Custom tests.
	for test in tests or []:
	    section = 'arcinfosys.aris.%s'%test
	    if not self.config.has_section(section):
		raise ServiceUNKNOWN('Missing section %s to define '
				     'the test %s.'%(section, test))
	    if not self.config.has_option(section, 'type'):
		raise ServiceUNKNOWN('The type variable is missing is %s.'
				     %section)
	    type = self.config.get(section, 'type')
	    if type == 'limit':
		code, msg = self.custom_verify_limit(section, obj)
	    elif type == 'regex':
		code, msg = self.custom_verify_regex(section, obj)
	    else:
		raise ServiceUNKNOWN('Unhandled type %s in %s.'%(type, section))
	    if code:
		self.log.error(msg)
		self.nagios_report.update_status_code(code)

    def verify_nordugrid_cluster(self, dn, ent):
	"""Validate and do custom checks on a nordugrid-cluster entry."""

	try:
	    cluster = NorduGridCluster(self.subschema, dn, ent)
	except LDAPValidationError, xc:
	    self.log.error('Validation of cluster entry %s failed.'%dn)
	    self.log.error(str(xc))
	    self.nagios_report.update_status_code(CRITICAL)
	    return None
	self.maybe_dump_obj(cluster, 'nordugrid-cluster entry')

	tests = self.opts.cluster_tests
	if not tests:
	    for setting in [
		    'cluster_tests[%s]'%cluster.name,
		    'cluster_tests[%s]'%self.opts.host,
		]:
		if self.config.has_option('arcinfosys.aris', setting):
		    raw_tests = self.config.get('arcinfosys.aris', setting)
		    tests = [test.strip() for test in raw_tests.split(',')]
		    break
	self.custom_verify_obj(cluster, tests)
	return cluster

    def verify_nordugrid_queue(self, cluster, dn, ent):
	"""Validate and do custom checks on a nordugrid-queue entry."""

	try:
	    queue = NorduGridQueue(self.subschema, dn, ent)
	except LDAPValidationError, xc:
	    self.log.error('Validation of queue entry %s failed.'%dn)
	    self.log.error(str(xc))
	    self.nagios_report.update_status(CRITICAL)
	    return None
	self.maybe_dump_obj(queue, 'nordugrid-queue entry')

	tests = self.opts.queue_tests
	if not tests:
	    for setting in [
		    'queue_tests[%s/%s]'%(cluster.name, queue.name),
		    'queue_tests[%s]'%queue.name,
		    'queue_tests[%s]'%self.opts.host,
		]:
		if self.config.has_option('arcinfosys.aris', setting):
		    raw_tests = self.config.get('arcinfosys.aris', setting)
		    tests = [test.strip() for test in raw_tests.split(',')]
		    break
	self.custom_verify_obj(queue, tests)
	return queue

    def verify_glue_entry(self, dn, ent, clusters, queues):

	# Schema-check and construct an object representation.
	obj = None
	for objcls in ent['objectClass']:
	    if objcls in glue_class_map:
		try:
		    obj = glue_class_map[objcls](self.subschema, dn, ent)
		    self.log.debug('Schema-checked %s.'%dn)
		except LDAPValidationError, xc:
		    self.log.error(
			    'Schema-check failed for %s: %s'%(dn, xc))
		    self.nagios_report.update_status_code(CRITICAL)
		break
	if obj is None or not self.opts.compare_glue:
	    return obj

	def get_cluster(cluster_name):
	    if not cluster_name in clusters:
		self.log.error('No cluster %s corresponding to %s.'
			% (cluster_name, obj.structural_objectclass))
		self.log.info('Present clusters are %s'%', '.join(clusters))
		self.nagios_report.update_status_code(CRITICAL)
	    else:
		return clusters[cluster_name]

	def get_queue(cluster_name, queue_name):
	    if not (cluster_name, queue_name) in queues:
		self.log.error('No queue %s/%s corresponding to %s.'
		    % (cluster_name, queue_name, obj.structural_objectclass))
		self.log.info('Present queues are %s.'%', '.join(queues))
	    else:
		return queues[(cluster_name, queue_name)]

	# Compare GlueCE with ARC entries.
	if isinstance(obj, GlueCE):
	    cluster = get_cluster(obj.GlueCEInfoHostName)
	    if cluster is None:
		return
	    queue = get_queue(obj.GlueCEInfoHostName, obj.GlueCEName)
	    if queue is None:
		return
	    self.log.debug('Comparing %s with ARC entries.'%queue.name)
	    vl = GlueCE_Validator(obj, '%s/%s'%(cluster.name, queue.name),
				  self.log)
	    vl.compare_with(cluster, queue)
	    if vl.error_count:
		self.nagios_report.update_status_code(CRITICAL)

	# Compare GlueCluster with ARC entries.
	elif isinstance(obj, GlueCluster):
	    cluster = get_cluster(obj.GlueClusterUniqueID)
	    if cluster is None:
		return
	    vl = GlueCluster_Validator(obj, cluster.name, self.log)
	    vl.compare_with(cluster)
	    if vl.error_count:
		self.nagios_report.update_status_code(CRITICAL)

	# Compare GlueSubCluster with ARC entries.
	elif isinstance(obj, GlueSubCluster):
	    subcluster_id = obj.GlueSubClusterUniqueID
	    vl = GlueSubCluster_Validator(obj, subcluster_id, self.log)
	    if '/' in subcluster_id:
		cluster_name, queue_name = subcluster_id.split('/')
		cluster = get_cluster(cluster_name)
		if cluster is None:
		    return
		queue = get_queue(cluster_name, queue_name)
		if queue is None:
		    return
		if cluster.homogeniety:
		    self.log.error('Expected inhomogenious cluster for '
				   'GlueSubClusterUniqueID = %r.'\
				   % obj.GlueSubClusterUniqueID)
		    self.nagios_report.update_status_code(CRITICAL)
		vl.compare_with(cluster, queue)
	    else:
		cluster = get_cluster(subcluster_id)
		if cluster is None:
		    return
		if not cluster.homogeniety:
		    self.log.error('Expected homogenious cluster for '
				   'GlueSubClusterUniqueID = %r.'\
				   % obj.GlueSubClusterUniqueID)
		    self.nagios_report.update_status_code(CRITICAL)
		vl.compare_with(cluster, cluster)

	return obj

    def _validate_and_construct(self, cls, dn, entry):
	try:
	    return cls(self.subschema, dn, entry)
	except LDAPValidationError, xc:
	    self.log.error('Validation of %s failed.'%dn)
	    self.log.error(str(xc))
	    self.nagios_report.update_status_code(CRITICAL)

    def _report_missing(self, status, msg):
	if status:
	    self.nagios_report.update_status(status, msg)
	else:
	    self.log.warning(msg)

    def _dn_error(self, dn, msg):
	if getattr(self, '_last_dn', None) != dn:
	    self.log.error('Issues for dn "%s"'%dn)
	    self._last_dn = dn
	self.log.error('  - %s'%msg)

    def check_glue2(self):
	self.lconn = ldap.initialize(self.opts.ldap_uri)
	try:
	    self.subschema = \
		    ldapschema.parse(self.opts.glue2_schema, log = self.log,
				     builddir = self.tmpdir())
	except ldapschema.ParseError:
	    self.nagios_report.update_status(UNKNOWN,
		    'Could not parse GLUE2 schema.')
	    return
	except IOError:
	    self.nagios_report.update_status(self.opts.if_dependent_schema,
		    'Using schema from tested server.')
	    self.log.warning(
		    'The schema definition had to be fetched from the CE '
		    'itself, as %s is missing or inaccessible. '
		    'Use --glue2-schema to specify an alternative URL.'
		    % self.opts.glue2_schema)
	    self.subschema = self.fetch_subschema()

	# Stage 1.  Run though each entry and check what can be checked
	# without information about the full dataset.  Accumulate what we need
	# for the next phase.
	sr = self.search_s(self.opts.ldap_basedn or 'o=glue',
			   ldap.SCOPE_SUBTREE, '(objectClass=GLUE2Entity)',
			   attrlist = ['*', 'structuralObjectClass'])
	entry_count = 0
	entry_counts = {}
	if_missing = {}
	for class_name in self.opts.warn_if_missing.split(','):
	    if class_name:
		entry_counts[class_name] = 0
		if_missing[class_name] = WARNING
	for class_name in self.opts.critical_if_missing.split(','):
	    if class_name:
		entry_counts[class_name] = 0
		if_missing[class_name] = CRITICAL
	errors = Statdict()
	rt = RelationTracker()
	for dn, ent in sr:
	    entry_count += 1
	    try:
		o = glue2.construct_from_ldap_entry(self.subschema, dn, ent)
		if o is None:
		    self.log.warning(
			    '%s: Unknown structural object class, using '
			    'generic LDAP validation.'%dn)
		    LDAPObject(self.subschema, dn, ent)
		    continue
	    except (ValueError, LDAPValidationError), xc:
		errors['invalid entries'] += 1
		self.log.error('%s: %s'%(dn, xc))
		continue

	    # This is to check consistency of a link to a parent admin domain.
	    parent_domain_id = None
	    for dncomp in str2dn(dn)[1:]:
		if dncomp[0] == 'GLUE2DomainID':
		    parent_domain_id = dncomp[1]
		    break

	    # Check uniqueness of the primary key and record its dn.
	    if o.glue2_primary_key:
		pk = o.glue2_primary_key
		pkv = getattr(o, pk)
		if isinstance(pkv, list):
		    if len(pkv) > 1:
			self._dn_error(dn,
				'Multivalued primary key %s.'%pk)
			errors['multivalued PKs'] += 1
		if rt.seen_pk(pk, pkv):
		    self._dn_error(dn,
			    'Duplicate primary key %s=%s.'%(pk, pkv))
		    errors['duplicate PKs'] += 1
		else:
		    rt.add_pk(pk, pkv, dn)

	    # Checks and accumulations related to foreign keys.
	    for fk in o.glue2_all_foreign_keys():
		pk = fk.other_class.glue2_primary_key
		links = o.glue2_get_fk_links(fk)

		# Check target multiplicity and record the link.
		if not glue2.matching_multiplicity(fk.other_mult,
						   len(links)):
		    self._dn_error(dn,
			    '%s contains %d links, but the multiplicity '
			    'of the target of this relation is %s.'
			    %(fk.name, len(links),
			      glue2.multiplicity_indicator(fk.other_mult)))
		    errors['multiplicity violations'] += 1
		for link in links:
		    rt.add_ref(pk, link, fk, dn)

		# If the entry appears under an admin domain and links to an
		# admin domain, they should be the same.  We don't distinguish
		# user and admin domains here, but user domains should not be
		# the parent of services, anyway.
		if parent_domain_id and \
			fk.other_class == glue2.GLUE2AdminDomain:
		    for link in links: # 0 or 1
			if link != parent_domain_id:
			    self._dn_error(dn,
				    'This entry links to an admin domain other '
				    'than the domain under which it appears.')

	    # Count entries of each objectclass.
	    class_name = o.glue2_class_name()
	    if class_name in entry_counts:
		entry_counts[class_name] += 1

	# Stage 2.  Run though each foreign key link and perform related
	# checks.
	hierarchical_foreign_keys = \
		set(self.opts.hierarchical_foreign_keys.split(','))
	for ((pk, v), rte) in rt.iteritems():
	    for fk, refs in rte.referred_by.iteritems():

		# Multiplicity Checks.
		if not rte.seen_in:
		    self.log.error(
			    'No match for primary key %s=%s referred by %s'
			    %(pk, v, fk.name))
		    errors['missing primary keys'] += 1
		if not glue2.matching_multiplicity(fk.local_mult, len(refs)):
		    self.log.error(
			    '%d entries refer to %s=%s through %s but the '
			    'source multiplicity of this relation is %s.'
			    %(len(refs), pk, v, fk.name,
			      glue2.multiplicity_indicator(fk.local_mult)))
		    errors['multiplicity violations'] += 1

		# DIT Checks.
		if len(rte.seen_in) != 1:
		    continue
		dn = list(rte.seen_in)[0]
		if fk.name == 'GLUE2ExtensionEntityForeignKey':
		    for ref in refs:
			if not is_immediate_subdn(ref, dn):
			    self.log.error(
				    'The extension entry %s should be '
				    'immediately below %s.'%(ref, dn))
			    errors['DIT issues'] += 1
		elif fk.other_class == glue2.GLUE2Service and \
			fk.relation >= glue2.AGGREGATION:
		    for ref in refs:
			if not is_proper_subdn(ref, dn):
			    self.log.error(
				    'The component %s belongs below its '
				    'service %s.'%(ref, dn))
			    errors['DIT issues'] += 1
		elif fk.name in hierarchical_foreign_keys or \
			self.opts.hierarchical_aggregates \
			    and fk.relation >= glue2.AGGREGATION:
		    for ref in refs:
			if not is_proper_subdn(ref, dn):
			    self.log.error('"%s" belongs below "%s"'%(ref, dn))
			    errors['DIT issues'] += 1

	# Report.
	for class_name, count in entry_counts.iteritems():
	    if count == 0:
		errors['absent expected object classes'] += 1
		self.log.error('There are no entries with objectclass %s.'
			       % class_name)
	if errors:
	    self.nagios_report.update_status(CRITICAL, 'Found %s.'
		    % ', '.join('%d %s'%(c, s) for s, c in errors.iteritems()))
	self.nagios_report.update_status(OK,
		'Validated %d entries.'%entry_count)

    def check_aris(self):
	"""The entry point for the ARIS probe."""

	self.lconn = ldap.initialize(self.opts.ldap_uri)
	self.subschema = self.fetch_subschema()

	# Query cluster entries.
	basedn = self.opts.ldap_basedn or 'Mds-Vo-name=local, o=grid'
	if self.opts.clusters:
	    filters = [filter_format('nordugrid-cluster-name=%s', cluster)
		       for cluster in self.opts.clusters]
	    filter = '(&(objectClass=nordugrid-cluster)(|%s))'%''.join(filters)
	else:
	    filter = '(objectClass=nordugrid-cluster)'
	sr = self.search_s(basedn, ldap.SCOPE_ONELEVEL, filter)
	cluster_count = len(sr)
	if cluster_count == 0:
	    msg = 'No clusters found.'
	    if self.opts.if_no_clusters:
		self.nagios_report.update_status(self.opts.if_no_clusters, msg)
	    else:
		self.log.warning(msg)

	# Report error if expected entries are missing.
	if self.opts.clusters:
	    found_clusters = set()
	    for _, ent in sr:
		found_clusters.update(ent['nordugrid-cluster-name'])
	    for cluster in self.opts.clusters:
		if not cluster in found_clusters:
		    self.log.error('Missing entry for %s.'%cluster)
		    self.nagios_report.update_status_code(CRITICAL)

	# These are indexed for later comparison with Glue entries.
	clusters = {} # indexed by cluster.name
	queues = {}   # indexed by (cluster.name, queue.name)
	for dn, ent in sr:
	    # Check nordugrid-cluster entry.
	    cluster = self.verify_nordugrid_cluster(dn, ent)
	    if cluster is None:
		self.log.warning('Skipping queue checks for invalid cluster '
				 'entry %s.'%dn)
		continue
	    clusters[cluster.name] = cluster

	    # Query and check the corresponding queue entries.
	    self.log.debug('Checking queues for %s.'%cluster.name)
	    qbasedn = 'nordugrid-cluster-name=%s,%s'%(cluster.name, basedn)
	    qsr = self.search_s(qbasedn, ldap.SCOPE_SUBTREE,
				'(objectClass=nordugrid-queue)')
	    if len(qsr) == 0:
		msg = 'No queue defined for %s.'%cluster.name
		if self.opts.if_no_queues:
		    self.nagios_report.update_status(
			    self.opts.if_no_queues, msg)
		else:
		    self.log.warning(msg)
	    else:
		middleware = getattr(cluster, 'middleware',
				     ['unknown middleware'])
		self.log.info('- Cluster %r runs %s.'
		    % (cluster.name, ', '.join(middleware)))
		for qdn, qent in qsr:
		    queue = self.verify_nordugrid_queue(cluster, qdn, qent)
		    queues[(cluster.name, queue.name)] = queue
		    status = getattr(queue, 'status', 'unknown')
		    self.log.info('-- Queue %r is %s.'%(queue.name, status))

	# Check Glue entries if enabled.
	if self.opts.enable_glue:
	    glue_counts = {}
	    sr = self.search_s('Mds-Vo-name=resource,o=grid',
			       ldap.SCOPE_ONELEVEL, 'objectClass=*')
	    for dn, ent in sr:
		obj = self.verify_glue_entry(dn, ent, clusters, queues)
		if obj:
		    oc = obj.structural_objectclass
		    if not oc in glue_counts:
			glue_counts[oc] = 1
		    else:
			glue_counts[oc] += 1
	    for oc, count in glue_counts.iteritems():
		self.log.info('Validated %s.'
		    % counted_noun(count, '%s entry'%oc, '%s entries'%oc))

	# Try to access the nordugrid-cluster-contactstring if requested.
	if self.opts.check_contact:
	    self.require_voms_proxy()
	    for cluster in clusters.itervalues():
		try:
		    url = cluster.contactstring
		    xs = arcutils.arcls(url, log = self.log)
		    self.log.info('%s contains %d entries.'%(url, len(xs)))
		except arcutils.CalledProcessError:
		    self.nagios_report.update_status(CRITICAL,
			'Contact URL %s is inaccessible.'%url)
		except AttributeError:
		    self.nagios_report.update_status(CRITICAL,
			'The cluster %s has no contact string.'%cluster.name)

	# Pass a default status and exit.
	try:
	    cinfo = ' (%s)'%', '.join([', '.join(c.middleware)
				       for c in clusters.itervalues()])
	    qinfo = ' (%s)'%', '.join([q.status for q in queues.itervalues()])
	except Exception:
	    cinfo = ''
	    qinfo = ''
	self.nagios_report.update_status(OK,
		'%s%s, %s%s'
		    %(counted_noun(cluster_count, 'cluster'), cinfo,
		      counted_noun(len(queues), 'queue'), qinfo))
	return self.nagios_exit(subject = 'ARIS service')

    def valid_egiis_entry(self, dn, ent):
	"""Validate a single EGIIS entry and return an pair of the number of
	errors and an `EGIIS_Object` for the entry or None."""

	try:
	    return MdsServiceLdap(self.subschema, dn, ent)
	except LDAPValidationError, xc:
	    self.log.error(str(xc))
	    self.nagios_report.update_status_code(CRITICAL)
	    return None

    def check_egiis(self):
	"""The entry point for the EGIIS probe."""

	# Query the EGIIS for subschema and entries.
	basedn = self.opts.ldap_basedn \
		or 'Mds-Vo-name=%s, o=grid'%self.opts.index_name
	self.lconn = ldap.initialize(self.opts.ldap_uri)
	self.subschema = self.fetch_subschema()
	sr = self.search_s(basedn, ldap.SCOPE_BASE,
			   'objectClass=MdsServiceLdap')
	if len(sr) == 0:
	    raise ServiceWARNING('No EGIIS entries found.')

	# Check the entries.
	entcnts = {}
	for dn, ent in sr:
	    egiis = self.valid_egiis_entry(dn, ent)
	    if egiis:
		self.log.info('Good entry for %s'%egiis.ldap_suffix)
		entcnts[egiis.reg_status] = 1 + entcnts.get(egiis.reg_status, 0)

	# Report the result.
	entcnts = entcnts.items()
	entcnts.sort(lambda a, b: cmp(a[0], b[0]))
	countstrings = ['%d %s'%(cnt, egiis_reg_status_to_string(st).lower())
			for st, cnt in entcnts]
	self.nagios_report.update_status(
		OK, 'EGIIS ok: %s.'%', '.join(countstrings))
	return self.nagios_exit(subject = 'EGIIS service')
