#!/usr/bin/perl -w

# Copyright (c) Members of the EGEE Collaboration. 2009.
# See http://www.eu-egee.org/partners/ for details on the copyright
# holders.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# Authors: Jan Just Keijser 
#      NIKHEF Amsterdam, the Netherlands
#      janjust@nikhef.nl

# $Id:  $
#
# Wrapper script to wrap the current environment into a single
# environment variable GLEXEC_ENV. This variable is passed 
# onto the glexec child process, where it can be unpacked to
# restore the environment variables that were lost when the
# set-uid glexec was invoked.
#
# Intended usage:
#   export GLEXEC_ENV=`glexec_wrapenv.pl`
#   /opt/glite/sbin/glexec glexec_unwrapenv.pl -- <YOUR-COMMAND>
#
# By default the following environment variables are NOT wrapped:
#   HOME LOGNAME USER X509_USER_PROXY _  (yes that's '_' !)
# A user can add more env vars to be excluded using either
#  --exclude=A --exclude=B
# or
#  --exclude=A,B,...
#

use strict;
use warnings;

use Compress::Zlib qw(deflateInit Z_OK);
use Getopt::Long   qw(GetOptions);
use MIME::Base64   qw(encode_base64);

# These variables are excluded by default
my @env_blacklist = ( "HOME", "LOGNAME", "USER", "X509_USER_PROXY", "_" );

my @exclude_env;
my $key;
my $buf;
my $encoded_buf = '';
my $output      = '';

GetOptions ("exclude=s" => \@exclude_env);
@exclude_env = split( /,/, join( ',', @exclude_env, @env_blacklist) );

# go through all environment variables and encode them as separate 
# key-value pair entities. This will enable us to later unpack them.
foreach $key (keys(%ENV))
{ 
    if ( ! grep { /$key/ } @exclude_env )
    {
        $buf          = $key . "=" . $ENV{$key};
        $encoded_buf .= encode_base64($buf, '') . " ";
    }
    else
    {
        printf STDERR "Skipping $key\n";
    }
}

# Compress the encoded env vars to save some memory
my $x = deflateInit()
    or die "Cannot create a deflation stream\n" ;

my ($deflated_buf, $status) = $x->deflate( $encoded_buf );
$status == Z_OK or die "deflation failed\n";
$output = $deflated_buf;

($deflated_buf, $status) = $x->flush();
$status == Z_OK or die "deflation failed\n";
$output .= $deflated_buf;

# Finally, encode the compressed stream again and print it out
print encode_base64( $output, '' );

