#!/usr/bin/perl -w

# Copyright (c) Members of the EGEE Collaboration. 2009.
# See http://www.eu-egee.org/partners/ for details on the copyright
# holders.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# Authors: Jan Just Keijser 
#      NIKHEF Amsterdam, the Netherlands
#      janjust@nikhef.nl

# $Id:  $
#
# Helper script to restore the environment variables previously
# wrapped into the environment variable GLEXEC_ENV using the
# glexec_wrapenv.pl script. 
#
# Intended usage:
#   export GLEXEC_ENV=`glexec_wrapenv.pl`
#   /opt/glite/sbin/glexec glexec_unwrapenv.pl -- <YOUR-COMMAND>
#
# By default the following environment variables are NOT unwrapped:
#   HOME LOGNAME USER X509_USER_PROXY _  (yes that's '_' !)
# A user can add more env vars to be excluded using either
#  --exclude=A --exclude=B
# or
#  --exclude=A,B,...
#

use strict;
use warnings;

use Compress::Zlib qw(inflateInit Z_STREAM_END Z_OK);
use Getopt::Long   qw(GetOptions);
use MIME::Base64   qw(decode_base64);

# These variables are excluded by default
my @env_blacklist = ( "HOME", "LOGNAME", "USER", "X509_USER_PROXY", "_" );
my @exclude_env;

GetOptions ("exclude=s" => \@exclude_env);
@exclude_env = split( /,/, join( ',', @exclude_env, @env_blacklist) );

$ENV{GLEXEC_ENV} 
    or die "GLEXEC_ENV not set. No environment to pass on";

# First, unwrap the Base64 encoded blob
my $decoded_buf = decode_base64( $ENV{GLEXEC_ENV} );

# Then, decompress it into it's original space-separated set of Base64 blobs
my $x = inflateInit()
    or die "Cannot create a inflation stream\n" ;

my ($output, $status) = $x->inflate( \$decoded_buf );

die "inflation failed\n"
    unless $status == Z_STREAM_END or $status == Z_OK;

# Split the space-separated set of Base64 blobs again into an array
my @vars = split / /, $output;

for (my $i = 0; $i <= $#vars; $i++)
{
    # Decode each Base64 encoded blob into a key=value pair
    my $keyvalue_pair = decode_base64( $vars[$i] );
    my $pos = -1;

    # Look for the first '=' sign
    if (($pos = index( $keyvalue_pair, '=' )) > -1 )
    {
        # NOTE: using tricks like (\w+) (\w+) will NOT work
        # when environment variables span multiple lines

        # the "key" is everything before the first '=' sign
        my $key   = substr( $keyvalue_pair, 0, $pos );
        # the "value" is everything after the first '=' sign
        my $value = substr( $keyvalue_pair, $pos+1 );

        # if the variable is not on our exclusion list, set it
        if ( ! grep { /$key/ } @exclude_env )
        {
            $ENV{$key} = $value;
        }
    }
    else
    {
        # this should never happen, really
        printf STDERR  "no = sign found in [$keyvalue_pair]!\n";
    }
}

# Finally, execute the user payload command
exec ( @ARGV );

