/**
 *  Copyright (c) Members of the EGEE Collaboration. 2004-2008.
 *  See http://www.eu-egee.org/partners/ for details on the copyright
 *  holders.
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  GLite Data Management - Simple transfer API
 *  Authors: Gabor Gombas <Gabor.Gombas@cern.ch>
 *           Zoltan Farkas <Zoltan.Farkas@cern.ch>
 *           Paolo Tedesco <paolo.tedesco@cern.ch>
 *  Version info: $Id: transfer-simple.h,v 1.31 2009/07/02 15:28:18 szamsu Exp $
 *  Release: $Name:  $
 */
#ifndef GLITE_DATA_TRANSFER_SIMPLE_H
#define GLITE_DATA_TRANSFER_SIMPLE_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdarg.h>
#include <stdint.h>
#include <time.h>

/* constant used to refer to the desired FTS interfcae version */
#define GLITE_FTS_IFC_VERSION       "GLITE_FTS_IFC_VERSION"

/**********************************************************************
 * Data type declarations
 */

/* Opaque data structure used by the library */
typedef struct _glite_transfer_ctx          glite_transfer_ctx;

/**
 * @defgroup data_structures Transfer data structures, and data types
 * @{
 */
typedef struct _glite_transfer_Channel                  glite_transfer_Channel;
typedef struct _glite_transfer_Channel2                 glite_transfer_Channel2;
typedef struct _glite_transfer_Channel3                 glite_transfer_Channel3;
typedef struct _glite_transfer_ChannelAuditEntry        glite_transfer_ChannelAuditEntry;
typedef struct _glite_transfer_FileTransferStatus       glite_transfer_FileTransferStatus;
typedef struct _glite_transfer_JobStatus                glite_transfer_JobStatus;
typedef struct _glite_transfer_PlacementJob             glite_transfer_PlacementJob;
typedef struct _glite_transfer_TransferJob              glite_transfer_TransferJob;
typedef struct _glite_transfer_TransferJobElement       glite_transfer_TransferJobElement;
typedef struct _glite_transfer_TransferJobSummary       glite_transfer_TransferJobSummary;
typedef struct _glite_transfer_TransferJobSummary2      glite_transfer_TransferJobSummary2;
typedef struct _glite_transfer_TransferParams           glite_transfer_TransferParams;
typedef struct _glite_transfer_StringPair               glite_transfer_StringPair;
typedef struct _glite_transfer_Roles                    glite_transfer_Roles;
typedef struct _glite_transfer_VOShareElement           glite_transfer_VOShareElement;
typedef struct _glite_transfer_VOLimitElement           glite_transfer_VOLimitElement;
/* FileTransferStats and related structures */
typedef struct _glite_transfer_TransferSnapshot         glite_transfer_TransferSnapshot;
typedef struct _glite_transfer_TransferSummary          glite_transfer_TransferSummary;
typedef struct _glite_transfer_TransferActivity         glite_transfer_TransferActivity;
typedef struct _glite_transfer_TransferActivityPeriod   glite_transfer_TransferActivityPeriod;
typedef struct _glite_transfer_TransferAgent            glite_transfer_TransferAgent;


/**********************************************************************
 * Constants
 */

/**
 * Error categories. These are analogous to the exception classes in Java
 * */
typedef enum{
	GLITE_TRANSFER_EXCEPTION_EXISTS = -9,
    GLITE_TRANSFER_EXCEPTION_AUTHORIZATION = -8,
    GLITE_TRANSFER_EXCEPTION_CANNOTCANCEL = -7,
    GLITE_TRANSFER_EXCEPTION_FILENOTFOUND = -6,
    GLITE_TRANSFER_EXCEPTION_INTERNAL = -5,
    GLITE_TRANSFER_EXCEPTION_INVALIDARGUMENT = -4,
    GLITE_TRANSFER_EXCEPTION_NOTEXISTS = -3,
    GLITE_TRANSFER_EXCEPTION_SERVICEBUSY = -2,
    GLITE_TRANSFER_EXCEPTION_TRANSFER = -1,
    GLITE_TRANSFER_ERROR_NONE,
    GLITE_TRANSFER_ERROR_UNKNOWN,
    GLITE_TRANSFER_ERROR_INVALIDARGUMENT,
    GLITE_TRANSFER_ERROR_OUTOFMEMORY,
    GLITE_TRANSFER_ERROR_SERVICEDISCOVERY,
    GLITE_TRANSFER_ERROR_SOAP
} glite_transfer_errclass;

/**
 * Job states.
 * - Negative means a final state having an error
 * - Zero is successful final state
 * - Positive is a non-final state
 */
typedef enum{
    GLITE_TRANSFER_FAILED = -6,
    GLITE_TRANSFER_CATALOGFAILED = -5,
    GLITE_TRANSFER_FINISHED_DIRTY = -4,
    GLITE_TRANSFER_UNKNOWN = -3,
    GLITE_TRANSFER_CANCELED = -2,
    GLITE_TRANSFER_TRANSFERFAILED = -1,
    GLITE_TRANSFER_FINISHED = 0,
    GLITE_TRANSFER_SUBMITTED,
    GLITE_TRANSFER_PENDING,
    GLITE_TRANSFER_ACTIVE,
    GLITE_TRANSFER_CANCELING,
    GLITE_TRANSFER_WAITING,
    GLITE_TRANSFER_HOLD,
    GLITE_TRANSFER_DONE,
    GLITE_TRANSFER_READY,
    GLITE_TRANSFER_DONEWITHERRORS,
    GLITE_TRANSFER_FINISHING,
    GLITE_TRANSFER_AWAITING_PRESTAGE,
    GLITE_TRANSFER_PRESTAGING,
    GLITE_TRANSFER_WAITING_PRESTAGE,
    GLITE_TRANSFER_WAITING_CATALOG_RESOLUTION,
    GLITE_TRANSFER_WAITING_CATALOG_REGISTRATION
} glite_transfer_state;

/**
 * Channel states.
 */
typedef enum{
    GLITE_CHANNEL_UNKNOWN = -1,
    GLITE_CHANNEL_ACTIVE,
    GLITE_CHANNEL_INACTIVE,
    GLITE_CHANNEL_DRAIN,
    GLITE_CHANNEL_STOPPED,
    GLITE_CHANNEL_HALTED,
    GLITE_CHANNEL_ARCHIVED,
    GLITE_CHANNEL_DELETED
} glite_channel_state;


/**********************************************************************
 * Data tytpe definitions
 */

/**
 * Object to describe the status of a file transfer.
 */
struct _glite_transfer_FileTransferStatus{
    char                *logicalName; /**< The LFN of the file. */
    char                *sourceSURL; /**< The source of the transfer. */
    char                *destSURL; /**< The destination of the transfer. */
    glite_transfer_state        transferFileState; /**< The current state of the transfer. */
    int             numFailures; /**< 0 for a single successful transfer.*/
    char                *reason; /**< Some explanation of the current state.*/
    char                *reason_class; /**< Type of the above explanation.*/
    int             duration; /**< Duration of the transfer.*/
};

/**
 * Object to describe a job.
 */
struct _glite_transfer_JobStatus{
    char                *jobId; /**< The ID of the job, used for referring to this job. */
    glite_transfer_state        jobStatus; /**< The status of the job. */
    char                *channelName; /**< The name of the channel the job is assigned to. */
    char                *clientDN; /**< The DN of the client that started the job. */
    char                *reason; /**< Some explanation. */
    char                            *voName; /**< Name of the VO. */
    struct timespec         submitTime; /**< The time of the job submission. */
    int             numFiles; /**< Number of files in the job. */
    int             priority; /**< Priority of the job. */
};

/**
 * Describes the placement job description.
 * It consists of an array of logical files each describing an individual file transfer
 * within the job. sourceSE is the physical source SE that all the logical files must be
 * located on. destSE is the destination SE to transfer to. More complex operations
 * (e.g. arbitrary logical source) are provided by the data scheduler. jobParams are a
 * set of transfer layer specific paramaters that are applied to each file transfer (e.g.
 * gridFTP params); this is for future use and is not currently taken into account. The
 * credential is used by the transfer system to retrieve the appropriate proxy for the
 * transfer.
 */
struct _glite_transfer_PlacementJob{
    char                **logicalFiles; /**< Array of logical files to be transfered. */
    unsigned            __sizeLogicalFiles; /**< Number of items in the above array. */
    char                *sourceSE; /**< The physical source SE. */
    char                *destSE; /**< The physical destination SE. */
    glite_transfer_TransferParams   *jobParams; /**< Array of job parameters. */
    char                *credential; /**< The credential to get the appropriate proxy. */
};

/**
 * Describes the transfer job description with checksums.
 */
struct _glite_transfer_TransferJob{
    glite_transfer_TransferJobElement   **transferJobElements; /**< Array of TransferJobElements, each describing an individual file transfer within the job.*/
    unsigned                __sizetransferJobElements; /**< Number of items in the above array. */
    glite_transfer_TransferParams       *jobParams; /**< A set of transfer layer specific paramaters tha tare applied to each file transfer (e.g. gridFTP params).*/
    char                    *credential; /**< Used by the transfer system to retrieve the appropriate proxy for the transfer.*/
};

/**
 * An object to describe a source-destination pair with checksum.
 */
struct _glite_transfer_TransferJobElement{
    char                *source; /**< The source. */
    char                *dest; /**< The destination. */
    char                *checksum; /**< The checksum. */
};

/**
 * An object to describe a VO-share pair.
 */
struct _glite_transfer_VOShareElement{
    char                *VOName; /**< The VO name. */
    char                *share; /**< The share value. */
};

/**
 * An object to describe a VO-limit pair.
 */
struct _glite_transfer_VOLimitElement{
    char                *VOName; /**< The VO name. */
    char                *limit; /**< The limit value. */
};

/**
 * Describes the overall status of the job: the job state and some extra statistics.
 */
struct _glite_transfer_TransferJobSummary{
    glite_transfer_JobStatus    *jobStatus; /**< The current status of the job. */

    int             numDone; /**< Number of finished transfers, without the FPS updating the catalog. */
    int             numActive; /**< Number of running transfers. */
    int             numPending; /**< Number of pending transfers. */
    int             numCanceled; /**< Number of canceled transfers. */
    int             numCanceling; /**< Number of canceling transfers. */
    int             numFailed; /**< Number of failed transfers. */
    int             numFinished; /**< Number of finished transfers with catalog updates. */
    int             numSubmitted; /**< Total number of submitted transfers. */
    int             numHold; /**< Number of held transfers. */
    int             numWaiting; /**< Number of waiting transfers. */
    int             numCatalogFailed; /**< Number of transfers failed because of catalog error. */
    int             numRestarted; /**< Number of restarted transfers. */
};

/**
 * Describes the overall extended status of the job: the job state and some extra statistics.
 */
struct _glite_transfer_TransferJobSummary2{
    glite_transfer_JobStatus    *jobStatus; /**< The current status of the job. */

    int             numDone; /**< Number of finished transfers, without the FPS updating the catalog. */
    int             numActive; /**< Number of running transfers. */
    int             numPending; /**< Number of pending transfers. */
    int             numCanceled; /**< Number of canceled transfers. */
    int             numCanceling; /**< Number of canceling transfers. */
    int             numFailed; /**< Number of failed transfers. */
    int             numFinished; /**< Number of finished transfers with catalog updates. */
    int             numSubmitted; /**< Total number of submitted transfers. */
    int             numHold; /**< Number of held transfers. */
    int             numWaiting; /**< Number of waiting transfers. */
    int             numCatalogFailed; /**< Number of transfers failed because of catalog error. */
    int             numRestarted; /**< Number of restarted transfers. */
   int            numReady;
   int            numFinishing;
   int            numAwaitingPrestage;
   int            numPrestaging;
   int            numWaitingCatalogRegistration;
   int            numWaitingCatalogResolution;
   int            numWaitingPrestage;
};


/**
 * Parameters for the transfer.
 */
struct _glite_transfer_TransferParams{
    char                **keys; /**< Array of keys. */
    char                **values; /**< Array of values. */
    unsigned            num; /**< Number of items in the above arrays. */
};

/**
 * The transfer channel audit object
 */
struct _glite_transfer_ChannelAuditEntry{
   int            auditID;
    char                *channelName;
    char                *contact;
    int             nostreams;
    int             nofiles;
    int             bandwidth;
   char           *tcpBufferSize;
    int             nominalThroughput;
    glite_channel_state     channelState;
   char           *message;
   time_t         modificationTime;
   char           *adminDN;
   char           *ftsNode;
   char           *clientIP;
};


/**
 * The transfer channel object.
 */
struct _glite_transfer_Channel{
    char                            *channelName; /**< The name of the channel, used for referring to it. */
    char                            *sourceSite; /**< The source site of the channel. */
    char                            *destSite; /**< The destination site of the channel. */
    char                            *contact; /**< The contact address of the responsible of the channel. */
    int                             numberOfStreams; /**< Number of concurrent streams allowed on the channel. */
    int                             numberOfFiles; /**< Number of concurrent files allowed on the channel. */
    int                             bandwidth; /**< Current maximum bandwidth (capacity) in Mbits/s. */
    int                             nominalThroughput; /**< The throughput goal for the system to achieve. */
    int                             __sizeVOShares; /**< Number of VO/Share pairs */
    glite_transfer_VOShareElement   **VOShares; /**< VO/Share pairs */
    glite_channel_state             state; /**< The operational state of the channel. */
    int                             urlCopyFirstTxmarkTo;   /** <Number of seconds to mark activity on URL Copy */
    int                             targetDirCheck; /** <Flag to check destination directory */
};

/**
 * The transfer channel2 object.
 */
struct _glite_transfer_Channel2{
    char                            *channelName; /**< The name of the channel, used for referring to it. */
    char                            *sourceSite; /**< The source site of the channel. */
    char                            *destSite; /**< The destination site of the channel. */
    char                            *contact; /**< The contact address of the responsible of the channel. */
    char                            *tcpBufferSize; /** TCP buffer size. */
    char                            *lastModifierDn; /** Last person to change channel. */
    char                            *message; /** Why the channel was changed. */
    time_t                          lastModificationTime; /** time_t of when channel was last modified. */
    int                             numberOfStreams; /**< Number of concurrent streams allowed on the channel. */
    int                             numberOfFiles; /**< Number of concurrent files allowed on the channel. */
    int                             bandwidth; /**< Current maximum bandwidth (capacity) in Mbits/s. */
    int                             nominalThroughput; /**< The throughput goal for the system to achieve. */
    int                             __sizeVOShares; /**< Number of VO/Share pairs */
    glite_transfer_VOShareElement   **VOShares; /**< VO/Share pairs */
    int                             __sizeVOLimits; /**< Number of VO/Limit pairs */
    glite_transfer_VOLimitElement   **VOLimits; /**< VO/Limit pairs */
    glite_channel_state             state; /**< The operational state of the channel. */
    int                             urlCopyFirstTxmarkTo;   /** <Number of seconds to mark activity on URL Copy */
    int                             targetDirCheck; /** <Flag to check destination directory */
};

/**
 * The transfer channel3 object.
 */
struct _glite_transfer_Channel3{
    char                            *channelName; /**< The name of the channel, used for referring to it. */
    char                            *sourceSite; /**< The source site of the channel. */
    char                            *destSite; /**< The destination site of the channel. */
    char                            *contact; /**< The contact address of the responsible of the channel. */
    char                            *tcpBufferSize; /** TCP buffer size. */
    char                            *lastModifierDn; /** Last person to change channel. */
    char                            *message; /** Why the channel was changed. */
    time_t                          lastModificationTime; /** time_t of when channel was last modified. */
    int                             numberOfStreams; /**< Number of concurrent streams allowed on the channel. */
    int                             numberOfFiles; /**< Number of concurrent files allowed on the channel. */
    int                             bandwidth; /**< Current maximum bandwidth (capacity) in Mbits/s. */
    int                             nominalThroughput; /**< The throughput goal for the system to achieve. */
    int                             __sizeVOShares; /**< Number of VO/Share pairs */
    glite_transfer_VOShareElement   **VOShares; /**< VO/Share pairs */
    int                             __sizeVOLimits; /**< Number of VO/Limit pairs */
    glite_transfer_VOLimitElement   **VOLimits; /**< VO/Limit pairs */
    glite_channel_state             state; /**< The operational state of the channel. */
    int                             urlCopyFirstTxmarkTo;   /** <Number of seconds to mark activity on URL Copy */
    int                             targetDirCheck; /** <Flag to check destination directory */
    // new
    time_t                          lastActiveTime; /** The last time the channel was active. */
    int                             seLimit; /** Don't use this. You have been warned! */
    char                            *channelType; /** Type of a channel: DEDICATED or NONDEDICATED. */
    char                            *blockSize; /** Default block size of a transfer. */
    int                             httpTimeout; /** Timeouts are in seconds. */
    char                            *transferLogLevel; /** Possible values are: DEBUG, INFO (default), ERROR. */
    double                          preparingFilesRatio; /** Ratio of SRM activity to number of transfer slots. */
    char                            *transferType; /** Transfer type: URLCOPY|SRMCOPY. */
    int                             urlCopyPutTimeout;
    int                             urlCopyPutDoneTimeout;
    int                             urlCopyGetTimeout;
    int                             urlCopyGetDoneTimeout;
    int                             urlCopyTransferTimeout;
    int                             urlCopyTransferMarkersTimeout;
    int                             urlCopyNoProgressTimeout;     /** timeout without transfer progress. */
    double                          urlCopyTransferTimeoutPerMB;     /** Transfer timeout per megabyte. */
    char                            *srmCopyDirection; /** SRM copy direction: PUSH, PULL. */
    int                             srmCopyTimeout;    /** This is a global timeout. */
    int                             srmCopyRefreshTimeout;    /** Timeout between status updates. */
};


/**
 * A string pair object
 */
struct _glite_transfer_StringPair{
    char                *string1; /**< String 1. */
    char                *string2; /**< String 2. */
};

/**
 * The transfer roles object.
 */
struct _glite_transfer_Roles{
    char                *clientDN; /**< The DN of the client owning these roles. */
    char                *serviceAdmin; /**< Null if the client is not a service admin. Holds the principal giving it this role otherwise. */
    char                *submitter; /**< Null if the client does not have submit privileges. Holds the principal giving it this role otherwise. */
    int             __sizeVOManager; /**< Size of array VOManager. */
    glite_transfer_StringPair   **VOManager; /**< Holds a list of pairs VOName/principal. */
    int             __sizechannelManager; /**< Size of array channelManager */
    glite_transfer_StringPair   **channelManager; /**< Holds a list of pairs channelName/principal. */
};

/**
 * The transfer snapshot object.
 */
struct _glite_transfer_TransferSnapshot{
    time_t              *time; /**< The time when the snapshot was taken. */
    int             numSubmittedJobs;
    int             numPendingJobs;
    int             numActiveJobs;
    int             numDoneJobs;
    int             numHoldJobs;
    int             numFailedJobs;
    int             numCanceledJobs;
    int             numCancelingJobs;
    int             numFinishedDirtyJobs;
    int             numFinishedJobs;
    int             numSubmittedFiles;
    int             numPendingFiles;
    int             numActiveFiles;
    int             numDoneFiles;
    int             numWaitingFiles;
    int             numHoldFiles;
    int             numCancelingFiles;
    int             numCanceledFiles;
    int             numFailedFiles;
    int             numFinishedFiles;
    int             numCatalogFailedFiles;
};

/**
 * The transfer summary object.
 */
struct _glite_transfer_TransferSummary{
    char                *channelName; /**< The channel name this snapshot refers to. Null means all channels. */
    char                *voName; /**< The VO name this snapshot refers to. Null means all VOs. */
    int             __sizesnapshots; /**< Size of snapshots array. */
    glite_transfer_TransferSnapshot **snapshots; /**< The transfer snapshots, each refering to a single point in time. */
};

/**
 * The transfer activity period object.
 */
struct _glite_transfer_TransferActivityPeriod{
    time_t                  *startTime; /*< The start time of this period of measure. */
    time_t                  *endTime; /*< The end time of this period of measure. */
    float                   averageRate; /*< The average transfer rate during this period. */
    long                    bytesWritten; /*< The number of bytes written during this period. */
};

/**
 * The transfer activity object.
 */
struct _glite_transfer_TransferActivity{
    char                    *channelName; /**< The channel name this measure refers to. Null means all channels. */
    char                    *voName; /**< The VO name this measure refers to. Null means all VOs. */
    int                 __sizeactivityPeriods; /**< Size of activityPeriods array. */
    glite_transfer_TransferActivityPeriod   **activityPeriods; /**< The list of measured activity periods. */
};

/**
 * The transfer agent object.
 */
struct _glite_transfer_TransferAgent{
    char                    *name; /**< The agent's name. */
    char                    *DN; /**< The agent's Distinguished Name (DN). */
    char                    *host; /**< The host the agent is running in. */
    char                    *type; /**< The agent's type. This can be: 'channel', 'fps' -> vo agent, 'fts' -> vo agent. */
    time_t                  *lastActive; /**< The last time the agent explicitly stated it is alive. */
    char                    *state; /**< The last known state of the agent. */
    char                    *version; /**< The agent's version. */
    char                    *contact; /**< A contact to a person responsible for the agent. */
};

/**
 * @}
 */

/**********************************************************************
 * Function prototypes - library management functions
 */

/**
 * @defgroup transfer_libs Transfer context management
 * @{
 */

/**
 * Allocates a new transfer context.
 * @param endpoint The service endpoint.
 * @return The context or NULL if memory allocation has failed.
 */
glite_transfer_ctx *glite_transfer_new(const char *endpoint);

/**
 * Destroys a transfer context.
 * @param ctx The context to be disposed.
 */
void glite_transfer_free(glite_transfer_ctx *ctx);

/**
 * Add sites to a transfer context.
 * @param ctx The context to be used.
 * @param s Source site.
 * @param d Destination site.
 */
void glite_transfer_add_sites(glite_transfer_ctx *ctx, const char *s, const char *d);

/**
 * Returns the error message for the last failed operation.
 * The returned pointer is valid only till the next call to any of
 * the library's functions with the same context pointer.
 * @param ctx The transfer context.
 * @return The error message string.
 */
const char *glite_transfer_get_error(glite_transfer_ctx *ctx);

/**
 * Determines the class of the last error.
 * @param ctx The transfer context.
 * @return The proper errorclass value.
 */
glite_transfer_errclass glite_transfer_get_errclass(glite_transfer_ctx *ctx);

/**
 * Set the error message in the context.
 * @param ctx The transfer context.
 * @param errclass The type of error.
 * @param fmt The format string.
 * @param  ... Variables in the above string.
 */
void glite_transfer_set_error(glite_transfer_ctx *ctx,
    glite_transfer_errclass errclass, const char *fmt, ...);

/**
 * Set the error message in the context.
 * @param ctx The transfer context.
 * @param errclass The type of error.
 * @param fmt The format string.
 * @param ap A variable length argument list.
 */
void glite_transfer_set_verror(glite_transfer_ctx *ctx,
    glite_transfer_errclass errclass, const char *fmt, va_list ap);

/**
 * Parse a string to a glite_transfer_state value.
 * @param ctx The transfer context used for error reporting.
 * @param state The state string to be parsed.
 * @return The glite_transfer_state, specially ::GLITE_TRANSFER_UNKNOWN if the
 * state variable is missing or invalid.
 */
glite_transfer_state glite_transfer_state_parse(glite_transfer_ctx *ctx,
    const char *state);

/**
 * Convert a glite_transfer_state value to a human readable string.
 * @param state The state to be converted to string.
 * @return The string
 */
const char *glite_transfer_state_str(glite_transfer_state state);

/**********************************************************************
 * Function prototypes - methods of transfer data types
 */

/* XXX doc: for the following, ctx can be NULL if no error reporting is
 * required */

/**
 * Allocates a file transfer status entry.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param source The source domain of the transfer.
 * @param dest The destinatioin domain of the transfer.
 * @param state The starting state value.
 * @return The allocated FileTransferStatus object.
 */
glite_transfer_FileTransferStatus *glite_transfer_FileTransferStatus_new(glite_transfer_ctx *ctx,
    const char *source, const char *dest, glite_transfer_state state);
/**
 * Destroys a file transfer status entry.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status object to be disposed.
 */
void glite_transfer_FileTransferStatus_free(glite_transfer_ctx *ctx,
    glite_transfer_FileTransferStatus *status);
/**
 * Destroys a list of file transfer status entries.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param nitems Number of items in the array.
 * @param status Array of status objects to be disposed.
 */
void glite_transfer_FileTransferStatus_freeArray(glite_transfer_ctx *ctx,
    int nitems, glite_transfer_FileTransferStatus *status[]);
/**
 * Makes a copy of an file transfer status entry.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status entry to be duplicated.
 * @return The replica of the object.
 */
glite_transfer_FileTransferStatus *glite_transfer_FileTransferStatus_clone(glite_transfer_ctx *ctx,
    const glite_transfer_FileTransferStatus *status);
/**
 * Change the value of the LogicalName field of a FileTransferStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status object to be modified.
 * @param name The new LogicalName to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_FileTransferStatus_setLogicalName(glite_transfer_ctx *ctx,
    glite_transfer_FileTransferStatus *status, const char *name);
/**
 * Change the value of the Reason field of a FileTransferStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status object to be modified.
 * @param reason The new Reason to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_FileTransferStatus_setReason(glite_transfer_ctx *ctx,
    glite_transfer_FileTransferStatus *status, const char *reason);
/**
 * Change the value of the ReasonClass field of a FileTransferStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status object to be modified.
 * @param reasonClass The new ReasonClass to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_FileTransferStatus_setReasonClass(glite_transfer_ctx *ctx,
    glite_transfer_FileTransferStatus *status, const char *reasonClass);

/**
 * Create a new JobStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param jobId The ID of the job used for later reference on it.
 * @param jobStatus The status to be set.
 * @return The new object.
 */
glite_transfer_JobStatus *glite_transfer_JobStatus_new(glite_transfer_ctx *ctx,
    const char *jobId, glite_transfer_state jobStatus);
/**
 * Dispose a JobStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param status The status object to be modified.
 */
void glite_transfer_JobStatus_free(glite_transfer_ctx *ctx,
    glite_transfer_JobStatus *status);
/**
 * Dispose multiple JobStatus objects.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param nitems Number of items in the array.
 * @param status Array of the JobStatus objects to be disposed.
 */
void glite_transfer_JobStatus_freeArray(glite_transfer_ctx *ctx, int nitems,
    glite_transfer_JobStatus *status[]);
/**
 * Create a copy of a JobStatus object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object we want to replicate.
 * @return A replica of the object.
 */
glite_transfer_JobStatus *glite_transfer_JobStatus_clone(glite_transfer_ctx *ctx,
    const glite_transfer_JobStatus *orig);

/**
 * Create a new PlacementJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param sourceSE The physical source SE.
 * @param destSE The physical destination SE.
 * @return The new PlacementJob object.
 */
glite_transfer_PlacementJob *glite_transfer_PlacementJob_new(glite_transfer_ctx *ctx,
    const char *sourceSE, const char *destSE);
/**
 * Dispose a PlacementJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be disposed.
 */
void glite_transfer_PlacementJob_free(glite_transfer_ctx *ctx,
    glite_transfer_PlacementJob *job);
/**
 * Clone an existing PlacementJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_PlacementJob *glite_transfer_PlacementJob_clone(glite_transfer_ctx *ctx,
    const glite_transfer_PlacementJob *orig);
/**
 * Sets the parameters of a PlacementJob.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be modified.
 * @param params The transfer parameters to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_PlacementJob_setParams(glite_transfer_ctx *ctx,
    glite_transfer_PlacementJob *job,
    const glite_transfer_TransferParams *params);
/**
 * Adds a new LFN to the PlacementJob.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be modified.
 * @param logicalFile The LFN to be added to the PlacementJob.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_PlacementJob_add(glite_transfer_ctx *ctx,
    glite_transfer_PlacementJob *job, const char *logicalFile);

/**
 * Create a new TransferJobElement object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param src The source SURL of the transfer.
 * @param dst The destination SURL of the transfer.
 * @return The new object.
 */
glite_transfer_TransferJobElement *glite_transfer_TransferJobElement_new(glite_transfer_ctx *ctx,
    const char *src, const char *dst, const char *cksm);
/**
 * Dispose a TransferJobElement object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param elem The object to be disposed.
 */
void glite_transfer_TransferJobElement_free(glite_transfer_ctx *ctx,
    glite_transfer_TransferJobElement *elem);
/**
 * Dispose an array of TransferJobElement objects.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param nitems Number of items in the array.
 * @param elems Array of objects to be disposed.
 */
void glite_transfer_TransferJobElement_freeArray(glite_transfer_ctx *ctx,
    int nitems, glite_transfer_TransferJobElement *elems[]);
/**
 * Clone an existing TransferJobElement object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_TransferJobElement *glite_transfer_TransferJobElement_clone(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJobElement *orig);

/**
 * Create a new TransferJobSummary object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @return The new object.
 */
glite_transfer_TransferJobSummary *glite_transfer_TransferJobSummary_new(glite_transfer_ctx *ctx);
glite_transfer_TransferJobSummary2 *glite_transfer_TransferJobSummary2_new(glite_transfer_ctx *ctx);
/**
 * Dispose a TransferJobSummary object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param summary The object to be disposed.
 */
void glite_transfer_TransferJobSummary_free(glite_transfer_ctx *ctx,
    glite_transfer_TransferJobSummary *summary);
void glite_transfer_TransferJobSummary2_free(glite_transfer_ctx *ctx,
    glite_transfer_TransferJobSummary2 *summary);

/**
 * Clone an existing TransferJobSummary object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_TransferJobSummary *glite_transfer_TransferJobSummary_clone(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJobSummary *orig);
glite_transfer_TransferJobSummary2 *glite_transfer_TransferJobSummary2_clone(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJobSummary2 *orig);

/**
 * Create a new TransferJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @return The new object.
 */
glite_transfer_TransferJob *glite_transfer_TransferJob_new(glite_transfer_ctx *ctx);
/**
 * Dispose a TransferJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be disposed.
 */
void glite_transfer_TransferJob_free(glite_transfer_ctx *ctx,
    glite_transfer_TransferJob *job);
/**
 * Clone an existing TransferJob object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_TransferJob *glite_transfer_TransferJob_clone(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJob *orig);
/**
 * Set the parameters of a TransferJob.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be modified.
 * @param params The parameters to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_TransferJob_setParams(glite_transfer_ctx *ctx,
    glite_transfer_TransferJob *job,
    const glite_transfer_TransferParams *params);
/**
 * Add a new source-destination pair to a TransferJob.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param job The object to be modified.
 * @param src The new source SURL to add.
 * @param dst The new destination SURL to add.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_TransferJob_add(glite_transfer_ctx *ctx,
    glite_transfer_TransferJob *job,
    const char *src, const char *dst, const char *cksm);

/**
 * Create a new TransferParams object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @return The new object.
 */
glite_transfer_TransferParams *glite_transfer_TransferParams_new(glite_transfer_ctx *ctx);
/**
 * Dispose a TransferParams object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param params The object to be disposed.
 */
void glite_transfer_TransferParams_free(glite_transfer_ctx *ctx,
    glite_transfer_TransferParams *params);
/**
 * Clone an existing TransferParams object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_TransferParams *glite_transfer_TransferParams_clone(glite_transfer_ctx *ctx,
    const glite_transfer_TransferParams *orig);
/**
 * Add a new key-value pair to a TransferParams object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param params The object to be modified.
 * @param key The key to be added.
 * @param value The value assigned to the key.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_TransferParams_add(glite_transfer_ctx *ctx,
    glite_transfer_TransferParams *params, const char *key, const char *value);

/**
 * Create a new Channel object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param name The name of the channel, used for reference.
 * @param domainA The domain of the source SE.
 * @param domainB The domain of the destination SE.
 * @return The new object.
 */
glite_transfer_Channel *glite_transfer_Channel_new(glite_transfer_ctx *ctx,
    const char *name, const char *domainA, const char *domainB);

glite_transfer_Channel2 *glite_transfer_Channel2_new(glite_transfer_ctx *ctx,
   const char *name, const char *domainA, const char *domainB);

glite_transfer_Channel3 *glite_transfer_Channel3_new(
    glite_transfer_ctx  *ctx,
   const char           *name,
   const char           *domainA,
   const char           *domainB);

/**
 * Dispose a Channel object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param channel The object to be disposed.
 */
void glite_transfer_Channel_free(
    glite_transfer_ctx      *ctx,
    glite_transfer_Channel  *channel);

void glite_transfer_Channel2_free(
    glite_transfer_ctx      *ctx,
    glite_transfer_Channel2 *channel2);

void glite_transfer_Channel3_free(
    glite_transfer_ctx      *ctx,
    glite_transfer_Channel3 *channel3);

/**
 * Clone an existing Channel object.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param orig The original object to be cloned.
 * @return The replica of the object.
 */
glite_transfer_Channel *glite_transfer_Channel_clone(glite_transfer_ctx *ctx,
    const glite_transfer_Channel *orig);

glite_transfer_Channel2 *glite_transfer_Channel2_clone(glite_transfer_ctx *ctx,
   const glite_transfer_Channel2 *orig);

/**
 * Set the contact information on a channel.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param channel The object to be modified.
 * @param contact The new value of the contact field to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_Channel_setContact(glite_transfer_ctx *ctx,
    glite_transfer_Channel *channel, const char *contact);
/**
 * Set the state of a channel.
 * @param ctx The transfer context, may be NULL if no error reporting is required.
 * @param channel The object to be modified.
 * @param state The new channel state to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_transfer_Channel_setState(glite_transfer_ctx *ctx,
    glite_transfer_Channel *channel, glite_channel_state state);

/**
 * Create a nem Roles object
 */
glite_transfer_Roles *glite_transfer_Roles_new(glite_transfer_ctx *ctx);

/* Free an array of channel audit entries */
void glite_transfer_ChannelAuditEntry_freeArray(glite_transfer_ctx *ctx,
    int nitems, glite_transfer_ChannelAuditEntry *ca[]);

/* Create a new channel audit entry. */
glite_transfer_ChannelAuditEntry *glite_transfer_ChannelAuditEntry_new(
    glite_transfer_ctx *ctx, const char *name);

/* Free a channel audit entry. */
void glite_transfer_ChannelAuditEntry_free(glite_transfer_ctx *ctx,
    glite_transfer_ChannelAuditEntry *ca);

/* Clone an existing channel audit entry. */
glite_transfer_ChannelAuditEntry *glite_transfer_ChannelAuditEntry_clone(
    glite_transfer_ctx *ctx, const glite_transfer_ChannelAuditEntry *orig);

#ifdef __cplusplus
}
#endif

#endif /* GLITE_DATA_TRANSFER_SIMPLE_H */
