/**
 *  Copyright (c) Members of the EGEE Collaboration. 2004-2008.
 *  See http://www.eu-egee.org/partners/ for details on the copyright
 *  holders.
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  GLite Data Management - Simple channel management API
 *  Authors: Gabor Gombas <Gabor.Gombas@cern.ch>
 *           Paolo Tedesco <paolo.tedesco@cern.ch>
 *  Version info: $Id: channel-simple.h,v 1.20 2009/02/26 18:17:31 molnarzs Exp $
 *  Release: $Name:  $
 */
#ifndef GLITE_DATA_TRANSFER_CHANNEL_SIMPLE_H
#define GLITE_DATA_TRANSFER_CHANNEL_SIMPLE_H

#ifdef __GNUC__
    #define DEPRECATED __attribute__ ((deprecated))
#else
    #define DEPRECATED
#endif

#ifdef __cplusplus
extern "C" {
#endif

#include <glite/data/transfer/c/transfer-simple.h>

/**
 * transfer_log_level
 */
typedef enum {
    TRANSFER_LOG_LEVEL_INVALID,
    TRANSFER_LOG_LEVEL_DEBUG,
    TRANSFER_LOG_LEVEL_INFO,
    TRANSFER_LOG_LEVEL_WARN,
    TRANSFER_LOG_LEVEL_ERROR
} transfer_log_level;

/**
 * convert string -> transfer_log_level
 * @param value
 * @return transfer_log_level
 */
transfer_log_level transfer_log_level_parse(const char * value);

/**
 * convert transfer_log_level -> string
 * @param value
 * @return string
 */
const char * transfer_log_level_str(transfer_log_level value);

/**
 * srmcopy_direction
 */
typedef enum {
    SRMCOPY_DIRECTION_INVALID,
    SRMCOPY_DIRECTION_PUSH,
    SRMCOPY_DIRECTION_PULL
} srmcopy_direction;

/**
 * convert string -> srmcopy_direction
 * @param value
 * @return srmcopy_direction
 */
srmcopy_direction srmcopy_direction_parse(const char * value);

/**
 * convert srmcopy_direction -> string
 * @param value
 * @return string
 */
const char * srmcopy_direction_str(srmcopy_direction value);

/**
 * validate a string containing a size, e.g.
 * 1024
 * 1 KB
 * .1 MB
 * 0.01 GB
 * @param value the string to validate
 * @return 1 if successful, 0 otherwise
 */
int validate_size_string(const char * value);

/**********************************************************************
 * Constants
 */

/* Service type for Service Discovery */
#define GLITE_CHANNEL_SD_TYPE       "org.glite.ChannelManagement"

/* Environment variable to override service type */
#define GLITE_CHANNEL_SD_ENV        "GLITE_SD_CHANNEL_TYPE"

/**********************************************************************
 * Function prototypes - library management functions
 */

/**
 * @defgroup transfer_lib_management Transfer related management functions
 * @{
 */

/**
 * Get the current endpoint used by the transfer context.
 * @param ctx The transfer context.
 * @return The string containing the endpoint.
 */
const char *glite_channel_get_endpoint(glite_transfer_ctx *ctx);

/**
 * Converts the state-string to the proper ENUM value.
 * @param ctx The transfer context. Only used for error handling.
 * @param state The state string to be converted.
 * @return The ::glite_transfer_state type variable. NULL if there was an error.
 */
glite_channel_state glite_channel_state_parse(
    glite_transfer_ctx  *ctx,
    const char          *state);

/**
 * Converts the transfer-state variable to a human readable string.
 * @param state The varible to be converted.
 * @return The string containing the name of the state. "Unknown" in case the variable is not the name of a possible state.
 */
const char *glite_channel_state_str(glite_channel_state state);

/**
 * @}
 */

/**********************************************************************
 * Function prototypes - org.glite.data.transfer.ServiceBase
 */

/**
 * @defgroup ServiceBase org.glite.data.transfer.ServiceBase
 * @{
 */

/**
 * Get the version of the remote endpoint.
 * The returned string must be freed by the caller.
 * @param ctx The transfer context for error handling.
 * @return The version string. In case of an error NULL is returned.
 */
char *glite_channel_getVersion(glite_transfer_ctx *ctx);

/**
 * Get the schema version of the remote endpoint.
 * The returned string must be freed by the caller.
 * @param ctx The transfer context for error handling.
 * @return The schema-version string. In case of an error NULL is returned.
 */
char *glite_channel_getSchemaVersion(glite_transfer_ctx *ctx);

/**
 * Get the interface version of the remote endpoint.
 * The returned string must be freed by the caller.
 * @param ctx The transfer context for error handling.
 * @return The interface-version string. In case of an error NULL is returned.
 */
char *glite_channel_getInterfaceVersion(glite_transfer_ctx *ctx);

/**
 * Get the service metadata for a given key. The key must not be NULL.
 * The returned string must be freed by the caller.
 * @param ctx The transfer context for error handling.
 * @param key The key of which the value we want.
 * @return The interface-version string. In case of an error NULL is returned.
 */
char *glite_channel_getServiceMetadata(glite_transfer_ctx *ctx, const char *key);

/**
 * @}
 */

/**********************************************************************
 * Function prototypes - org.glite.data.transfer.channel.ChannelManagement
 */

/** @defgroup transfer_channel org.glite.data.transfer.channel.ChannelManagement
 * @{
 */

/**
 * Add a new transfer channel to the database.
 * @param ctx The transfer context.
 * @param channel The existing channel object to be added.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_add(
    glite_transfer_ctx              *ctx,
    const glite_transfer_Channel    *channel);

/**
 * Drop a channel from the database.
 * @param ctx The transfer context.
 * @param name The name of the channel to be dropped.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_drop(
    glite_transfer_ctx  *ctx,
    const char          *name);

/**
 * Get a channel object by its name.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @return The transfer-channel object, or NULL if there was an error.
 */
glite_transfer_Channel * glite_channel_getChannel(
    glite_transfer_ctx  *ctx,
    const char          *name);

glite_transfer_Channel2 * glite_channel_getChannel2(
    glite_transfer_ctx  *ctx,
    const char          *name);

glite_transfer_Channel3 * glite_channel_getChannel3(
    glite_transfer_ctx  *ctx,
    const char          *name);

/**
 * List all channels in the database.
 * @param ctx The transfer context.
 * @param resultCnt [OUT] The number of items in the return array.
 * @return A list containing the names of the channels, or NULL if there was an error.
 */
char **glite_channel_listChannels(glite_transfer_ctx *ctx, int *resultCnt);

/**
 * Manual intervention command on channel.
 * It changes the file_state for all files belonging to the given job from Hold to the
 * given file state. It will only work for files in the Hold state. The only permissible
 * requested states are 'Pending', 'Failed' or 'Canceling'.
 * @param ctx The transfer context.
 * @param jobId The ID of the job.
 * @param state The new state.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_changeStateForHeldJob(
    glite_transfer_ctx      *ctx,
    const char              *jobId,
    glite_transfer_state    state);

/**
 * Manual intervention command on channel.
 * It changes the file_state for all files belonging to all jobs on the given channel
 * which are in the Hold state to the given file state. It will only work for files
 * in the Hold state. The only permissible requested states are 'Pending', 'Failed'
 * or 'Canceling'.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param state The new state.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_changeStateForHeldJobsOnChannel(glite_transfer_ctx *ctx,
    const char *name, glite_transfer_state state);

/**
 * Set the theoretical bandwidth of a channel.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param utilisation The new bandwidth value.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setBandwidth(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 utilisation);

int glite_channel_setBandwidth2(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 utilisation,
    const char          *message);

/**
 * Set the contact information of a channel which can be queried in case of a problem.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param contact The contact info string.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setContact(glite_transfer_ctx *ctx, const char *name,
    const char *contact);
int glite_channel_setContact2(glite_transfer_ctx *ctx, const char *name,
    const char *contact, const char* message);

/**
 * Set the default TCP buffer size for a channel.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param buffer_size The TCP buffer size string.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setTCPBufferSize2(glite_transfer_ctx *ctx, const char *name,
    const char *buffer_size, const char* message);

/**
 * Set the maximal throughput value of a channel which should not be overrun.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param nominalThroughput The new throughput value.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setNominalThroughput(glite_transfer_ctx *ctx,
    const char *name, int nominalThroughput);
int glite_channel_setNominalThroughput2(glite_transfer_ctx *ctx,
    const char *name, int nominalThroughput, const char* message);

/**
 * Set the flag to check the target directory on put operation
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param target_dir_check The flag if to check for target directory availability
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setTargetDirCheck(glite_transfer_ctx *ctx, const char *name,
        int target_dir_check, const char* message);

/**
 * Set the number of seconds to mark first operations for URL copy as starting ones
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param url_copy_first_txmark_to if to check for target directory availability
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyFirstMarkTo(glite_transfer_ctx *ctx, const char *name,
        int url_copy_first_txmark_to, const char* message);

/**
 * Set the maximal number of files that can be transfered simultanously on a channel.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param numberOfFiles The new value.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setNumberOfFiles(glite_transfer_ctx *ctx, const char *name,
    int numberOfFiles);
int glite_channel_setNumberOfFiles2(glite_transfer_ctx *ctx, const char *name,
    int numberOfFiles, const char* message);

/**
 * Set the maximal number of streams a file transfer can be split.
 * @param ctx The transfer context.
 * @param name The name of the channel.
 * @param numberOfStreams The new value.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setNumberOfStreams(glite_transfer_ctx *ctx, const char *name,
    int numberOfStreams);
int glite_channel_setNumberOfStreams2(glite_transfer_ctx *ctx, const char *name,
    int numberOfStreams, const char* message);

/**
 * Set the state of a channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param state The state to be set.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setState(
    glite_transfer_ctx  *ctx,
    const char          *name,
    glite_channel_state state);
int glite_channel_setState2(
    glite_transfer_ctx  *ctx,
    const char          *name,
    glite_channel_state state,
    const char          * message);

#if 0
/**
 * setChannelType
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param ___
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setChannelType(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);
#endif

/**
 * setBlockSize
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the block size
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setBlockSize(
    glite_transfer_ctx  *ctx,
    const char          *name,
    const char          *value,
    const char          *message);

/**
 * setHttpTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the http timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setHttpTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          *message);

/**
 * setTransferLogLevel
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the transfer log level
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setTransferLogLevel(
    glite_transfer_ctx  *ctx,
    const char          *name,
    transfer_log_level  value,
    const char          *message);

/**
 * setPreparingFilesRatio
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the preparing files ratio
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setPreparingFilesRatio(
    glite_transfer_ctx  *ctx,
    const char          *name,
    double              value,
    const char          *message);

#if 0
/**
 * setTransferType
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the transfer type
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setTransferType(
    glite_transfer_ctx  *ctx,
    const char          *name,
    const char          *value,
    const char          *message);
#endif

/**
 * setUrlCopyPutTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the url copy put timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyPutTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          *message);

/**
 * setUrlCopyPutDoneTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the url copy putDone timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyPutDoneTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyGetTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the url copy get timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyGetTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyGetDoneTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the url copy getDone timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyGetDoneTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyTransferTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the transfer timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyTransferTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyTransferMarkersTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the transfer markers timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyTransferMarkersTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyNoProgressTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param the no-progress timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyNoProgressTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setUrlCopyTransferTimeoutPerMB
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param additional transfer timeout in seconds per MB
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setUrlCopyTransferTimeoutPerMB(
    glite_transfer_ctx  *ctx,
    const char          *name,
    double              value,
    const char          * message);

/**
 * setSrmCopyDirection
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value SRM copy direction
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setSrmCopyDirection(
    glite_transfer_ctx  *ctx,
    const char          *name,
    srmcopy_direction   value,
    const char          * message);

/**
 * setSrmCopyTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value the srm copy timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setSrmCopyTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * setSrmCopyRefreshTimeout
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param value srm copy refresh timeout
 * @param message the message for the change
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setSrmCopyRefreshTimeout(
    glite_transfer_ctx  *ctx,
    const char          *name,
    int                 value,
    const char          * message);

/**
 * Add a manager to the channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param principal The name of the principal.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_addManager(glite_transfer_ctx *ctx, const char *name,
    const char *principal);

/**
 * Remove a manager from the channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param principal The name of the principal.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_removeManager(glite_transfer_ctx *ctx, const char *name,
    const char *principal);

/**
 * List all managers of the channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param resultCnt [OUT] The number of items in the return array.
 * @return A list containing the managers of the channels, or NULL if there was an error.
 */
char **glite_channel_listManagers(glite_transfer_ctx *ctx, const char *name,
        int *resultCnt);

/**
 * Sets the share of a VO on a given channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param VOname The name of the VO
 * @param share The share value
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setVOShare(glite_transfer_ctx *ctx, const char *name,
        const char *VOname, int share);

/**
 * Sets the transfer limit of a VO on a given channel.
 * @param ctx The transfer context.
 * @param name The name of the channel
 * @param VOname The name of the VO
 * @param limit The limit value
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_setVOLimit(glite_transfer_ctx *ctx, const char *name,
        const char *VOname, int limit);

/**
 * Fetches the channel admin audit for the given period.
 * @param ctx The transfer context.
 * @param startTime The start time for the audit
 * @param endTime The end time for the audit
 * @param restrictType Check for these events only
 * @param lastNentries Get the last N events
 * @param channelName The channel for which to fetch the audit information
 * @return A list of channel audit entries
 */
glite_transfer_ChannelAuditEntry ** glite_channel_channelAudit(glite_transfer_ctx *ctx,
         const time_t startTime, const time_t endTime, const char * restrictType,
         const int lastNentries, const char * channelName, int* resultCount);

//------------------------------------------------------------------------------
//                          Groups Management
//------------------------------------------------------------------------------

/**
 * Get a list of the names of the defined site groups.
 * @param ctx [IN] The transfer context.
 * @param resultCnt [OUT] The number of items in the return array.
 * @return A list containing the names of the site groups, or NULL if there was an error.
 */
char ** glite_channel_getSiteGroupNames(
    glite_transfer_ctx  *ctx,
    int                 *resultCnt);

/**
 * Get the list of the sites that are members of the group with given name.
 * @param ctx [IN] The transfer context.
 * @param groupName [IN] The name of the group.
 * @param resultCnt [OUT] The number of items in the return array.
 * @return A list containing the names of the sites belonging to the group, or NULL if there was an error.
 */
char ** glite_channel_getSiteGroupMembers(
    glite_transfer_ctx  *ctx,
    char                *groupName,
    int                 *resultCnt);

/**
 * Add the site with given name to members of the given group. If the group does not exist it is created.
 * @param ctx [IN] The transfer context.
 * @param groupName [IN] The name of the group.
 * @param siteName [IN] The name of the site.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_addGroupMember(
    glite_transfer_ctx  *ctx,
    char                *groupName,
    char                *siteName);

/**
 * Remove the site with given name from members of the given group. If after the operation the group is empty, the group is deleted.
 * @param ctx [IN] The transfer context.
 * @param groupName [IN] The name of the group.
 * @param siteName [IN] The name of the site.
 * @return 0 when successful and -1 in case of an error.
 */
int glite_channel_removeGroupMember(
    glite_transfer_ctx  *ctx,
    char                *groupName,
    char                *siteName);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* GLITE_DATA_TRANSFER_CHANNEL_SIMPLE_H */
