/*!

	\file		eef_aos.h
	\brief	API of the AOS (Attribute Object Store)
	\author	Aram Verstegen

	This header contains the declarations of the Attribute Object Store.
	It contains the following core functions:
	-# AOS_set(label, value, type, bytes, parent): set a value in the AOS
	-# AOS_get_argument_by_label(label): retrieves a pointer to a value of type aos_argument_t, containing the data and associated meta information
  -# AOS_delete_argument_by_label(label): deletes the value identified by label from the list

	Additionally, it defines the following wrapper functions for these core functions:
	
	-# AOS_set_int(label, value): a wrapper to set a value of integer type
	-# AOS_set_string(label, value): a wrapper to set a value of string type
	-# AOS_set_pem(label, value): a wrapper to set a value of PEM string type
	-# AOS_get_value_by_label(label): retrieve a value associated with a label

  -# setAssertion(label, value): a wrapper to set an assertion into the AOS
  -# getAssertion(label): a wrapper to get an assertion from the AOS
  -# setObligation(label, value): a wrapper to set an obligation into the AOS
  -# getObligation(label): a wrapper to get an obligation from the AOS
  -# destroyObligation(label): a wrapper to delete an obligation from the AOS

*/

#ifndef AOS_API_H
#define AOS_API_H
#include <errno.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include "eef_polytypes.h"
#include "eef_return_codes.h"

EES_RC         /*@alt void@*/      AOS_Init (void);
EES_RC         /*@alt void@*/      AOS_Term (void);
EES_RC         /*@alt void@*/      AOS_Clean(void);

typedef struct aos_context_s       aos_context_t;
typedef struct aos_attribute_s     aos_attribute_t;
typedef struct aos_storage_s       aos_storage_t;
typedef struct aos_state_s         aos_state_t;


/* NEW AOS API */

aos_context_t*                     Context(aos_context_class_t context_class);

aos_context_t *                    createContext(aos_context_class_t);
EES_RC                             addContext(aos_context_t*);
EES_RC                             rewindContexts(void);
aos_context_t *                    getNextContext(aos_context_class_t, aos_storage_t*);
void                               setContextObligationId(aos_context_t*, char*);
char*                              getContextObligationId(aos_context_t*);
aos_context_class_t                getContextClass(aos_context_t*);

aos_attribute_t*                   createAttribute(void);
EES_RC                             addAttribute(aos_context_t*, aos_attribute_t*);
EES_RC                             rewindAttributes(aos_context_t* context);
aos_attribute_t*                   getNextAttribute(aos_context_t*);
EES_RC                             destroyAttribute(aos_context_t*, aos_attribute_t*);

EES_RC                             setAttributeId(aos_attribute_t*, const char* id);
EES_RC                             setAttributeIssuer(aos_attribute_t*, const char* issuer);
EES_RC                             setAttributeValue(aos_attribute_t*, const void* value, size_t size);
EES_RC                             setAttributeType(aos_attribute_t*, const char* type);

char*                              getAttributeId(aos_attribute_t*);
char*                              getAttributeIssuer(aos_attribute_t*);
char*                              getAttributeType(aos_attribute_t *);
char*                              getAttributeValueAsString(aos_attribute_t*);
int                                getAttributeValueAsInt(aos_attribute_t*);
void*                              getAttributeValueAsVoidPointer(aos_attribute_t*);

void                               aos_start_threading(void);
EES_RC                             aos_dump_argslist (void);

#endif

