#!/bin/bash
# cleanup-grid-accounts.sh

usage()
{
    exec 2>&1

    echo "Usage: $0 [options] [accounts]"
    echo "Options:"
    echo "	-c config_file"
    echo "	-d gridmapdir"
    echo "	-e   (name extra accounts to be cleaned up)"
    echo "	-F   (force cleanup of non-local directories)"
    echo "	-h   (help)"
    echo "	-i idle_time (in days)"
    echo "	-n   (dry run)"
    echo "	-v   (verbose)"

    exit 1
}

ACCOUNTS=
CONF=
DRYRUN=false
EXTRA=
FORCE=false
IDLE=15
IDLE_MIN=7
OPT_E=false
UID_MIN=100
VERBOSE=false
newline='
'

f=/etc/sysconfig/cleanup-grid-accounts
test -f $f && . $f

DIR=${GRIDMAPDIR:-/etc/grid-security/gridmapdir}
CONF=${CONF:-/etc/cleanup-grid-accounts.conf}

for i
do
    case $i in
    -c)
	CONF=/dev/null
	break
    esac
done

test -f $CONF && . $CONF

while test $# != 0
do
    case $1 in
    -c)
	shift
	test $# = 0 && usage
	CONF=$1
	< $CONF || exit
	. $CONF
	;;
    -d)
	shift
	test $# = 0 && usage
	DIR=$1
	;;
    -e)
	OPT_E=true
	;;
    -F)
	FORCE=true
	;;
    -h)
	usage
	;;
    -i)
	shift
	test $# = 0 && usage
	IDLE=$1
	;;
    -n)
	DRYRUN=true
	;;
    -v)
	VERBOSE=true
	;;
    -*)
	echo "$0: unknown option: $1" >&2
	usage
	;;
    *)
	if $OPT_E
	then
	    EXTRA="$EXTRA $*"
	else
	    ACCOUNTS="$*"
	fi
	break
    esac

    shift
done

echo "0$IDLE" | egrep '^0[0-9]+$' > /dev/null || usage

if test $IDLE -lt $IDLE_MIN
then
    echo "$0: ERROR: idle time $IDLE < $IDLE_MIN" >&2
    exit 1
fi

if test "x$ACCOUNTS" = x
then
    ACCOUNTS=`ls $DIR | egrep '^[^%]+[0-9]+$'`
fi

if test "x$EXTRA" != x
then
    ACCOUNTS="$ACCOUNTS $EXTRA"
fi

STALE=`mktemp ${TMPDIR:-/tmp}/cga-XXXXXX`

if test $? != 0 || test ! -f "$STALE"
then
    echo "$0: ERROR: cannot create temporary file!" >&2
    exit 1
fi

chmod a+r $STALE

trap "/bin/rm -f $STALE" HUP INT QUIT TERM

#
# avoid the SW areas (bug #49546)
#

SW_AREAS=`
    bash -lc 'env | sed -n "s/^VO_.*_SW_DIR=//p" | sort'
`

$VERBOSE && echo === Starting at `date`

for account in $ACCOUNTS
do
    uid=`id -u $account`

    if test "x$uid" = x
    then
	echo "$0: ERROR: skipping unknown user '$account'" >&2
	continue
    fi

    if test $uid -lt $UID_MIN
    then
	#
	# safety catch against accidental inclusion of root etc.
	#

	echo "$0: ERROR: skipping user '$account': UID $uid < $UID_MIN" >&2
	continue
    fi

    (
	eval dir=~$account

	cd $dir || exit

	cwd=`/bin/pwd`

	if ! $FORCE
	then
	    #
	    # if the directory is non-local, by default we do not touch it;
	    # for example, if the WNs share the grid account home directories
	    # with the CE, only the CE shall do the cleanup!
	    #

	    fs=`df -P . | awk 'NR == 2 { print $1 }'`
	    type=`mount | awk '$1 == "'"$fs"'" { print $5 }'`
	    local_fs='ext[23]|xfs'

	    if ! echo ":$type:$fs" | egrep "^:($local_fs):/" > /dev/null
	    then
		$VERBOSE && echo "Skipping non-local ($type) directory $cwd"
		exit
	    fi
	fi

	#
	# safety catches against somehow ending up in a system directory
	#

	case $cwd in
	/*/*)
	    # could be OK
	    ;;
	*)
	    echo "$0: ERROR: $account: bad home directory $cwd" >&2
	    exit
	esac

	sysdirs='/(afs|bin|boot|dev|etc|initrd|lib(64)?|proc|root|sbin|selinux|sys|usr)?/'

	if echo "$cwd/" | egrep "^$sysdirs" > /dev/null
	then
	    echo "$0: ERROR: $account: bad home directory $cwd" >&2
	    exit
	fi

	#
	# avoid the SW areas (bug #49546)
	#

	TARGETS=.

	for swa in $SW_AREAS
	do
	    case $cwd/ in
	    $swa|$swa/)
		echo "$0: WARNING: $cwd is a SW area - cleanup will be partial!" >&2
		TARGETS='.globus .lcgjm gram_job_mgr_*.log'
		break
	    esac
	done

	$VERBOSE && echo "Cleaning up $cwd"

	#
	# We must check the "ctime" and not the "mtime", because a file
	# can have its "mtime" set to an arbitrary value, for example when
	# the file was restored from a "tar" ball.  In a Globus GASS cache
	# some files have their "mtime" fields used to store information
	# that has nothing to do with their last modification times
	# (a value of 1 Jan 1970 is the clearest example).
	#
	# We cannot let "find" remove the stale entries: we must first do
	# a complete scan to determine all hard links to any stale file,
	# and then schedule all of them for removal.  A Globus GASS cache
	# makes extensive use of hard links.  Removing a link updates the
	# "ctime" of the file, which would prevent removal of the other
	# links to the same file.
	#
	# Files with newlines in their names are handled specially.
	#
	# Since the account may have been used for multi-user pilot jobs,
	# we cannot insist that all files below the home directory are
	# owned by the account itself: a pilot may have used the "glexec"
	# utility to create a payload work space owned by another UID.
	#
	# This implies the cleanup has to be done by root.  If the file
	# system is mounted from an NFS server with "root_squash" enabled,
	# the cleanup cron job should be installed on the NFS server.
	#

	find $TARGETS -depth -ctime +$IDLE -ls \( \
		! -path "*$newline*" -print -o -exec perl -e '
		    $_ = $ARGV[0];
		    s-\n-=//=-g;
		    print "$_\n"
		' {} \; \
	    \) -exec echo // \; | perl -pe '
		BEGIN {
		    our $sep = "\n//\n";
		    $/ = $sep;
		}

		s-$sep--e;
		s-\D.*\n- -s;
		s-$-\n-;
	    ' > $STALE

	if $DRYRUN
	then
	    RMDIR='print "rmdir  \\\n$_\n"'
	    UNLINK='print "unlink \\\n$_\n"'
	else
	    RMDIR=rmdir
	    UNLINK=unlink
	fi

	if $VERBOSE
	then
	    STATS='
		$t = localtime($mtime);	# "Thu Oct 13 04:54:34 1994"
		$t =~ s/... //;

		$p = (time - $mtime > 15552000) ? "..:..:.." : ":.. ....";
		$t =~ s/$p//e;

		$types = " pc d b - l s   ";
		$type = substr($types, $mode >> 12, 1);
		$m = sprintf("%06o", $mode);
		substr($m, 0, 2) = "$type " if ($type ne " ");

		printf "%s %3d %5u %5u %11u %s \\\n%s\n",
		    $m, $nlink, $uid, $gid, $size, $t, $_;
	    '
	else
	    STATS=
	fi

	perl -ne '
	    BEGIN {
		$| = 1;
	    }

	    chomp;

	    ($inode = $_) =~ s/\D.*//;
	    s/^\d+ //;
	    s-=//=-\n-g;

	    next if m{^\.$};
	    next if m{^\./\.bash(_logout|_profile|rc)$};
	    next if m{^\./\.ssh(/|$)};
	    next if m{^\./\.globus(/\.gass_cache)?$};
	    next if m{^\./\.globus/\.gass_cache/(config|tmp)$};
	    next if m{^\./\.globus/\.gass_cache/(global|local)(/md5)?$};
	    next if m{^\./\.lcgjm(/\.(incoming[^/]*|remote_io_ptr))?$};

	    (undef, $ino, $mode, $nlink, $uid, $gid, undef, $size,
	     undef, $mtime, $ctime) = lstat();

	    #
	    # safety catches
	    #

	    next if $ino != $inode || $uid == 0;

	    #
	    # catch files that in the meantime have started getting used again
	    #

	    next if ! -d _ && time - $mtime < '`expr $IDLE '*' 86400`';

	    '"$STATS"'

	    (-d _ ? '"$RMDIR"' : '"$UNLINK"') or warn "$_: $!\n";
	' < $STALE
    )
done

rm $STALE

$VERBOSE && echo === Finished at `date`

