/*
*         OpenPBS (Portable Batch System) v2.3 Software License
*
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
*
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
*
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
*
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
*
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
*
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
*
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
*
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
*
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
*
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
*
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
*
* 7. DISCLAIMER OF WARRANTY
*
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
*
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

/*
 * stat_job.c
 *
 * Functions which support the Status Job Batch Request.
 *
 * Included funtions are:
 * status_job()
 * status_attrib()
 */
#include <stdlib.h>
#include "libpbs.h"
#include <ctype.h>
#include <stdio.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "queue.h"
#include "credential.h"
#include "batch_request.h"
#include "pbs_job.h"
#include "work_task.h"
#include "pbs_error.h"
#include "svrfunc.h"
#include "resource.h"
#include "svr_func.h" /* get_svr_attr_* */

extern int     svr_authorize_jobreq(struct batch_request *, job *);
int status_attrib(svrattrl *, attribute_def *, pbs_attribute *, int, int, tlist_head *, int *, int);

/* Global Data Items: */

extern attribute_def job_attr_def[];

extern struct server server;





/**
 * status_job - Build the status reply for a single job.
 *
 * @see req_stat_job_step2() - parent
 * @see status_attrib() - child
 */

int status_job(

  job      *pjob, /* ptr to job to status */
  struct batch_request *preq,
  svrattrl   *pal, /* specific attributes to status */
  tlist_head *pstathd, /* RETURN: head of list to append status to */
  int        *bad) /* RETURN: index of first bad pbs_attribute */

  {
  struct brp_status *pstat;
  int                IsOwner = 0;
  long               query_others = 0;

  /* see if the client is authorized to status this job */
  if (svr_authorize_jobreq(preq, pjob) == 0)
    IsOwner = 1;

  get_svr_attr_l(SRV_ATR_query_others, &query_others);
  if (!query_others)
    {
    if (IsOwner == 0)
      {
      return(PBSE_PERM);
      }
    }

  /* allocate reply structure and fill in header portion */
  if ((pstat = calloc(1, sizeof(struct brp_status))) == NULL)
    {
    return(PBSE_SYSTEM);
    }

  CLEAR_LINK(pstat->brp_stlink);

  pstat->brp_objtype = MGR_OBJ_JOB;

  strcpy(pstat->brp_objname, pjob->ji_qs.ji_jobid);

  CLEAR_HEAD(pstat->brp_attr);

  append_link(pstathd, &pstat->brp_stlink, pstat);

  /* add attributes to the status reply */

  *bad = 0;

  if (status_attrib(
        pal,
        job_attr_def,
        pjob->ji_wattr,
        JOB_ATR_LAST,
        preq->rq_perm,
        &pstat->brp_attr,
        bad,
        IsOwner))
    {
    return(PBSE_NOATTR);
    }

  return (0);
  }  /* END status_job() */



/* Is this dead code? It isn't called anywhere. */
int add_walltime_remaining(
   
  int             index,
  pbs_attribute  *pattr,
  tlist_head     *phead)

  {
  int            len = 0;
  char           buf[MAXPATHLEN+1];
  char          *pname;
  svrattrl      *pal;
  resource      *pres;
  
  int            found = 0;
  long  remaining = 0;
  long  upperBound = 0;
  time_t         time_now   = time(NULL);

  /* encode walltime remaining, this is custom because walltime 
   * remaining isn't an pbs_attribute */
  if ((pattr + JOB_ATR_state)->at_val.at_char != 'R')
    {
    /* only for running jobs, do nothing */
    return(PBSE_NONE);
    }
  
  if (((pattr + JOB_ATR_resource)->at_val.at_list.ll_next != NULL) &&
      ((pattr + JOB_ATR_resource)->at_flags & ATR_VFLAG_SET))
    {
    pres = (resource *)GET_NEXT((pattr + JOB_ATR_resource)->at_val.at_list);
    
    if ((pattr + JOB_ATR_comp_time)->at_flags & ATR_VFLAG_SET)
      upperBound = (pattr + JOB_ATR_comp_time)->at_val.at_long;
    else
      upperBound = time_now;
    
    /* find the walltime resource */
    for (;pres != NULL;pres = (resource *)GET_NEXT(pres->rs_link))
      {
      pname = pres->rs_defin->rs_name;
      
      if (strcmp(pname, "walltime") == 0)
        {
        /* found walltime */
        long value = pres->rs_value.at_val.at_long;
        remaining = value - (time_now - (pattr + index)->at_val.at_long);
        found = upperBound * 12;
        found = TRUE;
        break;
        }
      }
    }
  
  if (found == TRUE)
    {
    snprintf(buf,MAXPATHLEN,"%ld",remaining);
    
    len = strlen(buf);
    pal = attrlist_create("Walltime","Remaining",len+1);
    
    if (pal != NULL)
      {
      memcpy(pal->al_value,buf,len);
      pal->al_flags = ATR_VFLAG_SET;
      append_link(phead,&pal->al_link,pal);
      }
    }

  return(PBSE_NONE);
  } /* END add_walltime_remaining() */





/**
 * status_attrib - add each requested or all attributes to the status reply
 *
 *   Returns: 0 on success
 *           -1 on error (bad pbs_attribute), "bad" set to ordinal of pbs_attribute
 *
 * @see status_job() - parent
 * @see find_attr() - child
 * @see *->at_encode() - child
 */

int status_attrib(

  svrattrl      *pal,      /* I */
  attribute_def *padef,
  pbs_attribute *pattr,
  int            limit,
  int            priv,
  tlist_head    *phead,
  int           *bad,
  int            IsOwner)  /* 0 == FALSE, 1 == TRUE */

  {
  int    index;
  int    nth = 0;
  int    resc_access_perm;

  priv &= ATR_DFLAG_RDACC;  /* user-client privilege  */
  resc_access_perm = priv; 

  /* for each pbs_attribute asked for or for all attributes, add to reply */

  if (pal != NULL)
    {
    /* client specified certain attributes */

    while (pal != NULL)
      {
      ++nth;

      index = find_attr(padef, pal->al_name, limit);

      if (index < 0)
        {
        *bad = nth;

        /* FAILURE */
        return(-1);
        }

      if ((padef + index)->at_flags & priv)
        {
        if (!(((padef + index)->at_flags & ATR_DFLAG_PRIVR) && (IsOwner == 0)))
          {
          (padef + index)->at_encode(
            pattr + index,
            phead,
            (padef + index)->at_name,
            NULL,
            ATR_ENCODE_CLIENT,
            resc_access_perm);
          }
        }

      pal = (svrattrl *)GET_NEXT(pal->al_link);
      }

    /* SUCCESS */
    return(PBSE_NONE);
    }    /* END if (pal != NULL) */

  /* attrlist not specified, return all readable attributes */

  for (index = 0;index < limit;index++)
    {
    if (((padef + index)->at_flags & priv) &&
        !((padef + index)->at_flags & ATR_DFLAG_NOSTAT))
      {
      if (!(((padef + index)->at_flags & ATR_DFLAG_PRIVR) && (IsOwner == 0)))
        {
        (padef + index)->at_encode(
          pattr + index,
          phead,
          (padef + index)->at_name,
          NULL,
          ATR_ENCODE_CLIENT,
          resc_access_perm);

        /* add walltime remaining if started */
        if ((index == JOB_ATR_start_time) &&
            ((pattr + index)->at_flags & ATR_VFLAG_SET))
          add_walltime_remaining(index, pattr, phead);
        }
      }
    }    /* END for (index) */

  /* SUCCESS */
  return(PBSE_NONE);
  }  /* END status_attrib() */

/* END stat_job.c */

