#include "license_pbs.h" /* See here for the software license */
/*
 * The entry point function for MOM.
 */
#include <pbs_config.h>   /* the master config generated by configure */
#include "mom_main.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#ifdef _CRAY
#include <termios.h>
#endif /* _CRAY */

#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <limits.h>
#include <netdb.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#if (PLOCK_DAEMONS & 4)
#include <sys/lock.h>
#endif /* PLOCK_DAEMONS */
#include <netinet/in.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#ifdef _CRAY
#include <sys/category.h>
#include <sys/usrv.h>
#include <sys/sysv.h>
#endif /* _CRAY */
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/utsname.h>
#include <dirent.h>


#include "libpbs.h"
#include "pbs_ifl.h"
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "pbs_job.h"
#include "mom_mach.h"
#include "mom_func.h"
#include "mom_comm.h"
#include "svrfunc.h"
#include "pbs_error.h"
#include "log.h"
#include "../lib/Liblog/pbs_log.h"
#include "../lib/Liblog/log_event.h"
#include "../lib/Liblog/chk_file_sec.h"
#include "../lib/Liblog/setup_env.h"
#include "../lib/Libnet/lib_net.h" /* socket_avail_bytes_on_descriptor */
#include "net_connect.h"
#include "rpp.h"
#include "dis.h"
#include "dis_init.h"
#include "resmon.h"
#include "pbs_nodes.h"
#include "dis.h"
#include "csv.h"
#include "utils.h"
#include "u_tree.h"
#ifdef PENABLE_LINUX26_CPUSETS
#include "pbs_cpuset.h"
#endif
#include "rpp.h"
#include "threadpool.h"
#include "mom_hierarchy.h"
#include "../lib/Libutils/u_lock_ctl.h" /* lock_init */
#include "mom_server.h"
#include "mom_job_func.h" /* mom_job_purge */
#include "net_cache.h"
#include "mom_job_cleanup.h"

#include "mcom.h"
#include "mom_server_lib.h" /* shutdown_to_server */

#ifdef NOPOSIXMEMLOCK
#undef _POSIX_MEMLOCK
#endif /* NOPOSIXMEMLOCK */

#ifdef _POSIX_MEMLOCK
#include <sys/mman.h>
#endif /* _POSIX_MEMLOCK */

#define MAX_UPDATES_BEFORE_SENDING  20
#define PMOMTCPTIMEOUT 60  /* duration in seconds mom TCP requests will block */

/* Global Data Items */

char  *program_name;
int    MOMIsLocked = 0;
int    MOMIsPLocked = 0;
int    ServerStatUpdateInterval = DEFAULT_SERVER_STAT_UPDATES;
int    CheckPollTime            = CHECK_POLL_TIME;
int    ForceServerUpdate = 0;

int    verbositylevel = 0;
double cputfactor = 1.00;
unsigned int default_server_port = 0;

int    exiting_tasks = 0;
float  ideal_load_val = -1.0;
int    internal_state = 0;

/* mom data items */
#ifdef NUMA_SUPPORT
int            num_node_boards;
nodeboard      node_boards[MAX_NODE_BOARDS]; 
int            numa_index;
#else
char           path_meminfo[MAX_LINE];
#endif

extern pthread_mutex_t *log_mutex;

int          thread_unlink_calls = FALSE;
/* by default, enforce these policies */
int          ignwalltime = 0; 
int          ignmem = 0;
int          igncput = 0;
int          ignvmem = 0; 
/* end policies */
int          spoolasfinalname = 0;
int          maxupdatesbeforesending = MAX_UPDATES_BEFORE_SENDING;
char        *apbasil_path     = NULL;
char        *apbasil_protocol = NULL;
int          reject_job_submit = 0;
int          attempttomakedir = 0;
int          reduceprologchecks;
int          lockfds = -1;
int          multi_mom = 0;
time_t       loopcnt;  /* used for MD5 calc */
float        max_load_val = -1.0;
int          hostname_specified = 0;
char         mom_host[PBS_MAXHOSTNAME + 1];
char         mom_alias[PBS_MAXHOSTNAME + 1];
char         TMOMRejectConn[MAXLINE];   /* most recent rejected connection */
char         mom_short_name[PBS_MAXHOSTNAME + 1];
int          num_var_env;
int          received_cluster_addrs;
time_t       requested_cluster_addrs;
time_t       first_update_time = 0;
char        *path_epilog;
char        *path_epilogp;
char        *path_epiloguser;
char        *path_epiloguserp;
char        *path_epilogpdel;
char        *path_jobs;
char        *path_prolog;
char        *path_prologp;
char        *path_prologuser;
char        *path_prologuserp;
char        *path_mom_hierarchy;
char        *path_spool;
char        *path_undeliv;
char        *path_aux;
char        *path_home = PBS_SERVER_HOME;
char        *mom_home;

extern dynamic_string *mom_status;
extern int  multi_mom;
extern unsigned int pbs_rm_port;
char        *path_layout;
extern char *msg_daemonname;          /* for logs     */
extern char *msg_info_mom; /* Mom information message   */
gid_t  pbsgroup;
uid_t pbsuser;
unsigned int pbs_mom_port = 0;
unsigned int pbs_rm_port = 0;
tlist_head mom_polljobs; /* jobs that must have resource limits polled */
tlist_head svr_newjobs; /* jobs being sent to MOM */
tlist_head svr_alljobs; /* all jobs under MOM's control */
tlist_head mom_varattrs; /* variable attributes */
int  termin_child = 0;  /* boolean - one or more children need to be terminated this iteration */
int  exec_with_exec = 0;
time_t  time_now = 0;
time_t  last_poll_time = 0;
extern tlist_head svr_requests;

extern struct var_table vtable; /* see start_exec.c */
double  wallfactor = 1.00;
long  log_file_max_size = 0;
long  log_file_roll_depth = 1;

time_t          last_log_check;
char           *nodefile_suffix = NULL;    /* suffix to append to each host listed in job host file */
char           *submithost_suffix = NULL;  /* suffix to append to submithost for interactive jobs */
char           *TNoSpoolDirList[TMAX_NSDCOUNT];
char           *TRemChkptDirList[TMAX_RCDCOUNT];

job            *JobsToResend[MAX_RESEND_JOBS];

resizable_array  *exiting_job_list;
resizable_array  *things_to_resend;
char             *AllocParCmd = NULL;  /* (alloc) */

int      src_login_batch = TRUE;
int      src_login_interactive = TRUE;

mom_hierarchy_t *mh;

char    jobstarter_exe_name[MAXPATHLEN + 1];
int     jobstarter_set = 0;

#ifdef PENABLE_LINUX26_CPUSETS
hwloc_topology_t topology = NULL;       /* system topology */

int      memory_pressure_threshold = 0; /* 0: off, >0: check and kill */
short    memory_pressure_duration  = 0; /* 0: off, >0: check and kill */
int      MOMConfigUseSMT           = 1; /* 0: off, 1: on */
#endif

int      is_reporter_mom = FALSE;
int      is_login_node   = FALSE;

/* externs */

extern char *server_alias;
extern unsigned int pe_alarm_time;
extern long     MaxConnectTimeout;

extern resizable_array *received_statuses;
extern hash_table_t    *received_table;

time_t          pbs_tcp_timeout = PMOMTCPTIMEOUT;
char            tmpdir_basename[MAXPATHLEN];  /* for $TMPDIR */

char            rcp_path[MAXPATHLEN];
char            rcp_args[MAXPATHLEN];
char            xauth_path[MAXPATHLEN];

time_t          LastServerUpdateTime = 0;  /* NOTE: all servers updated together */
int             UpdateFailCount = 0;

time_t          MOMStartTime         = 0;
int             MOMPrologTimeoutCount;
int             MOMPrologFailureCount;

char            MOMConfigVersion[64];
char            MOMUNameMissing[64];

int             MOMConfigDownOnError      = 0;
int             MOMConfigRestart          = 0;
int             MOMConfigRReconfig        = 0;
long            system_ncpus = 0;
char           *auto_ideal_load = NULL;
char           *auto_max_load   = NULL;
#ifdef NVIDIA_GPUS
int             MOMNvidiaDriverVersion    = 0;
int             use_nvidia_gpu = TRUE;
#endif  /* NVIDIA_GPUS */

#define TMAX_JE  64

pjobexec_t      TMOMStartInfo[TMAX_JE];


/* prototypes */

void            resend_things();
extern void     add_resc_def(char *, char *);
extern void     mom_server_all_diag(char **BPtr, int *BSpace);
extern void     mom_server_all_init(void);
extern void     mom_server_all_update_stat(void);
extern void     mom_server_all_update_gpustat(void);
extern int      mark_for_resend(job *);
extern int      mom_server_add(char *name);
extern int      mom_server_count;
extern int      post_epilogue(job *, int);
extern int      mom_checkpoint_init(void);
extern void     mom_checkpoint_check_periodic_timer(job *pjob);
extern void     mom_checkpoint_set_directory_path(char *str);

#ifdef NVIDIA_GPUS
#ifdef NVML_API
extern int      init_nvidia_nvml();
extern int      shut_nvidia_nvml();
#endif  /* NVML_API */
extern int      check_nvidia_setup();
#endif  /* NVIDIA_GPUS */

void prepare_child_tasks_for_delete();
static void mom_lock(int fds, int op);

#ifdef NUMA_SUPPORT
int setup_nodeboards();
#endif /* NUMA_SUPPORT */



/* Local Data Items */

static char *log_file = NULL;

enum PMOMStateEnum
  {
  MOM_RUN_STATE_RUNNING,
  MOM_RUN_STATE_EXIT,
  MOM_RUN_STATE_KILLALL,
  MOM_RUN_STATE_RESTART,
  MOM_RUN_STATE_LAST
  };

static enum PMOMStateEnum mom_run_state;

static int recover = JOB_RECOV_RUNNING;
static int recover_set = FALSE;

static int      call_hup = 0;
static int      nconfig;
static char    *path_log;

struct config_list
  {
  struct config       c;

  struct config_list *c_link;
  };

/* NOTE:  must adjust RM_NPARM in resmom.h to be larger than number of parameters
          specified below */

static unsigned long setxauthpath(char *);
static unsigned long setrcpcmd(char *);
static unsigned long setpbsclient(char *);
static unsigned long configversion(char *);
static unsigned long cputmult(char *);
static unsigned long setallocparcmd(char *);
static unsigned long setidealload(char *);
static unsigned long setignwalltime(char *);
static unsigned long setignmem(char *);
static unsigned long setigncput(char *);
static unsigned long setignvmem(char *);
static unsigned long setlogevent(char *);
static unsigned long setloglevel(char *);
static unsigned long setumask(char *);
static unsigned long setpreexec(char *);
static unsigned long setmaxload(char *);
static unsigned long setenablemomrestart(char *);
static unsigned long prologalarm(char *);
static unsigned long restricted(char *);
static unsigned long jobstartblocktime(char *);
static unsigned long usecp(char *);
static unsigned long wallmult(char *);
static unsigned long setpbsserver(char *);
static unsigned long setnodecheckscript(char *);
unsigned long setnodecheckinterval(char *);
static unsigned long settimeout(char *);
extern unsigned long mom_checkpoint_set_checkpoint_interval(char *);
extern unsigned long mom_checkpoint_set_checkpoint_script(char *);
extern unsigned long mom_checkpoint_set_restart_script(char *);
extern unsigned long mom_checkpoint_set_checkpoint_run_exe_name(char *);
static unsigned long setdownonerror(char *);
static unsigned long setstatusupdatetime(char *);
static unsigned long setcheckpolltime(char *);
static unsigned long settmpdir(char *);
static unsigned long setlogfilemaxsize(char *);
static unsigned long setlogfilerolldepth(char *);
static unsigned long setlogfilesuffix(char *);
static unsigned long setlogdirectory(char *);
static unsigned long setlogkeepdays(char *);
static unsigned long setvarattr(char *);
static unsigned long setautoidealload(char *);
static unsigned long setautomaxload(char *);
static unsigned long setnodefilesuffix(char *);
static unsigned long setnospooldirlist(char *);
static unsigned long setmomhost(char *);
static unsigned long setrreconfig(char *);
static unsigned long setsourceloginbatch(char *);
static unsigned long setsourcelogininteractive(char *);
static unsigned long setspoolasfinalname(char *);
static unsigned long setattempttomakedir(char *);
static unsigned long setremchkptdirlist(char *);
static unsigned long setmaxconnecttimeout(char *);
unsigned long aliasservername(char *);
unsigned long jobstarter(char *value);
#ifdef PENABLE_LINUX26_CPUSETS
static unsigned long setusesmt(char *);
static unsigned long setmempressthr(char *);
static unsigned long setmempressdur(char *);
#endif
static unsigned long setreduceprologchecks(char *);
static unsigned long setextpwdretry(char *);
static unsigned long setexecwithexec(char *);
static unsigned long setmaxupdatesbeforesending(char *);
static unsigned long setthreadunlinkcalls(char *);
static unsigned long setapbasilpath(char *);
static unsigned long setapbasilprotocol(char *);
static unsigned long setreportermom(char *);
static unsigned long setloginnode(char *);
static unsigned long setrejectjobsubmission(char *);
unsigned long rppthrottle(char *value);

static struct specials
  {
  char            *name;
  u_long(*handler)();
  } special[] = {
  { "alloc_par_cmd",       setallocparcmd },
  { "auto_ideal_load",     setautoidealload },
  { "auto_max_load",       setautomaxload },
  { "xauthpath",           setxauthpath },
  { "rcpcmd",              setrcpcmd },
  { "rcp_cmd",             setrcpcmd },
  { "pbsclient",           setpbsclient },
  { "configversion",       configversion },
  { "cputmult",            cputmult },
  { "ideal_load",          setidealload },
  { "ignwalltime",         setignwalltime },
  { "ignmem",              setignmem },
  { "igncput",             setigncput },
  { "ignvmem",             setignvmem },
  { "logevent",            setlogevent },
  { "loglevel",            setloglevel },
  { "max_load",            setmaxload },
  { "enablemomrestart",    setenablemomrestart },
  { "prologalarm",         prologalarm },
  { "restricted",          restricted },
  { "jobstartblocktime",   jobstartblocktime },
  { "usecp",               usecp },
  { "wallmult",            wallmult },
  { "clienthost",          setpbsserver },  /* deprecated - use pbsserver */
  { "pbsserver",           setpbsserver },
  { "node_check_script",   setnodecheckscript },
  { "node_check_interval", setnodecheckinterval },
  { "timeout",             settimeout },
  { "checkpoint_interval", mom_checkpoint_set_checkpoint_interval },
  { "checkpoint_script",   mom_checkpoint_set_checkpoint_script },
  { "restart_script",      mom_checkpoint_set_restart_script },
  { "checkpoint_run_exe",  mom_checkpoint_set_checkpoint_run_exe_name },
  { "down_on_error",       setdownonerror },
  { "status_update_time",  setstatusupdatetime },
  { "check_poll_time",     setcheckpolltime },
  { "tmpdir",              settmpdir },
  { "log_directory",       setlogdirectory },
  { "log_file_max_size",   setlogfilemaxsize },
  { "log_file_roll_depth", setlogfilerolldepth },
  { "log_file_suffix",     setlogfilesuffix },
  { "log_keep_days",       setlogkeepdays },
  { "varattr",             setvarattr },
  { "nodefile_suffix",     setnodefilesuffix },
  { "nospool_dir_list",    setnospooldirlist },
  { "mom_host",            setmomhost },
  { "remote_reconfig",     setrreconfig },
  { "job_output_file_umask", setumask },
  { "preexec",             setpreexec },
  { "source_login_batch",  setsourceloginbatch },
  { "source_login_interactive", setsourcelogininteractive },
  { "spool_as_final_name", setspoolasfinalname },
  { "remote_checkpoint_dirs", setremchkptdirlist },
  { "max_conn_timeout_micro_sec",   setmaxconnecttimeout },
  { "alias_server_name", aliasservername },
  { "job_starter", jobstarter},
#ifdef PENABLE_LINUX26_CPUSETS
  { "use_smt",                      setusesmt      },
  { "memory_pressure_threshold",    setmempressthr },
  { "memory_pressure_duration",     setmempressdur },
#endif
  { "reduce_prolog_checks",         setreduceprologchecks},
  { "rpp_throttle", rppthrottle },
  { "thread_unlink_calls", setthreadunlinkcalls },
  { "attempt_to_make_dir", setattempttomakedir },
  { "ext_pwd_retry",       setextpwdretry },
  { "exec_with_exec",      setexecwithexec },
  { "max_updates_before_sending", setmaxupdatesbeforesending },
  { "apbasil_path",        setapbasilpath },
  { "reject_job_submission", setrejectjobsubmission },
  { "apbasil_protocol",    setapbasilprotocol },
  { "reporter_mom",        setreportermom },
  { "login_node",          setloginnode },
  { NULL,                  NULL }
  };


static char *arch(struct rm_attribute *);
static char *opsys(struct rm_attribute *);
static char *requname(struct rm_attribute *);
static char *validuser(struct rm_attribute *);
static char *reqmsg(struct rm_attribute *);
char        *reqgres(struct rm_attribute *);
static char *reqstate(struct rm_attribute *);
static char *getjoblist(struct rm_attribute *);
static char *reqvarattr(struct rm_attribute *);
/* static char *nullproc(struct rm_attribute *); */


struct config common_config[] =
  {
  { "arch",      {arch} },             /* machine architecture           */
  { "opsys",     {opsys} },            /* operating system               */
  { "uname",     {requname} },         /* user name     ???              */
  { "validuser", {validuser} },        /* valid user    ???              */
  { "message",   {reqmsg} },           /* message       ???              */
  { "gres",      {reqgres} },          /* generic resource (licenses...) */
  { "state",     {reqstate} },         /* state of pbs_mom               */
  { "jobs",      {getjoblist} },       /* job list this pbs_mom          */
  { "varattr",   {reqvarattr} },       /* ???                            */
  { NULL,        {NULL} }
  };

int                     LOGLEVEL = 0;  /* valid values (0 - 10) */
int                     LOGKEEPDAYS = 0; /* days each log file should be kept before deleting */
int                     EXTPWDRETRY = 3; /* # of times to try external pwd check */
int                     DEBUGMODE = 0;
int                     DOBACKGROUND = 1;
char                    DEFAULT_UMASK[1024];
char                    PRE_EXEC[1024];
long                    TJobStartBlockTime = 5; /* seconds to wait for job to launch before backgrounding */
long                    TJobStartTimeout = 300; /* seconds to wait for job to launch before purging */


char                   *ret_string;
long                    ret_size;

struct config         *config_array = NULL;

struct config_list    *config_list = NULL;
sigset_t  allsigs;
int   rm_errno;
unsigned int            reqnum = 0;  /* the packet number */

int   port_care = TRUE; /* secure connecting ports */
uid_t   uid = 0;  /* uid we are running with */
unsigned int   alarm_time = 10; /* time before alarm */

extern AvlTree            okclients;  /* accept connections from */
char                  **maskclient = NULL; /* wildcard connections */
int   mask_num = 0;
int   mask_max = 0;
u_long   localaddr = 0;

char   extra_parm[] = "extra parameter(s)";
char   no_parm[]    = "required parameter not found";
char   varattr_delimiter[] = ";";

int   cphosts_num = 0;

struct cphosts         *pcphosts = NULL;

static int  config_file_specified = 0;
static char  config_file[_POSIX_PATH_MAX] = "config";

char                    PBSNodeMsgBuf[1024];
char                    PBSNodeCheckPath[1024];
int                     PBSNodeCheckInterval = 1;
int                     PBSNodeCheckProlog = 0;
int                     PBSNodeCheckEpilog = 0;
static char            *MOMExePath = NULL;
static time_t           MOMExeTime = 0;


/* sync w/#define JOB_SUBSTATE_XXX (in include/pbs_job.h)*/

const char *PJobSubState[] =
  {
  "TRANSIN",                /* Transit in, wait for commit */
  "TRANSICM",               /* Transit in, wait for commit */
  "TRNOUT",                 /* transiting job outbound */
  "TRNOUTCM",               /* transiting outbound, rdy to commit */
  "SUBSTATE04",
  "SUBSTATE05",
  "SUBSTATE06",
  "SUBSTATE07",
  "SUBSTATE08",
  "SUBSTATE09",
  "QUEUED",                 /* job queued and ready for selection */
  "PRESTAGEIN",             /* job queued, has files to stage in */
  "SUBSTATE12",
  "SYNCRES",                /* job waiting on sync start ready */
  "STAGEIN",                /* job staging in files then wait */
  "STAGEGO",                /* job staging in files and then run */
  "STAGECMP",               /* job stage in complete */
  "SUBSTATE17",
  "SUBSTATE18",
  "SUBSTATE19",
  "HELD",      /* job held - user or operator */
  "SYNCHOLD",  /* job held - waiting on sync regist */
  "DEPNHOLD",  /* job held - waiting on dependency */
  "SUBSTATE23",
  "SUBSTATE24",
  "SUBSTATE25",
  "SUBSTATE26",
  "SUBSTATE27",
  "SUBSTATE28",
  "SUBSTATE29",
  "WAITING",   /* job waiting on execution time */
  "SUBSTATE31",
  "SUBSTATE32",
  "SUBSTATE33",
  "SUBSTATE34",
  "SUBSTATE35",
  "SUBSTATE36",
  "STAGEFAIL", /* job held - file stage in failed */
  "SUBSTATE38",
  "SUBSTATE39",
  "PRERUN",    /* job sent to MOM to run */
  "STARTING",  /* final job start initiated */
  "RUNNING",   /* job running */
  "SUSPEND",   /* job suspended, CRAY only */
  "SUBSTATE44",
  "SUBSTATE45",
  "SUBSTATE46",
  "SUBSTATE47",
  "SUBSTATE48",
  "SUBSTATE49",
  "EXITING",   /* Start of job exiting processing */
  "EXIT_WAIT", /* Waiting for a response from other mom's */
  "STAGEOUT",  /* job staging out (other) files   */
  "STAGEDEL",  /* job deleteing staged out files  */
  "EXITED",    /* job exit processing completed   */
  "ABORT",     /* job is being aborted by server  */
  "NOTERM_REQUE",
  "PREOBIT",   /* preobit job status */
  "OBIT",      /* (MOM) job obit notice sent */
  "COMPLETED",
  "RERUN",     /* job is rerun, recover output stage */
  "RERUN1",    /* job is rerun, stageout phase */
  "RERUN2",    /* job is rerun, delete files stage */
  "RERUN3",    /* job is rerun, mom delete job */
  "RETSTD",    /* job has checkpoint file, return stdout / stderr files to server
                * spool dir so that job can be restarted
                */
  NULL
  };


/* sync w/#define IS_XXX */

const char *PBSServerCmds[] =
  {
  "NULL",
  "HELLO",
  "CLUSTER_ADDRS",
  "UPDATE",
  "STATUS",
  "GPU_STATUS",
  NULL
  };


/*
** These routines are in the "dependent" code.
*/

extern void dep_initialize(void);
extern void dep_cleanup(void);

/* External Functions */

extern void catch_child(int);
extern void init_abort_jobs(int);
extern void scan_for_exiting();
extern void scan_for_terminated();
extern int TMomCheckJobChild(pjobexec_t *, int, int *, int *);
extern int TMomFinalizeJob3(pjobexec_t *, int, int, int *);
extern void exec_bail(job *, int);
extern void check_state(int);


/* Local public functions */

static void    stop_me(int);
static void    PBSAdjustLogLevel(int);
int            TMOMScanForStarting(void);
unsigned long  getsize(resource *);
unsigned long  gettime(resource *);

/* Local private functions */

void check_log(void);





char *nullproc(

  struct rm_attribute *attrib)

  {
  log_err(-1, __func__, "should not be called");

  return(NULL);
  }  /* END nullproc() */




static char *arch(

  struct rm_attribute *attrib)  /* I */

  {
  struct config *cp;

  if (attrib != NULL)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  if (config_array == NULL)
    {
    return(PBS_MACH);
    }

  /* locate arch string */

  for (cp = config_array;cp->c_name != NULL;cp++)
    {
    if (cp->c_u.c_value == NULL)
      continue;

    if (strcmp(cp->c_name, "arch"))
      continue;

    return(cp->c_u.c_value);
    }  /* END for (cp) */

  return(PBS_MACH);
  }  /* END arch() */




static char *opsys(

  struct rm_attribute *attrib)  /* I */

  {
  struct config *cp;

  if (attrib != NULL)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  if (config_array == NULL)
    {
    return(PBS_MACH);
    }

  /* locate opsys string */

  for (cp = config_array;cp->c_name != NULL;cp++)
    {
    if (cp->c_u.c_value == NULL)
      continue;

    if (strcmp(cp->c_name, "opsys"))
      continue;

    return(cp->c_u.c_value);
    }  /* END for (cp) */

  return(PBS_MACH);
  }  /* END opsys() */





char *
getuname(void)

  {

  struct utsname  n;
  static char    *name = NULL;

  if (name == NULL)
    {
    if (uname(&n) == -1)
      {
      return(NULL);
      }

    sprintf(ret_string, "%s %s %s %s %s",

            n.sysname,
            n.nodename,
            n.release,
            n.version,
            n.machine);

    name = strdup(ret_string);
    }  /* END if (name == NULL) */

  return(name);
  }  /* END getuname() */




static char *reqmsg(

  struct rm_attribute *attrib)

  {
  if (attrib != NULL)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  return(PBSNodeMsgBuf);
  }  /* END reqmsg() */




static char *getjoblist(

  struct rm_attribute *attrib) /* I */

  {
  static char *list = NULL;
  static int listlen = 0;
  job *pjob;
  int firstjob = 1;

#ifdef NUMA_SUPPORT
  char  mom_check_name[PBS_MAXSERVERNAME];
  char *dot;
#endif 

  if (list == NULL)
    {
    if ((list = calloc(BUFSIZ + 50, sizeof(char)))==NULL)
      {
      /* FAILURE - cannot alloc memory */

      fprintf(stderr,"ERROR: could not calloc!\n");

      /* since memory cannot be allocated, report no jobs */

      return (" ");
      }

    listlen = BUFSIZ;
    }

  *list = '\0'; /* reset the list */

  if ((pjob = (job *)GET_NEXT(svr_alljobs)) == NULL)
    {
    /* no jobs - return space character */

    return(" ");
    }

#ifdef NUMA_SUPPORT 
  /* initialize the name to check for for this numa mom */
  strcpy(mom_check_name,mom_host);

  if ((dot = strchr(mom_check_name,'.')) != NULL)
    *dot = '\0';

  sprintf(mom_check_name + strlen(mom_check_name),"-%d/",numa_index);
#endif

  for (;pjob != NULL;pjob = (job *)GET_NEXT(pjob->ji_alljobs))
    {
#ifdef NUMA_SUPPORT
    /* skip over jobs that aren't on this node */
    if (strstr(pjob->ji_wattr[JOB_ATR_exec_host].at_val.at_str,mom_check_name) == NULL)
      continue;
#endif
    if (!firstjob)
      strcat(list, " ");

    strcat(list, pjob->ji_qs.ji_jobid);

    if ((int)strlen(list) >= listlen)
      {
      int   new_list_len = listlen + BUFSIZ;
      char *tmpList;

      tmpList = realloc(list, new_list_len);

      if (tmpList == NULL)
      	{
        /* FAILURE - cannot alloc memory */

        fprintf(stderr,"ERROR: could not realloc!\n");

        /* since memory cannot be allocated, report no jobs */

        return(" ");
      	}

      memset(tmpList + listlen, 0, new_list_len - listlen);

      list = tmpList;
      listlen = new_list_len;
      }

    firstjob = 0;
    }  /* END for (pjob) */

  if (list[0] == '\0')
    {
    /* no jobs - return space character */

    strcat(list, " ");
    }

  return(list);
  }  /* END getjoblist() */




#define TMAX_VARBUF   65536

static char *reqvarattr(

  struct rm_attribute *attrib)  /* I */

  {
  static char    *list = NULL;
  char           *child_spot;
  static int      listlen = 0;

  struct varattr *pva;
  int             fd, len, child_len;
  int             first_line;
  FILE           *child;

  char           *ptr;
  char           *ptr2;

  char            tmpBuf[TMAX_VARBUF + 1];

  if (list == NULL)
    {
    list = calloc(BUFSIZ + 1024, sizeof(char));

    if (list == NULL)
      {
      /* FAILURE - cannot alloc memory */

      log_err(errno, __func__, "cannot alloc memory");

      return(" ");
      }

    listlen = BUFSIZ;
    }

  *list = '\0'; /* reset the list */

  if ((pva = (struct varattr *)GET_NEXT(mom_varattrs)) == NULL)
    {
    return(" ");
    }

  for (;pva != NULL;pva = (struct varattr *)GET_NEXT(pva->va_link))
    {
    /* loop for each $varattr parameter */

    if ((pva->va_lasttime == 0) || (time_now >= (pva->va_ttl + pva->va_lasttime)))
      {
      if ((pva->va_ttl == -1) && (pva->va_lasttime != 0))
        {
        if (pva->va_value[0] != '\0')
          {
          if (*list != '\0')
            strcat(list, varattr_delimiter);

          strcat(list, pva->va_value);
          }

        if ((int)strlen(list) >= listlen)
          {
          listlen += BUFSIZ;

          list = realloc(list, listlen);

          if (list == NULL)
            {
            log_err(errno, __func__, "cannot alloc memory");

            return(" ");
            }
          }

        continue;  /* ttl of -1 is only run once */
        }

      /* TTL is satisfied, reload value */

      pva->va_lasttime = time_now;

      if (pva->va_value == NULL)
        pva->va_value = calloc(TMAX_VARBUF, sizeof(char));

      /* execute script and get a new value */

      if ((child = popen(pva->va_cmd, "r")) == NULL)
        {
        sprintf(pva->va_value, "error: %d %s",
          errno,
          strerror(errno));
        }
      else
        {
        fd = fileno(child);

        child_spot = tmpBuf;
        child_len  = 0;
        child_spot[0] = '\0';

        while (child_len < TMAX_VARBUF)
          {
          len = read(fd, child_spot, TMAX_VARBUF - child_len);

          if ((len <= 0) &&
              (errno != EINTR))
            break;
          else if (len < 0)
            continue;

          child_len  += len;
          child_spot += len;
          }

        if (len == -1)
          {
          /* FAILURE - cannot read var script output */
          log_err(errno, __func__, "pipe read");

          sprintf(pva->va_value, "? %d",
            RM_ERR_SYSTEM);

          pclose(child);

          continue;
          }

        /* SUCCESS */

        pclose(child);

        tmpBuf[child_len] = '\0';

        /* Transfer returned data into var value field */

        first_line = TRUE;

        ptr = strtok(tmpBuf,"\n;");

        ptr2 = pva->va_value;

        ptr2[0] = '\0';

        /*
         * OUTPUT FORMAT:  Take what script gives us.
         * Script should output 1 or more lines of Name=value1+value2+...
         */

        while (ptr != NULL)
          {
          if (!first_line)
            strcat(ptr2,varattr_delimiter);

          strcat(ptr2,ptr);

          first_line = FALSE;

          ptr = strtok(NULL,"\n;");
          }  /* END while (ptr != NULL) */
        }    /* END else ((child = popen(pva->va_cmd,"r")) == NULL) */
      }      /* END if ((pva->va_lasttime == 0) || ...) */

    if (pva->va_value[0] != '\0')
      {
      if (*list != '\0')
        strcat(list, varattr_delimiter);

      strcat(list, pva->va_value);
      }

    if ((int)strlen(list) >= listlen)
      {
      listlen += BUFSIZ;
      list = realloc(list, listlen);

      if (list == NULL)
        {
        log_err(errno, __func__, "cannot alloc memory");

        return(" ");
        }
      }
    }    /* END for (pva) */

  if (list[0] == '\0')
    strcat(list, " ");

  return(list);
  }  /* END reqvarattr() */





char *reqgres(

  struct rm_attribute *attrib)  /* I (ignored) */

  {
  struct config *cp;

  static char   GResBuf[1024];
  char          tmpLine[1024];

  int           sindex;

  if (attrib != NULL)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  /* build gres string */

  /* FORMAT:  <GRES>:<VALUE>[+<GRES>:<VALUE>]... */

  GResBuf[0] = '\0';

  if (config_array == NULL)
    {
    return(GResBuf);
    }

  for (cp = config_array; cp->c_name != NULL; cp++)
    {
    if (cp->c_u.c_value == NULL)
      continue;

    /* verify parameter is not special */

    for (sindex = 0;sindex < RM_NPARM;sindex++)
      {
      if (special[sindex].name == NULL)
        break;

      if (!strcmp(special[sindex].name, cp->c_name))
        break;
      }  /* END for (sindex) */

    if ((sindex < RM_NPARM) &&
        (special[sindex].name != NULL) &&
        (!strcmp(special[sindex].name, cp->c_name)))
      {
      /* specified parameter is special parameter */

      continue;
      }

    /* verify parameter is not common */
    for (sindex = 0;sindex < RM_NPARM;sindex++)
      {
      if (common_config[sindex].c_name == NULL)
        break;

      if (!strcmp(common_config[sindex].c_name, cp->c_name))
        break;
      }  /* END for (sindex) */

    if ((sindex < RM_NPARM) &&
        (common_config[sindex].c_name != NULL) &&
        !strcmp(common_config[sindex].c_name, cp->c_name) &&
        strcmp(common_config[sindex].c_name, "gres"))
      {
      /* specified parameter is common parameter */

      continue;
      }

    if (!strncmp(cp->c_name, "size", strlen("size")))
      continue;

    if (GResBuf[0] != '\0')
      {
      strncat(GResBuf, "+", sizeof(GResBuf) - 1 - strlen(GResBuf));
      }

    snprintf(tmpLine, 1024, "%s:%s",
             cp->c_name,
             cp->c_u.c_value);

    strncat(GResBuf, tmpLine, (sizeof(GResBuf) - strlen(GResBuf) - 1));
    }  /* END for (cp) */

  return(GResBuf);
  }  /* END reqgres() */




static char *reqstate(

  struct rm_attribute *attrib)  /* I (ignored) */

  {
  static char state[1024];

  if ((internal_state & INUSE_DOWN) && (MOMConfigDownOnError != 0))
    strcpy(state, "down");
  else if (internal_state & INUSE_BUSY)
    strcpy(state, "busy");
  else
    strcpy(state, "free");

  return(state);
  }  /* END reqstate() */




static char *requname(

  struct rm_attribute *attrib)

  {
  char *cp;

  if (attrib != NULL)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  cp = getuname();

  return(cp);
  }  /* END requname() */





static char *validuser(

  struct rm_attribute *attrib)

  {
  struct passwd *p;

  if ((attrib == NULL) || (attrib->a_value == NULL))
    {
    log_err(-1, __func__, no_parm);
    rm_errno = RM_ERR_NOPARAM;

    return(NULL);
    }

  p = getpwnam_ext(attrib->a_value);

  if (p != NULL)
    {
    return("yes");
    }

  return("no");
  }    /* END validuser() */


char *loadave(

  struct rm_attribute *attrib)

  {
#ifdef NUMA_SUPPORT
  /* the hardware provides no way to get information on specific cores,
   * so there's no way to obtain the load average of a numa node */
  return(NULL);
#else
  static char  ret_string[20];
  double       la;

  if (attrib)
    {
    log_err(-1, __func__, extra_parm);

    rm_errno = RM_ERR_BADPARAM;

    return(NULL);
    }

  if (get_la(&la) != 0)
    {
    rm_errno = RM_ERR_SYSTEM;

    return(NULL);
    }

  sprintf(ret_string, "%.2f",

          la);

  return(ret_string);
#endif /* NUMA_SUPPORT */
  }  /* END loadave() */


/*
** Search the array of resources read from the config files.
*/

struct config *rm_search(

        struct config *where,  /* I */
        char          *what)   /* I */

  {

  struct config *cp;

  if (where == NULL || what == NULL)
    {
    return NULL;
    }

  for (cp = where;cp->c_name != NULL;cp++)
    {
    if (strcmp(cp->c_name, what) == 0)
      {
      return(cp);
      }
    }    /* END for (cp) */

  return(NULL);
  }  /* END rm_search() */





/*
** Search the various resource lists.
*/

char *dependent(

  char               *res,  /* I */
  struct rm_attribute *attr) /* I */

  {

  struct config        *ap;

  extern struct config  standard_config[];

  extern struct config  dependent_config[];

  ap = rm_search(common_config, res);

  if (ap != NULL)
    {
    return(ap->c_u.c_func(attr));
    }

  ap = rm_search(standard_config, res);

  if (ap != NULL)
    {
    return(ap->c_u.c_func(attr));
    }

  ap = rm_search(dependent_config, res);

  if (ap != NULL)
    {
    return(ap->c_u.c_func(attr));
    }

  rm_errno = RM_ERR_UNKNOWN;

  return(NULL);
  }  /* END dependent() */





/*
** Initialize standard resource array
*/

void initialize(void)

  {
  log_record(PBSEVENT_SYSTEM, 0, __func__, "independent");

  dep_initialize();

  return;
  }




void cleanup(void)

  {
  dep_cleanup();

  return;
  }




/*
** Clean up after a signal.
*/

void die(

  int sig)

  {
  if (sig > 0)
    {
    sprintf(log_buffer, "caught signal %d",
            sig);

    log_record(PBSEVENT_SYSTEM, 0, __func__, log_buffer);
    }
  else
    {
    log_record(PBSEVENT_SYSTEM, 0, __func__, "abnormal termination");
    }

  cleanup();

#if defined(NVIDIA_GPUS) && defined(NVML_API)
  shut_nvidia_nvml();
#endif  /* NVIDIA_GPUS and NVML_API */

  log_close(1);

  exit(1);
  }  /* END die() */




/*
** Check for fatal memory allocation error.
*/

void memcheck(

  char *buf)

  {
  if (buf != NULL)
    {
    return;
    }

  log_err(-1, "memcheck", "memory allocation failed");

  die(0);

  return;
  }  /* END memcheck() */





/*
** Check the ret_string buffer to make sure that there is
** enought room starting at *spot to hold len characters more.
** If not, realloc the buffer and make *spot point to
** the corresponding place that it used to point to in
** the old buffer.
*/

void checkret(

  char **spot,
  long   len)

  {
  char *hold;

  if ((*spot - ret_string) < (ret_size - len))
    {
    return;
    }

  ret_size += len * 2;  /* new buf size */

  sprintf(log_buffer, "size increased to %ld",
          ret_size);

  log_record(PBSEVENT_SYSTEM, 0, __func__, log_buffer);

  hold = realloc(ret_string, ret_size); /* new buf */

  memcheck(hold);

  *spot = *spot - ret_string + hold;  /* new spot in buf */

  ret_string = hold;

  return;
  }  /* END checkret() */




char *skipwhite(

  char *str)

  {
  for (;*str;str++)
    {
    if (!isspace(*str))
      break;
    }

  return(str);
  }





char *tokcpy(

  char *str,
  char *tok)

  {
  for (;*str;str++, tok++)
    {
    if (!isalnum(*str) && *str != ':' && *str != '_')
      break;

    *tok = *str;
    }  /* END tokcpy() */

  *tok = '\0';

  return(str);
  }  /* END tokcpy() */




void rmnl(

  char *str)

  {
  int i;

  i = strlen(str);

  while (--i)
    {
    if ((*(str + i) != '\n') && !isspace((int)*(str + i)))
      break;

    *(str + i) = '\0';
    }

  return;
  }


/*
 * Parse a boolean config option value.
 * Return 1 (true), 0 (false), -1 (error).
 * Accepts: "true", "yes", "on", "1", "false", "no", "off", "0"
 */

static int setbool(

  char *value) /* I */

  {
  int enable = -1;

  if (value != NULL)
    {

    switch (value[0])
      {

      case 't':
      case 'T':
      case 'y':
      case 'Y':
      case '1':
        enable = 1;
        break;

      case 'f':
      case 'F':
      case 'n':
      case 'N':
      case '0':
        enable = 0;
        break;

      case 'o':
      case 'O':
        if ((strcasecmp(value,"on") == 0))
          enable = 1;
        else if ((strcasecmp(value,"off") == 0))
          enable = 0;
        break;

      }

    }

  return(enable);
  }



u_long addclient(

  char *name)  /* I */

  {
  struct addrinfo *addr_info;
  struct in_addr   saddr;
  u_long           ipaddr;

  /* FIXME: must be able to retry failed lookups later */

  if (getaddrinfo(name, NULL, NULL, &addr_info) != 0)
    {
    sprintf(log_buffer, "host %s not found",
            name);

    log_err(-1, __func__, log_buffer);

    return(0);
    }

  saddr = ((struct sockaddr_in *)addr_info->ai_addr)->sin_addr;
  freeaddrinfo(addr_info);

  ipaddr = ntohl(saddr.s_addr);

  okclients = AVL_insert(ipaddr, 0, NULL, okclients);
  
  return(ipaddr);
  }  /* END addclient() */





static u_long setpbsclient(

  char *value)  /* I */

  {
  u_long rc;

  if ((value == NULL) || (value[0] == '\0'))
    {
    /* FAILURE */

    return(1);
    }

  rc = addclient(value);

  if (rc != 0)
    {
    /* FAILURE */

    return(1);
    }

  return(0);
  }  /* END setpbsclient() */




/* FIXME: we need to handle a non-default port number */

static u_long setpbsserver(

  char *value)  /* I */

  {
  if ((value == NULL) || (*value == '\0'))
    {
    return(1);    /* FAILURE - nothing specified */
    }

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);


  return(mom_server_add(value));
  }  /* END setpbsserver() */




static u_long settmpdir(

  char *Value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, Value);

  if (*Value != '/')
    {
    log_err(-1, __func__, "tmpdir must be a full path");

    return(0);
    }

  snprintf(tmpdir_basename, sizeof(tmpdir_basename), "%s", Value);

  return(1);
  }




static u_long setexecwithexec(

  char *value)

  {
  static char *id = "setexecwithexec";

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, value);

  if (!strncasecmp(value, "t", 1) || (value[0] == '1') || !strcasecmp(value, "on") )
    exec_with_exec = 1;
  else
    exec_with_exec = 0;

  return(1);
  } /* END setexecwithexec() */




static u_long setxauthpath(

  char *Value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, Value);

  if (*Value != '/')
    {
    log_err(-1, __func__, "xauthpath must be a full path");

    return(0);
    }

  snprintf(xauth_path, sizeof(xauth_path), "%s", Value);

  return(1);
  }




static u_long setrcpcmd(

  char *Value)  /* I */

  {
  static char *ptr;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, Value);

  if (*Value != '/')
    {
    log_err(-1, __func__, "rcpcmd must be a full path");

    /* FAILURE */

    return(0);
    }

  snprintf(rcp_path, sizeof(rcp_path), "%s", Value);

  strcpy(rcp_args, "");

  if ((ptr = strchr(rcp_path, ' ')) != NULL)
    {
    *ptr = '\0';

    if (*(ptr + 1) != '\0')
      {
      snprintf(rcp_args, sizeof(rcp_args), "%s", ptr + 1);
      }
    }

  /* SUCCESS */

  return(1);
  }  /* END setrcpcmd() */





static u_long setlogevent(

  char *value)

  {
  char *bad;

  *log_event_mask = strtol(value, &bad, 0);

  if ((*bad == '\0') || isspace((int)*bad))
    {
    return(1);
    }

  return(0);
  }  /* END setlogevent() */





/* NOTE:  maskclient is global */

static u_long restricted(

  char *name)

  {
  char **tmpMaskClient;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, name);

  if (mask_max == 0)
    {
    if ((maskclient = (char **)calloc(4, sizeof(char *))) == NULL)
      {
      /* FAILURE - cannot alloc memory */

      log_err(errno, __func__, "cannot alloc memory");

      return(-1);
      }

    mask_max = 4;
    }

  maskclient[mask_num] = strdup(name);

  if (maskclient[mask_num] == NULL)
    {
    /* FAILURE - cannot alloc memory */

    log_err(errno, __func__, "cannot alloc memory");

    return(-1);
    }

  mask_num++;

  if (mask_num == mask_max)
    {
    mask_max *= 2;

    tmpMaskClient = (char **)realloc(
      maskclient,
      mask_max * sizeof(char *));

    if (tmpMaskClient == NULL)
      {
      /* FAILURE - cannot alloc memory */

      log_err(errno, __func__, "cannot alloc memory");

      return(-1);
      }

    maskclient = tmpMaskClient;
    }

  /* SUCCESS */

  return(1);
  }  /* END restricted() */





static u_long configversion(

  char *Value)  /* I */

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, Value);

  if (Value == NULL)
    {
    /* FAILURE */

    return(0);
    }

  snprintf(MOMConfigVersion, sizeof(MOMConfigVersion), "%s", Value);

  /* SUCCESS */

  return(1);
  }  /* END configversion() */





static u_long setdownonerror(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "down_on_error", value);

  if ((enable = setbool(value)) != -1)
    MOMConfigDownOnError = enable;

  return(1);
  }  /* END setdownonerror() */



static u_long setenablemomrestart(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "enablemomrestart", value);

  if ((enable = setbool(value)) != -1)
    MOMConfigRestart = enable;

  return(1);
  }  /* END setenablemomrestart() */




static u_long cputmult(

  char *value)  /* I */

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if ((cputfactor = atof(value)) == 0.0)
    {
    return(0); /* error */
    }

  return(1);
  }  /* END cputmult() */





static u_long wallmult(

  char *value)

  {
  double tmpD;

  if (value == NULL)
    {
    /* FAILURE */

    return(0);
    }

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  tmpD = atof(value);

  if ((tmpD == 0.0) && (value[0] != '\0'))
    {
    /* FAILURE */

    return(0);
    }

  /* SUCCESS */

  wallfactor = tmpD;

  return(1);
  }  /* END wallmult() */




static u_long usecp(

  char *value)  /* I */

  {
  char        *pnxt;
  static int   cphosts_max = 0;

  struct cphosts   *newp = NULL;

  /* FORMAT:  <HOST>:<FROM> <TO> */

  /*
   * HvB and Willem added this for logging purpose
   */

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if (cphosts_max == 0)
    {
    pcphosts = calloc(2, sizeof(struct cphosts));

    if (pcphosts == NULL)
      {
      sprintf(log_buffer, "%s: out of memory while allocating pcphosts",
        __func__);

      log_err(-1, __func__, log_buffer);

      return(0);
      }

    cphosts_max = 2;
    }
  else if (cphosts_max == cphosts_num)
    {
    newp = realloc(
      pcphosts,
      (cphosts_max + 2) * sizeof(struct cphosts));

    if (newp == NULL)
      {
      /* FAILURE */

      sprintf(log_buffer,"%s: out of memory while reallocating pcphosts",
        __func__);

      log_err(-1, __func__, log_buffer);

      return(0);
      }

    pcphosts = newp;

    cphosts_max += 2;
    }

  pnxt = strchr(value, (int)':');

  if (pnxt == NULL)
    {
    /* request failed */

    sprintf(log_buffer, "invalid host specification: %s",
      value);

    log_err(-1, __func__, log_buffer);

    return(0);
    }

  *pnxt++ = '\0';

  pcphosts[cphosts_num].cph_hosts = strdup(value);

  if (pcphosts[cphosts_num].cph_hosts == NULL)
    {
    /* FAILURE */

    sprintf(log_buffer, "%s: out of memory in strdup(cph_hosts)",
      __func__);

    log_err(-1, __func__, log_buffer);

    return(0);
    }

  value = pnxt; /* now ptr to path */

  while (!isspace(*pnxt))
    {
    if (*pnxt == '\0')
      {
      sprintf(log_buffer, "invalid '%s' specification %s: "
        "missing destination path",
        __func__,
        value);

      log_err(-1, __func__, log_buffer);

      free(pcphosts[cphosts_num].cph_hosts);

      return(0);
      }

    pnxt++;
    }

  *pnxt++ = '\0';

  pcphosts[cphosts_num].cph_from = strdup(value);

  if (pcphosts[cphosts_num].cph_from == NULL)
    {
    sprintf(log_buffer, "%s: out of memory in strdup(cph_from)",
      __func__);

    log_err(-1, __func__, log_buffer);

    free(pcphosts[cphosts_num].cph_hosts);

    return(0);
    }

  pcphosts[cphosts_num].cph_to = strdup(skipwhite(pnxt));

  if (pcphosts[cphosts_num].cph_to == NULL)
    {
    sprintf(log_buffer, "%s: out of memory in strdup(cph_to)",
      __func__);

    log_err(-1, __func__, log_buffer);

    free(pcphosts[cphosts_num].cph_hosts);
    free(pcphosts[cphosts_num].cph_from);

    return(0);
    }

  cphosts_num++;

  return(1);
  }  /* END usecp() */




static unsigned long prologalarm(

  char *value)  /* I */

  {
  int i;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "prologalarm",
    value);

  i = (int)atoi(value);

  if (i <= 0)
    {
    return(0); /* error */
    }

  pe_alarm_time = (unsigned int)i;

  return(1);
  }  /* END prologalarm() */





static unsigned long setloglevel(

  char *value)  /* I */

  {
  int i;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  i = (int)atoi(value);

  if (i < 0)
    {
    return(0);  /* error */
    }

  LOGLEVEL = (unsigned int)i;

  return(1);
  }  /* END setloglevel() */




static unsigned long setmaxupdatesbeforesending(
    
  char *value)

  {
  int i;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  i = (int)atoi(value);

  if (i < 0)
    return(0); /* error */

  maxupdatesbeforesending = i;

  /* SUCCESS */
  return(1);
  } /* END setmaxupdatesbeforesending() */





static unsigned long setumask(

  char *value)  /* I */

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "setumask",
    value);

  snprintf(DEFAULT_UMASK, sizeof(DEFAULT_UMASK), "%s", value);

  return(1);
  }  /* END setumask() */




static unsigned long setpreexec(

  char *value)  /* I */

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "setpreexec",
    value);

  snprintf(PRE_EXEC, sizeof(PRE_EXEC), "%s", value);

#if SHELL_USE_ARGV == 0
  log_err(0, __func__, "pbs_mom not configured with enable-shell-user-argv option");
#endif

  return(1);
  }  /* END setpreexec() */


static unsigned long setsourceloginbatch(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "source_login_batch", value);

  if ((enable = setbool(value)) != -1)
    src_login_batch = enable;

  return(1);
  }  /* END setsourceloginbatch() */


static unsigned long setsourcelogininteractive(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "source_login_interactive", value);

  if ((enable = setbool(value)) != -1)
    src_login_interactive = enable;

  return(1);
  }  /* END setsourcelogininteractive() */



static unsigned long jobstartblocktime(

  char *value)  /* I */

  {
  int i;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "startblocktime",
    value);

  i = (int)strtol(value, NULL, 10);

  if ((i < 0) || ((i == 0) && (value[0] != '0')))
    {
    return(0);  /* error */
    }

  TJobStartBlockTime = i;

  return(1);
  }  /* END jobstartblocktime() */





static unsigned long setstatusupdatetime(

  char *value)  /* I */

  {
  int i;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "setstateuspdatetime",
    value);

  i = (int)strtol(value, NULL, 10);

  if (i < 1)
    {
    return(0);  /* error */
    }

  ServerStatUpdateInterval = (unsigned int)i;

  return(1);
  }  /* END setstatusupdatetime() */





static unsigned long setcheckpolltime(

  char *value)  /* I */

  {
  int i;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "setcheckpolltime",
    value);

  i = (int)strtol(value, NULL, 10);

  if (i < 1)
    {
    return(0);  /* error */
    }

  CheckPollTime = (unsigned int)i;

  return(1);
  }  /* END setcheckpolltime() */




/*
** Add static resource or shell escape line from config file.
** This is a support routine for read_config().
*/

static void add_static(

  char *str,     /* I */
  char *file,    /* I */
  int   linenum) /* I */

  {
  int  i;
  char  name[50];

  struct config_list *cp;

  str = tokcpy(str, name); /* resource name */
  str = skipwhite(str);   /* resource value */

  /* FORMAT:  <ATTR> [!]<VALUE> */

  if (*str == '!') /* shell escape command */
    {
    /* remove trailing newline */

    rmnl(str);
    }
  else
    {
    /* get the value */
    i = strlen(str);

    while (--i)
      {
      /* strip trailing blanks */

      if (!isspace((int)*(str + i)))
        break;

      *(str + i) = '\0';
      }
    }

  cp = (struct config_list *)calloc(1, sizeof(struct config_list));

  memcheck((char *)cp);

  cp->c_link = config_list;
  cp->c.c_name = strdup(name);

  memcheck(cp->c.c_name);

  cp->c.c_u.c_value = strdup(str);

  memcheck(cp->c.c_u.c_value);

  sprintf(log_buffer, "%s[%d] add name %s value %s",
    file,
    linenum,
    name,
    str);

  log_record(PBSEVENT_DEBUG, 0, "add_static", log_buffer);

  config_list = cp;

  return;
  }  /* END add_static() */





static unsigned long setidealload(

  char *value)

  {
  char  newstr[50] = "ideal_load ";
  float  val;

  val = atof(value);

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "ideal_load",
    value);

  if (val < 0.0)
    {
    return(0); /* error */
    }

  ideal_load_val = val;

  if (max_load_val < 0.0)
    max_load_val = val; /* set a default */

  strcat(newstr, value);

  return(1);
  }  /* END setidealload() */




static unsigned long setignwalltime(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "ignwalltime", value);

  if ((enable = setbool(value)) != -1)
    ignwalltime = enable;

  return(1);
  }  /* END setignwalltime() */



static unsigned long setignmem(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "ignmem", value);

  if ((enable = setbool(value)) != -1)
    ignmem = enable;

  return(1);
  } /* END setignmem() */



static unsigned long setigncput(

  char *value) /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "igncput", value);

  if ((enable = setbool(value)) != -1)
    igncput = enable;

  return(1);
  }


static unsigned long setignvmem(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "ignvmem", value);

  if ((enable = setbool(value)) != -1)
    ignvmem = enable;

  return(1);
  }  /* END setignvmem() */


static unsigned long setautoidealload(

  char *value)

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "auto_ideal_load",
    value);

  auto_ideal_load = strdup(value);

  /*
    add_static(auto_ideal_load,"config",0);

    nconfig++;
  */

  return(1);
  }  /* END setautoidealload() */





static unsigned long setallocparcmd(

  char *value)  /* I */

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "allocparcmd",
    value);

  AllocParCmd = strdup(value);

  return(1);
  }  /* END setallocparcmd() */





static unsigned long setautomaxload(

  char *value)

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "auto_max_load",
    value);

  auto_max_load = strdup(value);

  /*
    add_static(auto_ideal_load,"config",0);

    nconfig++;
  */

  return(1);
  }  /* END setautomaxload() */





static unsigned long setmaxconnecttimeout(

  char *value)  /* I */

  {
  MaxConnectTimeout = strtol(value, NULL, 10);

  if (MaxConnectTimeout < 0)
    {
    MaxConnectTimeout = 10000;

    return(0);
    }

  return(1);
  }



static unsigned long setreduceprologchecks(

  char *value)

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "reduceprologchecks",
    value);
  
  if (!strncasecmp(value,"t",1) || (value[0] == '1') || !strcasecmp(value,"on") )
    reduceprologchecks = TRUE;
  else
    reduceprologchecks = FALSE;

  return(1);
  } /* END setreduceprologchecks() */



static unsigned long setnodecheckscript(

  char *value)

  {
  char   newstr[1024] = "node_check_script ";

  struct stat sbuf;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "node_check_script",
    value);

  if ((stat(value, &sbuf) == -1) || !(sbuf.st_mode & S_IXUSR))
    {
    /* FAILURE */

    /* file does not exist or is not executable */

    return(0);
    }

  snprintf(PBSNodeCheckPath, sizeof(PBSNodeCheckPath), "%s", value);

  strcat(newstr, value);

  /* SUCCESS */

  return(1);
  }  /* END setnodecheckscript() */





unsigned long setnodecheckinterval(

  char *value)

  {
  char newstr[1024] = "node_check_interval ";

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "node_check_interval",
    value);

  PBSNodeCheckInterval = (int)strtol(value, NULL, 10);

  if (strstr(value, "jobstart"))
    PBSNodeCheckProlog = 1;

  if (strstr(value, "jobend"))
    PBSNodeCheckEpilog = 1;

  strcat(newstr, value);

  return(1);
  }  /* END setnodecheckinterval() */



static unsigned long settimeout(

  char *value)

  {
  char newstr[1024];

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "timeout",
    value);

  DIS_tcp_settimeout(strtol(value, NULL, 10));

  snprintf(newstr, sizeof(newstr), "%s %s",
           "timeout",
           value);

  return(1);
  }  /* END settimeout() */





static unsigned long setmaxload(

  char *value)  /* I */

  {
  char  newstr[50] = "max_load ";
  float  val;

  val = atof(value);

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "max_load", value);

  if (val < 0.0)
    {
    return(0); /* error */
    }

  max_load_val = val;

  if (ideal_load_val < 0.0)
    ideal_load_val = val;

  strcat(newstr, value);

  return(1);
  }  /* END max_load() */





static unsigned long setlogfilemaxsize(

  char *value)  /* I */

  {
  log_file_max_size = strtol(value, NULL, 10);

  if (log_file_max_size < 0)
    {
    log_file_max_size = 0;

    return(0);
    }

  return(1);
  }


unsigned long rppthrottle(

  char *value)  /* I */

  {
  long rpp_throttle_time;
  rpp_throttle_time = strtol(value, NULL, 10);
  /* call into Libifl to tell it what the sleep time is */
  set_rpp_throttle_sleep_time(rpp_throttle_time);

  return(rpp_throttle_time);
  }





static unsigned long setlogfilerolldepth(

  char *value)  /* I */

  {
  log_file_roll_depth = strtol(value, NULL, 10);

  if (log_file_roll_depth < 1)
    {
    log_file_roll_depth = 1;

    return(0);
    }

  return(1);
  }



static unsigned long setlogdirectory(

  char *value)  /* I */

  {
  path_log = strdup(value);

  return(1);
  }




static unsigned long setlogfilesuffix(

  char *value)  /* I */

  {
  log_init(value, NULL);

  return(1);
  }



static unsigned long setlogkeepdays(

  char *value)  /* I */

  {
  int i;

  i = (int)atoi(value);

  if (i < 0)
    {
    return(0);  /* error */
    }

  LOGKEEPDAYS = i;

  return(1);
  }


static unsigned long setextpwdretry(

  char *value)  /* I */

  {
  int i;

  i = (int)atoi(value);

  if (i < 0)
    {
    return(0);  /* error */
    }

  EXTPWDRETRY = i;

  return(1);
  } /* END setextpwdretry() */



static u_long setvarattr(

  char *value)  /* I */

  {
  struct varattr *pva;
  char           *ptr;

  pva = calloc(1, sizeof(struct varattr));

  if (pva == NULL)
    {
    /* FAILURE */

    log_err(errno, __func__, "no memory");

    return(0);
    }

  CLEAR_LINK(pva->va_link);

  /* FORMAT:  <TTL> <PATH> */
  /* extract TTL */

  ptr = value;

  pva->va_ttl = strtol(ptr, NULL, 10);

  /* step forward to end of TTL */

  while ((!isspace(*ptr)) &&
         (*ptr != '\0'))
    ptr++;

  if (*ptr == '\0')
    {
    free(pva);

    return(0);
    }

  /* skip white space */

  while (isspace(*ptr))
    ptr++;

  if (*ptr == '\0')
    {
    free(pva);

    return(0);
    }

  /* preserve command and args */

  pva->va_cmd = strdup(ptr);

  append_link(&mom_varattrs, &pva->va_link, pva);

  /* SUCCESS */

  return(1);
  }  /* END setvarattr() */




static unsigned long setthreadunlinkcalls(

  char *value) /* I */

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "threadunlinkcalls",
    value);

  if (!strncasecmp(value,"t",1) || (value[0] == '1') || !strcasecmp(value,"on") )
    thread_unlink_calls = TRUE;
  else
    thread_unlink_calls = FALSE;

  return(1);
  } /* END setthreadunlinkcalls() */





static unsigned long setnodefilesuffix(

  char *value)  /* I */

  {
  char *ptr;

  ptr = strtok(value, ",");

  nodefile_suffix = strdup(ptr);

  ptr = strtok(NULL, ",");

  if (ptr != NULL)
    submithost_suffix = strdup(ptr);

  /* SUCCESS */

  return(1);
  }  /* END setnodexfilesuffix() */




static unsigned long setmomhost(

  char *value)  /* I */

  {
  hostname_specified = 1;

  snprintf(mom_host, sizeof(mom_host), "%s", value);

  /* SUCCESS */

  return(1);
  }  /* END setmomhost() */


static u_long setrreconfig(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "remote_reconfig", value);

  if ((enable = setbool(value)) != -1)
    MOMConfigRReconfig = enable;

  return(1);
  }  /* END setrreconfig() */


static unsigned long setnospooldirlist(

  char *value)  /* I */

  {
  char *TokPtr;
  char *ptr;

  int   index = 0;

  char  tmpLine[1024];

  ptr = strtok_r(value, " \t\n:,", &TokPtr);

  while (ptr != NULL)
    {
    TNoSpoolDirList[index] = strdup(ptr);

    snprintf(tmpLine, sizeof(tmpLine), "added NoSpoolDir[%d] '%s'",
             index,
             ptr);

    log_record(
      PBSEVENT_SYSTEM,
      PBS_EVENTCLASS_SERVER,
      "setnospooldirlist",
      tmpLine);

    index++;

    if (index >= TMAX_NSDCOUNT)
      break;

    ptr = strtok_r(NULL, " \t\n:,", &TokPtr);
    }  /* END while (ptr != NULL) */

  /* SUCCESS */

  return(1);
  }  /* END setnospooldirlist() */


unsigned long aliasservername( char *value)
  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if (value)
    {
    server_alias = (char *)calloc(1, strlen(value)+1);
    if (server_alias)
      {
      strcpy(server_alias, value);
      }
    }
  
  return(1);
  } /* END aliasservername() */




static unsigned long setspoolasfinalname(

  char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "spool_as_final_name", value);

  if ((enable = setbool(value)) != -1)
    spoolasfinalname = enable;

  return(1);
  }  /* END setspoolasfinalname() */



static unsigned long setapbasilpath(

  char *value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "apbasil_path", value);

  if (value != NULL)
    {
    if (value[0] == '/')
      apbasil_path = strdup(value);
    else
      {
      snprintf(log_buffer, sizeof(log_buffer),
        "Path must be an absolute path, but is %s", value);
      log_err(-1, __func__, log_buffer);
      }
    }

  return(1);
  } /* END setapbasilpath() */



static unsigned long setapbasilprotocol(

  char *value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "apbasil_protocol", value);

  if (value != NULL)
    {
    /* only versions 1.0 to 1.3 are supported for now. update later */
    if ((strlen(value) != 3) ||
        (value[0] != '1') ||
        (value[1] != '.') ||
        ((value[2] != '0') &&
         (value[2] != '1') &&
         (value[2] != '2') &&
         (value[2] != '3')))
      {
      snprintf(log_buffer, sizeof(log_buffer), 
        "Value must be 1.[0-3] but is %s", value);
      log_err(-1, __func__, log_buffer);
      }
    else
      apbasil_protocol = strdup(value);
    }

  return(1);
  } /* END setapbasilprotocol() */




static unsigned long setreportermom(

  char *value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if (value != NULL)
    {
    if ((value[0] == 't') ||
        (value[0] == 'T') ||
        (value[0] == 'y') ||
        (value[0] == 'Y'))
      is_reporter_mom = TRUE;
    else
      is_reporter_mom = FALSE;
    }

  return(1);
  } /* END setreportermom() */




static unsigned long setloginnode(

  char *value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if (value != NULL)
    {
    if ((value[0] == 't') ||
        (value[0] == 'T') ||
        (value[0] == 'y') ||
        (value[0] == 'Y'))
      is_login_node = TRUE;
    else
      is_login_node = FALSE;
    }

  return(1);
  } /* END setloginnode() */




static unsigned long setrejectjobsubmission(

  char *value)

  {
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, value);

  if (!strncasecmp(value,"t",1) ||
      (value[0] == '1') ||
      (!strcasecmp(value,"on")))
    reject_job_submit = TRUE;
  else
    reject_job_submit = FALSE;

  return(1);
  } /* END setrejectjobsubmission() */




static unsigned long setattempttomakedir(

  char *value)

  {
  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "attempttomakedir",
    value);

  if (!strncasecmp(value,"t",1) || (value[0] == '1') || !strcasecmp(value,"on") )
    attempttomakedir = TRUE;
  else
    attempttomakedir = 0;

  return(1);
  } /* END setattempttomakedir() */


/********************************************************
 *  jobstarter - set the name of the job starter program
 *  to the value given in the mom config file
 *
 *  Return: 1 success
 *          0 file named by value does not exist. fail
 *******************************************************/
unsigned long jobstarter(char *value)  /* I */
  {
  struct stat sbuf;

  log_record(
    PBSEVENT_SYSTEM,
    PBS_EVENTCLASS_SERVER,
    "checkpoint_run_exe",
    value);

  if ((stat(value, &sbuf) == -1) || !(sbuf.st_mode & S_IXUSR))
    {
    /* file does not exist or is not executable */

    return(0);  /* error */
    }

  snprintf(jobstarter_exe_name, sizeof(jobstarter_exe_name), "%s", value);

  jobstarter_set = 1;

  return(1);
  }  /* END jobstarter() */




static unsigned long setremchkptdirlist(

  char *value)  /* I */

  {
  char *TokPtr;
  char *ptr;

  int   index = 0;
  char  tmpLine[1024];

  while ((TRemChkptDirList[index] != NULL) && (index < TMAX_RCDCOUNT))
    {
    index++;
    }

  if (index >= TMAX_RCDCOUNT)
    return (1);

  ptr = strtok_r(value, " \t\n:,", &TokPtr);

  while (ptr != NULL)
    {
    TRemChkptDirList[index] = strdup(ptr);

    snprintf(tmpLine, sizeof(tmpLine), "added RemChkptDir[%d] '%s'",
             index,
             ptr);

    log_record(
      PBSEVENT_SYSTEM,
      PBS_EVENTCLASS_SERVER,
      "setremchkptdirlist",
      tmpLine);

    index++;

    if (index >= TMAX_RCDCOUNT)
      break;

    ptr = strtok_r(NULL, " \t\n:,", &TokPtr);
    }  /* END while (ptr != NULL) */

  /* SUCCESS */

  return (1);
  }  /* END setremchkptdirlist() */





void check_log(void)

  {
  last_log_check = time_now;

  /* periodically record the version and loglevel */

  sprintf(log_buffer, msg_info_mom, PACKAGE_VERSION, LOGLEVEL);

  log_event(
    PBSEVENT_SYSTEM | PBSEVENT_FORCE,
    PBS_EVENTCLASS_SERVER,
    msg_daemonname,
    log_buffer);

  if (LOGKEEPDAYS > 0)
    {
    /* remove logs older than log_keep_days */

    snprintf(log_buffer,sizeof(log_buffer),"checking for old pbs_mom logs in dir '%s' (older than %d days)",
      path_log,
      LOGKEEPDAYS);

    log_event(
      PBSEVENT_SYSTEM | PBSEVENT_FORCE,
      PBS_EVENTCLASS_SERVER,
      msg_daemonname,
      log_buffer);

    if (log_remove_old(path_log,(LOGKEEPDAYS * SECS_PER_DAY)) != 0)
      {
      log_err(-1,"check_log","failure occurred when checking for old pbs_mom logs");
      }
    }

  if (log_file_max_size <= 0)
    {
    return;
    }

  if (log_size() >= log_file_max_size)
    {
    log_event(
      PBSEVENT_SYSTEM | PBSEVENT_FORCE,
      PBS_EVENTCLASS_SERVER,
      msg_daemonname,
      "Rolling log file");

    log_roll(log_file_roll_depth);
    }

  return;
  }  /* END check_log() */





/*
** Open and read the config file.  Save information in a linked
** list.  After reading the file, create an array, copy the list
** elements to the array and free the list.
*/

/* NOTE:  add new mom config parameters to 'special[]' */

int read_config(

  char *file)  /* I */

  {
  FILE                 *conf;

  struct stat            sb;

  struct config_list *cp;

  struct config  *ap;
  char                   line[120];
  char                   name[50];
  char                  *str;
  char                  *ptr;

  int                    linenum;
  int                    i;

  int                    IgnConfig = 0;

  int                    rc;

  int n, list_len;
  char *server_list_ptr;
  char *tp;

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer, "updating configuration using file '%s'",
            (file != NULL) ? file : "NULL");

    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, log_buffer);
    }

  for (i = 0;i < mask_num;i++)
    {
    free(maskclient[i]);
    }

  mask_num = 0;

  if (file == NULL)
    file = config_file;

  rc = 0;

  if (file[0] == '\0')
    {
    log_record(
      PBSEVENT_SYSTEM,
      PBS_EVENTCLASS_SERVER,
      __func__,
      "ALERT:  no config file specified");

    IgnConfig = 1;  /* no config file */
    }

  if ((IgnConfig == 0) && (stat(file, &sb) == -1))
    {
    IgnConfig = 1;

    sprintf(log_buffer, "fstat: %s",
            file);

    log_err(errno, __func__, log_buffer);

    if (config_file_specified != 0)
      {
      /* file specified and not there, return failure */

      log_record(
        PBSEVENT_SYSTEM,
        PBS_EVENTCLASS_SERVER,
        __func__,
        "ALERT:  cannot open config file - no file");

      rc = 1;
      }
    else
      {
      /* "config" file not located, return success */

      if (LOGLEVEL >= 3)
        {
        sprintf(log_buffer, "cannot open file '%s'",
                file);

        log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, log_buffer);
        }

      rc = 0;
      }
    }  /* END if ((IgnConfig == 0) && (stat(file,&sb) == -1)) */

  if (IgnConfig == 0)
    {
#if !defined(DEBUG) && !defined(NO_SECURITY_CHECK)

    if (chk_file_sec(file, 0, 0, S_IWGRP | S_IWOTH, 1, NULL))
      {
      /* not authorized to access specified file, return failure */

      log_record(
        PBSEVENT_SYSTEM,
        PBS_EVENTCLASS_SERVER,
        __func__,
        "ALERT:  cannot open config file - permissions");

      IgnConfig = 1;

      rc = 1;
      }

#endif  /* NO_SECURITY_CHECK */
    }    /* END if (ignConfig == 0) */

  if (IgnConfig == 0)
    {
    if ((conf = fopen(file, "r")) == NULL)
      {
      sprintf(log_buffer, "fopen: %s",
              file);

      log_err(errno, __func__, log_buffer);

      IgnConfig = 1;

      rc = 1;
      }
    }    /* END if (IgnConfig == 0) */

  if (IgnConfig == 0)
    {
    nconfig = 0;
    linenum = 0;

    memset(line, 0, sizeof(line));

    while (fgets(line, sizeof(line) - 1, conf))
      {
      linenum++;

      memset(name, 0, sizeof(name));

      if (line[0] == '#') /* comment */
        {
        memset(line, 0, sizeof(line));
        continue;
        }

      if ((ptr = strchr(line, '#')) != NULL)
        {
        /* allow inline comments */

        *ptr = '\0';
        }

      str = skipwhite(line); /* pass over initial whitespace */

      if (*str == '\0')
        {
        memset(line, 0, sizeof(line));
        continue;
        }

      if (LOGLEVEL >= 6)
        {
        sprintf(log_buffer, "processing config line '%.64s'", str);

        log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, log_buffer);
        }

      if (*str == '$')
        {
        /* special command */

        str = tokcpy(++str, name); /* resource name */

        for (i = 0;special[i].name;i++)
          {
          if (strcasecmp(name, special[i].name) == 0)
            break;
          }  /* END for (i) */

        if (special[i].name == NULL)
          {
          /* didn't find it */

          sprintf(log_buffer, "special command name %s not found (ignoring line)",
                  name);

          log_err(-1, __func__, log_buffer);
    
          memset(line, 0, sizeof(line));

          continue;
          }

        str = skipwhite(str);  /* command param */

        rmnl(str);

        if (special[i].handler(str) == 0)
          {
          sprintf(log_buffer, "%s[%d] special command %s failed with %s",
            file,
            linenum,
            name,
            str);

          log_err(-1, __func__, log_buffer);
          }

        memset(line, 0, sizeof(line));

        continue;
        }

      add_static(str, file, linenum);

      nconfig++;

      memset(line, 0, sizeof(line));
      }  /* END while (fgets()) */

    /*
    ** Create a new array.
    */

    if (config_array != NULL)
      {
      for (ap = config_array;ap->c_name != NULL;ap++)
        {
        free(ap->c_name);
        free(ap->c_u.c_value);
        }

      free(config_array);
      }

    config_array = (struct config *)calloc(nconfig + 1, sizeof(struct config));

    memcheck((char *)config_array);

    /*
    ** Copy in the new information saved from the file.
    */

    for (i = 0, ap = config_array;i < nconfig;i++, ap++)
      {
      *ap = config_list->c;
      cp = config_list->c_link;

      free(config_list); /* don't free name and value strings */
      config_list = cp; /* they carry over from the list */
      }

    ap->c_name = NULL;  /* one extra */

    fclose(conf);
    }  /* END if (IgnConfig == 0) */

  if (mom_server_count == 0)
    {
    /* No server names in torque/mom_priv/config.  Get names from torque/server_name. */

    server_list_ptr = pbs_get_server_list();
    list_len = csv_length(server_list_ptr);

    for (n = 0; n < list_len; n++)
      {
      tp = csv_nth(server_list_ptr, n);

      if (tp)
        {
        /* Trim any leading space */
        while(isspace(*tp)) tp++;

        setpbsserver(tp);
        }
      }
    }

  return(rc);
  }  /* END read_config() */



#ifdef PENABLE_LINUX26_CPUSETS
static u_long setusesmt(

    char *value)  /* I */

  {
  int enable;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "use_smt", value);

  if ((enable = setbool(value)) != -1)
    MOMConfigUseSMT = enable;

  return(1);
  }  /* END setusesmt() */


static u_long setmempressthr(

  char *value)

  {
  long long val;

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "memory_pressure_threshold", value);

  if ((val = atoll(value)) < 0)
    return(0);  /* error, leave as is */

  if (val > INT_MAX)
    val = INT_MAX;

  memory_pressure_threshold = (int)val;

  return(1);
  }

static u_long setmempressdur(

  char *value)

  {
  int val;

  val = atoi(value);

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "memory_pressure_duration", value);

  if ((val = atoi(value)) < 0)
    return(0);  /* error, leave as is */

  if (val > SHRT_MAX)
    val = SHRT_MAX;

  memory_pressure_duration = (short)val;

  return(1);
  }
#endif



/*
** Get an rm_attribute structure from a string.  If a NULL is passed
** for the string, use the previously remembered string.
*/

struct rm_attribute *momgetattr(

  char *str) /* I */

  {
  static char  cookie[] = "tag:"; /* rm_attribute to ignore */
  static char *hold = NULL;
  static char  qual[80] = "";
  static char  valu[4096] = "";

  static struct rm_attribute attr =
    {
    qual, valu
    };

  int         level, i;

  if (str == NULL) /* if NULL is passed, use prev value */
    str = hold;

  if (str == NULL)
    return(NULL);

  /* FORMAT: ??? */

  do
    {
    str = skipwhite(str);

    if (*str++ != '[')
      {
      return(NULL);
      }

    str = skipwhite(str);  /* copy qualifier */

    str = tokcpy(str, qual);
    str = skipwhite(str);

    if (*str++ != '=')
      {
      return(NULL);
      }

    level = 0;

    for (i = 0;*str;str++, i++)
      {
      if (*str == '[')
        {
        level++;
        }
      else if (*str == ']')
        {
        if (level == 0)
          break;

        level--;
        }

      valu[i] = *str;
      }

    if (*str++ != ']')
      {
      return(NULL);
      }

    valu[i] = '\0';

    if (LOGLEVEL >= 7)
      {
      sprintf(log_buffer, "found %s = %s",
              qual,
              valu);

      log_record(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, __func__, log_buffer);
      }
    }
  while (strncmp(qual, cookie, sizeof(cookie) - 1) == 0);

  hold = str;

  if (LOGLEVEL >= 5)
    {
    sprintf(log_buffer, "passing back %s = %s",
            qual,
            valu);

    log_record(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, __func__, log_buffer);
    }

  return(&attr);
  }  /* END momgetattr() */





/*
** Check the request against the format of the line read from
** the config file.  If it is a static value, there should be
** no params.  If it is a shell escape, the parameters (if any)
** should match the command line for the system call.
*/

char *conf_res(

  char               *resline, /* I */
  struct rm_attribute *attr)    /* I */

  {
  char *name[RM_NPARM];
  char *value[RM_NPARM];
  int   used[RM_NPARM];  /* (boolean) */
  char  param[80];
  char *d;
  int   i;
  int   fd;
  int   len;
  FILE *child;
  char *child_spot;
  int   child_len;

  if (resline == NULL)
    {
    return("");
    }

  if (resline[0] != '!')
    {
    /* static value */

    if (attr != NULL)
      {
      sprintf(ret_string, "? %d",
              RM_ERR_BADPARAM);

      return(ret_string);
      }

    return(resline);
    }

  /*
  ** From here on we are going to put together a shell command
  ** to do the requestor's bidding.  Parameter substitution
  ** is the first step.
  */

  for (i = 0; i < RM_NPARM - 1; i++)
    {
    /* remember params */

    if (attr == NULL)
      {
      /* FAILURE */

      break;
      }

    name[i] = strdup(attr->a_qualifier);

    memcheck(name[i]);

    value[i] = strdup(attr->a_value);

    memcheck(value[i]);

    used[i] = 0;

    attr = momgetattr(NULL);
    }  /* END for (i) */

  if (attr != NULL)
    {
    /* too many params */
    log_err(-1, __func__, "too many params");

    sprintf(ret_string, "? %d", RM_ERR_BADPARAM);

    goto done;
    }

  name[i] = '\0';

  for (d = ret_string, resline++;*resline;)
    {
    /* scan command */

    if (*resline == '%')
      {
      /* possible token */

      char *hold;

      hold = tokcpy(resline + 1, param);

      for (i = 0;name[i];i++)
        {
        if (strcmp(param, name[i]) == 0)
          break;
        }

      if (name[i])
        {
        /* found a match */

        char *x = value[i];

        while (*x)
          {
          *d++ = *x++;
          }

        resline = hold;

        used[i] = 1;
        }
      else
        {
        *d++ = *resline++;
        }
      }
    else
      {
      *d++ = *resline++;
      }
    }

  for (i = 0;name[i];i++)
    {
    if (!used[i])
      {
      /* parameter sent but not used */
      log_err(-1, __func__, "unused parameters");

      sprintf(ret_string, "? %d", RM_ERR_BADPARAM);

      goto done;
      }
    }    /* END for (i) */

  *d = '\0';

  DBPRT(("command: %s\n",
         ret_string))

  if ((child = popen(ret_string, "r")) == NULL)
    {
    log_err(errno, __func__, "popen");

    sprintf(ret_string, "? %d", RM_ERR_SYSTEM);

    goto done;
    }

  fd = fileno(child);

  child_spot = ret_string;
  child_len = 0;
  child_spot[0] = '\0';

retryread:

  while ((len = read(fd, child_spot, ret_size - child_len)) > 0)
    {
    for (i = 0;i < len;i++)
      {
      if (child_spot[i] == '\n')
        break;
      }

    if (i < len)
      {
      /* found newline */

      child_len += i + 1;

      break;
      }

    child_len += len;

    child_spot += len;

    checkret(&child_spot, len);
    }

  if (len == -1)
    {
    if (errno == EINTR)
      {
      goto retryread;
      }

    log_err(errno, __func__, "pipe read");

    sprintf(ret_string, "? %d", RM_ERR_SYSTEM);

    fclose(child);

    goto done;
    }

  pclose(child);

  if (child_len > 0)
    ret_string[child_len - 1] = '\0'; /* hack off newline */

done:

  for (i = 0;name[i] != NULL;i++)
    {
    /* free up params */

    free(name[i]);
    free(value[i]);
    }  /* END for (i) */

  return(ret_string);
  }  /* END conf_res() */




static void catch_abort(

  int sig)

  {

  struct rlimit rlimit;

  /*
   * Reset ourselves to the default signal handler to try and
   * prevent recursive core dumps.
   */

  struct sigaction act;

  sigemptyset(&act.sa_mask);
  act.sa_flags   = 0;
  act.sa_handler = SIG_DFL;

  sigaction(SIGSEGV, &act, NULL);
  sigaction(SIGBUS, &act, NULL);
  sigaction(SIGFPE, &act, NULL);
  sigaction(SIGILL, &act, NULL);
  sigaction(SIGTRAP, &act, NULL);
  sigaction(SIGSYS, &act, NULL);

  log_err(sig, "mom_main", "Caught fatal core signal");
  rlimit.rlim_cur = RLIM_INFINITY;
  rlimit.rlim_max = RLIM_INFINITY;

  setrlimit(RLIMIT_CORE, &rlimit);
  abort();

  return;
  }  /* END catch_abort() */





static void catch_hup(

  int sig)

  {
  sprintf(log_buffer, "caught signal %d",
          sig);

  log_record(PBSEVENT_SYSTEM, 0, "catch_hup", "reset");

  call_hup = 1;

  rpp_dbprt = 1 - rpp_dbprt; /* toggle debug prints for RPP */


  return;
  }  /* END catch_hup() */




/*
 * Do a restart of resmom.
 * Read the last seen config file and
 * Clean up and reinit the dependent code.
 */

static void process_hup(void)

  {
  call_hup = 0;
  log_record(PBSEVENT_SYSTEM, 0, __func__, "reset");

  pthread_mutex_lock(log_mutex);
  log_close(1);
  log_open(log_file, path_log);
  pthread_mutex_unlock(log_mutex);
  log_file_max_size = 0;
  log_file_roll_depth = 1;
#ifdef PENABLE_LINUX26_CPUSETS
  memory_pressure_threshold = 0;
  memory_pressure_duration  = 0;
#endif
  clear_servers();
  read_config(NULL);
  check_log();
  cleanup();

  initialize();

  return;
  }  /* END process_hup() */




/*
** Got an alarm call.
** Close all general network connections, clean up and reinit the
** dependent code.
*/

void toolong(

  int sig)

  {
  log_record(PBSEVENT_SYSTEM, 0, __func__, "alarm call");

  if (LOGLEVEL >= 1)
    DBPRT(("alarm call\n"))

    return;
  }  /* END toolong() */







#ifdef DEBUG

void log_verbose(

  char *id,
  char *buf,
  int  len)

  {
  int i;
  char *cp;

  len = MIN(len, 50);

  cp = log_buffer;

  for (i = 0;i < len;i++)
    {
    int c = buf[i];

    if (isprint(c))
      {
      *cp++ = c;
      }
    else
      {
      sprintf(cp, "(%d)",
              c);

      cp += strlen(cp);
      }
    }

  *cp = '\0';

  log_record(PBSEVENT_DEBUG, 0, id, log_buffer);

  return;
  }  /* END log_verbose() */


#else
#define log_verbose(a, b, c)
#endif

/*
** See if an IP address matches any names stored as "restricted"
** access hosts.  Return 0 if a name matches, 1 if not.
*/

int bad_restrict(

  u_long ipadd)

  {

  struct hostent *host;

  struct in_addr in;
  int i, len1, len2;
  char *cp1, *cp2;

  in.s_addr = htonl(ipadd);

  if ((host = gethostbyaddr(
                (void *) & in,
                sizeof(struct in_addr),
                AF_INET)) == NULL)
    {
    return(1);
    }

  len1 = strlen(host->h_name) - 1;

  for (i = 0;i < mask_num;i++)
    {
    len2 = strlen(maskclient[i]) - 1;

    if (len1 < len2)
      continue;

    cp1 = (char *)&host->h_name[len1];

    cp2 = &maskclient[i][len2];

    /* check case insensitve */

    while ((len2 >= 0) && (tolower(*cp1) == tolower(*cp2)))
      {
      cp1--;
      cp2--;

      len2--;
      }  /* END while () */

    if (((len2 == 0) && (*cp2 == '*')) || (len2 == -1))
      {
      return(0);
      }
    }    /* END for (i) */

  return(1);
  }  /* END bad_restrict() */

/*
 * mom_lock - lock out other MOMs from this directory.
 */

static void mom_lock(
  int fds,
  int op)   /* F_WRLCK or F_UNLCK */
  {
  struct flock flock;

  flock.l_type   = op;
  flock.l_whence = SEEK_SET;
  flock.l_start  = 0;
  flock.l_len    = 0; /* whole file */

  if (fcntl(fds, F_SETLK, &flock) < 0)
    {
    char tmpPath[256];

    tmpPath[0] = '\0';

    if (getcwd(tmpPath, sizeof(tmpPath)) == NULL)
      tmpPath[0] = '\0';

    sprintf(log_buffer, "cannot lock '%s/mom.lock' - another mom running",
            (tmpPath[0] != '\0') ? tmpPath : "$MOM_HOME");

    log_err(errno, msg_daemonname, log_buffer);

    fprintf(stderr, "%s\n",
            log_buffer);

    exit(1);
    }

  return;
  }  /* END mom_lock() */




int could_be_mic_or_gpu_file(

  const char *jobid)

  {
  int         len = strlen(jobid);
  const char *start = jobid + len - 3;

  if (*start == 'g')
    {
    if ((start[1] == 'p') &&
        (start[2] == 'u'))
      return(TRUE);
    }
  else if (*start == 'm')
    {
    if ((start[1] == 'i') &&
        (start[2] == 'c'))
      return(TRUE);
    }

  return(FALSE);
  } /* END could_be_mic_or_gpu_file() */




void cleanup_aux()

  {
  struct dirent *pdirent;
  DIR           *auxdir;
  char           namebuf[MAXLINE];
  unsigned int   len;

  auxdir = opendir(path_aux);

  if (auxdir != NULL)
    {
    while ((pdirent = readdir(auxdir)) != NULL)
      {
      if (pdirent->d_name[0] == '.')
        continue;
        
      if (could_be_mic_or_gpu_file(pdirent->d_name) == TRUE)
        continue;
     
      if (mom_find_job(pdirent->d_name) == NULL)
        {
        /* this job doesn't exist */
        snprintf(namebuf, sizeof(namebuf), "%s/%s", path_aux, pdirent->d_name);
        unlink(namebuf);

        len = strlen(namebuf);
        if (sizeof(namebuf) - 3 > len)
          {
          strcpy(namebuf + len, "gpu");
          unlink(namebuf);
          }
        }
      }
    closedir(auxdir);
    }

  } /* END cleanup_aux() */




/*
** Process a request for the resource monitor.  The i/o
** will take place using DIS over a tcp fd or an rpp stream.
*/

int rm_request(

  struct tcp_chan *chan,
  int              version)

  {
  char                 name[100];
  char                 output[BUFSIZ << 2];
  int                  len;
  int                  command;
  int                  ret;
  int                  restrictrm = 0;
  char                *curr;
  char                *value;
  char                *cp;
  char                *body;
  int                  sindex;

  struct config       *ap;

  struct rm_attribute *attr;

  unsigned long             ipadd;
  u_short                   port;

  extern struct connection  svr_conn[];

  int   NotTrusted = 0;

  char *tmp;
  char *BPtr;
  int   BSpace;
  int   num_queries = 0;
  int   query_index;

  errno = 0;
  log_buffer[0] = '\0';

  ipadd = svr_conn[chan->sock].cn_addr;
  port = svr_conn[chan->sock].cn_port;

  if (version != RM_PROTOCOL_VER)
    {
    sprintf(log_buffer, "protocol version %d unknown",
      version);

    goto bad;
    }

  if (((port_care != FALSE) && (port >= IPPORT_RESERVED)) ||
      (AVL_is_in_tree_no_port_compare(ipadd, 0, okclients) == 0 ))
    {
    if (bad_restrict(ipadd))
      {
      sprintf(log_buffer, "bad attempt to connect - unauthorized (port: %d)",
        port);

      NotTrusted = 1;

      goto bad;
      }

    restrictrm = 1;
    }

  /* looks okay, find out how many queries and what command it is */
  num_queries = disrsi(chan, &ret);

  if (ret == DIS_SUCCESS)
    command = disrsi(chan, &ret);

  if (ret != DIS_SUCCESS)
    {
    sprintf(log_buffer, "no command %s",
            dis_emsg[ret]);

    goto bad;
    }

  switch (command)
    {
    case RM_CMD_CLOSE:  /* no response to this */
      return DIS_EOD;

      /*NOTREACHED*/

      break;

    case RM_CMD_REQUEST:

      /* query resource data */
      reqnum++;

      ret = diswsi(chan, RM_RSP_OK);

      if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "write request response failed: %s",
                dis_emsg[ret]);

        goto bad;
        }

      for (query_index = 0; query_index < num_queries; query_index++)
        {
        cp = disrst(chan, &ret);

        if (ret == DIS_EOD)
          {
          if (cp != NULL)
            {
            free(cp);
            cp = NULL;
            }

          break;
          }

        if (ret != DIS_SUCCESS)
          {
          sprintf(log_buffer, "problem with request line: %s",
                  dis_emsg[ret]);

          goto bad;
          }

        curr = skipwhite(cp);

        curr = tokcpy(curr, name);

        if (name[0] == '\0')
          {
          /* no name */

          sprintf(output, "%s=? %d",
                  cp,
                  RM_ERR_UNKNOWN);
          }
        else
          {
          if (!strncasecmp(name, "clearjob", strlen("clearjob")))
            {
            char *ptr = NULL;

            job *pjob = NULL, *pjobnext = NULL;

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              ptr = curr + 1;
              }

            /* purge job if local */

            if (ptr == NULL)
              {
              strcpy(output, "invalid clearjob request");
              }
            else
              {
              char tmpLine[1024];

              if (!strcasecmp(ptr, "all"))
                {
                if ((pjob = (job *)GET_NEXT(svr_alljobs)) != NULL)
                  {
                  while (pjob != NULL)
                    {
                    sprintf(tmpLine, "clearing job %s",
                            pjob->ji_qs.ji_jobid);

                    log_record(PBSEVENT_SYSTEM, 0, __func__, tmpLine);

                    pjobnext = (job *)GET_NEXT(pjob->ji_alljobs);

                    mom_job_purge(pjob);

                    pjob = pjobnext;

                    strcat(output, tmpLine);
                    strcat(output, "\n");
                    }
                  }

                strcat(output, "clear completed");
                }
              else if ((pjob = mom_find_job(ptr)) != NULL)
                {
                sprintf(tmpLine, "clearing job %s",
                        pjob->ji_qs.ji_jobid);

                log_record(PBSEVENT_SYSTEM, 0, __func__, tmpLine);

                mom_job_purge(pjob);

                strcpy(output, tmpLine);
                }
              }
            }
          else if (!strncasecmp(name, "clearmsg", strlen("clearmsg")))
            {
            /*  clear rm messages */

            PBSNodeMsgBuf[0] = '\0';

            strcpy(output, "messages cleared");

            log_record(PBSEVENT_SYSTEM, 0, __func__, "messages cleared");
            }
          else if (!strncasecmp(name, "cycle", strlen("cycle")))
            {
            /*  force immediate cycle */

            LastServerUpdateTime = 0;

            strcpy(output, "cycle forced");

            log_record(PBSEVENT_SYSTEM, 0, __func__, "reporting cycle forced");
            }
          else if (!strncasecmp(name, "status_update_time", strlen("status_update_time")))
            {
            /* set or report status_update_time */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setstatusupdatetime(curr + 1);
              }

            sprintf(output, "status_update_time=%d",

                    ServerStatUpdateInterval);
            }
          else if (!strncasecmp(name, "check_poll_time", strlen("check_poll_time")))
            {
            /* set or report check_poll_time */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setcheckpolltime(curr + 1);
              }

            sprintf(output, "check_poll_time=%d",
                    CheckPollTime);
            }
          else if (!strncasecmp(name, "jobstartblocktime", strlen("jobstartblocktime")))
            {
            /* set or report jobstartblocktime */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              jobstartblocktime(curr + 1);
              }

            sprintf(output, "jobstartblocktime=%ld",

                    TJobStartBlockTime);
            }
          else if (!strncasecmp(name, "loglevel", strlen("loglevel")))
            {
            /* set or report loglevel */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setloglevel(curr + 1);
              }

            sprintf(output, "loglevel=%d",
                    LOGLEVEL);
            }
          else if (!strncasecmp(name, "down_on_error", strlen("down_on_error")))
            {
            /* set or report down_on_error */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setdownonerror(curr + 1);
              }

            sprintf(output, "down_on_error=%d",

                    MOMConfigDownOnError);
            }
          else if (!strncasecmp(name, "enablemomrestart", strlen("enablemomrestart")))
            {
            /* set or report enablemomrestart */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setenablemomrestart(curr + 1);
              }

            sprintf(output, "enablemomrestart=%d",

                    MOMConfigRestart);
            }
          else if (!strncasecmp(name, "rcpcmd", strlen("rcpcmd")))
            {
            /* set or report rcp_path and rcp_args */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              {
              setrcpcmd(curr + 1);
              }

            sprintf(output, "rcpcmd=%s %s",

                    rcp_path, rcp_args);
            }
#ifdef PENABLE_LINUX26_CPUSETS
          else if (!strncasecmp(name, "memory_pressure_threshold", strlen("memory_pressure_threshold")))
            {
            /* set or report memory_pressure_threshold */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              setmempressthr(curr + 1);

            sprintf(output, "memory_pressure_threshold=%d", memory_pressure_threshold);
            }
          else if (!strncasecmp(name, "memory_pressure_duration", strlen("memory_pressure_duration")))
            {
            /* set or report memory_pressure_duration */

            if ((*curr == '=') && ((*curr) + 1 != '\0'))
              setmempressdur(curr + 1);

            sprintf(output, "memory_pressure_duration=%d", memory_pressure_duration);
            }
#endif
          else if (!strncasecmp(name, "version", strlen("version")))
            {
            /* report version */

            sprintf(output, "version=%s",
                    PACKAGE_VERSION);
            }
          else if ((!strncasecmp(name, "configversion", strlen("configversion"))) && (MOMConfigVersion[0] != '\0'))
            {
            /* report configversion */

            sprintf(output, "configversion=%s",
                    MOMConfigVersion);
            }
          else if (!strncasecmp(name, "diag", strlen("diag")))
            {
            char tmpLine[1024];
            char *ptr;

            int rc;
            time_t Now;

            job *pjob;

            struct varattr *pva;

            time(&Now);

            ptr = name + strlen("diag");

            verbositylevel = (int)strtol(ptr, NULL, 10);

            output[0] = '\0';

            BPtr = output;
            BSpace = sizeof(output);

            sprintf(tmpLine, "\nHost: %s/%s   Version: %s   PID: %ld\n",
                    mom_short_name,
                    mom_host,
                    PACKAGE_VERSION,
                    (long)getpid());

            MUStrNCat(&BPtr, &BSpace, tmpLine);

            mom_server_all_diag(&BPtr, &BSpace);

            sprintf(tmpLine, "HomeDirectory:          %s\n",
                    (mom_home != NULL) ? mom_home : "N/A");

            MUStrNCat(&BPtr, &BSpace, tmpLine);

#ifdef HAVE_SYS_STATVFS_H
              {
#include <sys/statvfs.h>

              struct statvfs VFSStat;

              if (statvfs(path_spool, &VFSStat) < 0)
                {
                MUSNPrintF(&BPtr, &BSpace, "ALERT:  cannot stat stdout/stderr spool directory '%s' (errno=%d) %s\n",
                           path_spool,
                           errno,
                           strerror(errno));
                }
              else
                {
                if (VFSStat.f_bavail > 0)
                  {
                  if (verbositylevel >= 1)
                    MUSNPrintF(&BPtr, &BSpace, "stdout/stderr spool directory: '%s' (%d blocks available)\n",
                               path_spool,
                               VFSStat.f_bavail);
                  }
                else
                  {
                  MUSNPrintF(&BPtr, &BSpace, "ALERT:  stdout/stderr spool directory '%s' is full.\nFree space then restart pbs_mom.\n",
                             path_spool);
                  }
                }
              }    /* END BLOCK */
#endif /* HAVE_SYS_STATVFS_H */

            if (MOMConfigVersion[0] != '\0')
              {
              sprintf(tmpLine, "ConfigVersion:          %s\n",
                      MOMConfigVersion);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (verbositylevel >= 3)
              {
#if SYSLOG
              MUStrNCat(&BPtr, &BSpace, "NOTE:  syslog enabled\n");
#else /* SYSLOG */
              MUStrNCat(&BPtr, &BSpace, "NOTE:  syslog not enabled (use 'configure --enable-syslog' to enable)\n");
#endif /* SYSLOG */
              }

            if (verbositylevel >= 3)
              {
              if (PBSNodeCheckPath[0] != '\0')
                {
                sprintf(tmpLine, "Node Health Check Script: %s (%d second update interval)\n",
                        PBSNodeCheckPath,
                        PBSNodeCheckInterval * ServerStatUpdateInterval);

                MUStrNCat(&BPtr, &BSpace, tmpLine);
                }
              }

            sprintf(tmpLine, "MOM active:             %ld seconds\n",
                    (long)Now - MOMStartTime);

            MUStrNCat(&BPtr, &BSpace, tmpLine);

            if (verbositylevel >= 1)
              {
              sprintf(tmpLine, "Check Poll Time:        %d seconds\n",
                      CheckPollTime);

              MUStrNCat(&BPtr, &BSpace, tmpLine);

              sprintf(tmpLine, "Server Update Interval: %d seconds\n",
                      ServerStatUpdateInterval);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (PBSNodeMsgBuf[0] != '\0')
              {
              sprintf(tmpLine, "MOM Message:            %s (use 'momctl -q clearmsg' to clear)\n",
                      PBSNodeMsgBuf);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (MOMUNameMissing[0] != '\0')
              {
              sprintf(tmpLine, "WARNING:  passwd file is corrupt (job requests user '%s' - not found in local passwd file)\n",
                      MOMUNameMissing);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (MOMPrologTimeoutCount > 0)
              {
              sprintf(tmpLine, "WARNING:  %d prolog timeouts (%d seconds) detected since start up - increase $prologalarm or investigate prolog\n",
                      MOMPrologTimeoutCount,
                      pe_alarm_time);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (MOMPrologFailureCount > 0)
              {
              sprintf(tmpLine, "WARNING:  %d prolog failures detected since start up - investigate prolog\n",
                      MOMPrologFailureCount);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            sprintf(tmpLine, "LogLevel:               %d (use SIGUSR1/SIGUSR2 to adjust)\n",

                    LOGLEVEL);

            MUStrNCat(&BPtr, &BSpace, tmpLine);

            if (verbositylevel >= 1)
              {
              sprintf(tmpLine, "Communication Model:    %s\n", "TCP");

              MUStrNCat(&BPtr, &BSpace, tmpLine);

              if ((MOMIsLocked == 1) || (MOMIsPLocked == 1) || (verbositylevel >= 4))
                {
                sprintf(tmpLine, "MemLocked:              %s",
                        (MOMIsLocked == 0) ? "FALSE" : "TRUE");

                if (MOMIsLocked == 1)
                  strcat(tmpLine, "  (mlock)");

                if (MOMIsPLocked == 1)
                  strcat(tmpLine, "  (plocked)");

                strcat(tmpLine, "\n");

                MUStrNCat(&BPtr, &BSpace, tmpLine);
                }
              }    /* END if (verbositylevel >= 1) */

            if (verbositylevel >= 1)
              {
              sprintf(tmpLine, "TCP Timeout:            %d seconds\n",
                      (int)pbs_tcp_timeout);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (verbositylevel >= 1)
              {

              struct stat s;

              int prologfound = 0;

              if (stat(path_prolog, &s) != -1)
                {
                MUSNPrintF(&BPtr, &BSpace, "Prolog:                 %s (enabled)\n",
                           path_prolog);

                prologfound = 1;
                }
              else if (verbositylevel >= 2)
                {
                MUSNPrintF(&BPtr, &BSpace, "Prolog:                 %s (disabled)\n",
                           path_prolog);
                }

              if (stat(path_prologp, &s) != -1)
                {
                MUSNPrintF(&BPtr, &BSpace, "Parallel Prolog:        %s (enabled)\n",
                           path_prologp);

                prologfound = 1;
                }

              if (prologfound == 1)
                {
                sprintf(tmpLine, "Prolog Alarm Time:      %d seconds\n",
                        pe_alarm_time);

                MUStrNCat(&BPtr, &BSpace, tmpLine);
                }
              }

            if (verbositylevel >= 2)
              {
              /* check alarm */

              rc = alarm(alarm_time);

              alarm(rc);

              sprintf(tmpLine, "Alarm Time:             %d of %d seconds\n",
                      rc,
                      alarm_time);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }

            if (verbositylevel >= 1)
              {
              /* display okclient list */

              long max_len = 1024;
              long final_len = 0;
              char *tmp_line = calloc(1, max_len + 1);
              if (tmp_line != NULL)
                {
                ret = AVL_list(okclients, &tmp_line, &final_len, &max_len);

                MUSNPrintF(&BPtr, &BSpace, "Trusted Client List:  %s:  %d\n",
                    tmp_line, ret);
                free(tmp_line);
                }
              else
                {
                MUSNPrintF(&BPtr, &BSpace,
                    "Trusted Client Could not be retrieved\n");
                }
              }

            if (verbositylevel >= 1)
              {
              tmpLine[0] = '\0';

              MUSNPrintF(&BPtr, &BSpace, "Copy Command:           %s %s\n",
                         rcp_path,
                         rcp_args);
              }

            /* joblist */

            if ((pjob = (job *)GET_NEXT(svr_alljobs)) == NULL)
              {
              sprintf(tmpLine, "NOTE:  no local jobs detected\n");

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }
            else
              {
              int            numvnodes = 0;
              resource      *pres;
              char          *resname;
              unsigned long  resvalue;
              char           SIDList[1024];
              task          *ptask;
              char          *VPtr;  /* job env variable value pointer */
              char          *SPtr;
              int            SSpace;
              int            vnindex;

              for (;pjob != NULL;pjob = (job *)GET_NEXT(pjob->ji_alljobs))
                {

                /* count the CPUs assigned to this mom */
                for (vnindex = 0; vnindex < pjob->ji_numvnod; vnindex++)
                  {
                  if (!strcmp(pjob->ji_vnods[vnindex].vn_host->hn_host, mom_alias))
                    {
                    numvnodes++;
                    }
                  }

                /* JobId, job state */
                sprintf(tmpLine, "job[%s]  state=%s",
                        pjob->ji_qs.ji_jobid,
                        PJobSubState[pjob->ji_qs.ji_substate]);
                MUStrNCat(&BPtr, &BSpace, tmpLine);

                /* Selected resources used by job (on this node) */
                if (verbositylevel >= 2)
                  {
                  if (pjob->ji_wattr[JOB_ATR_resc_used].at_type != ATR_TYPE_RESC)
                    {
                    free(cp);
                    return(DIS_INVALID);
                    }

                  pres = (resource *)GET_NEXT(pjob->ji_wattr[JOB_ATR_resc_used].at_val.at_list);
                  for (;pres != NULL;pres = (resource *)GET_NEXT(pres->rs_link))
                    {
                    if (pres->rs_defin == NULL) 
                      {
                      free(cp);
                      return(DIS_INVALID);
                      }

                    resname = pres->rs_defin->rs_name;
                    if (!(strcmp(resname, "mem")))
                      resvalue = getsize(pres);
                    else if (!(strcmp(resname, "vmem")))
                      resvalue = getsize(pres);
                    else if (!(strcmp(resname, "cput")))
                      resvalue = gettime(pres);
                    else
                      continue;

                    sprintf(tmpLine, " %s=%lu", resname, resvalue);
                    MUStrNCat(&BPtr, &BSpace, tmpLine);
                    }

#ifdef PENABLE_LINUX26_CPUSETS
                  /* report current memory pressure */
                  sprintf(tmpLine, " %s=%d", "mempressure", pjob->ji_mempressure_curr);
                  MUStrNCat(&BPtr, &BSpace, tmpLine);
#endif
                  }

                /* List of job's sessionIds */
                SPtr       = SIDList;
                SSpace     = sizeof(SIDList);
                SIDList[0] = '\0';

                for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
                     ptask != NULL;
                     ptask = (task *)GET_NEXT(ptask->ti_jobtask))
                  {
                  /* only check on tasks that we think should still be around */

                  if (ptask->ti_qs.ti_status != TI_STATE_RUNNING)
                    continue;

                  /* NOTE:  on linux systems, the session master should have
                     pid == sessionid */

                  MUSNPrintF(&SPtr, &SSpace, "%s%d",
                             (SIDList[0] != '\0') ? "," : "",
                             ptask->ti_qs.ti_sid);
                  }  /* END for (task) */

                sprintf(tmpLine, " sidlist=%s",
                        SIDList);
                MUStrNCat(&BPtr, &BSpace, tmpLine);

                if (verbositylevel >= 4)
                  {
                  /* report job variables */

                  VPtr = get_job_envvar(pjob, "BATCH_PARTITION_ID");

                  if (VPtr != NULL)
                    {
                    sprintf(tmpLine, "  BATCH_PARTITION_ID=%s",
                            VPtr);

                    MUStrNCat(&BPtr, &BSpace, tmpLine);
                    }

                  VPtr = get_job_envvar(pjob, "BATCH_ALLOC_COOKIE");

                  if (VPtr != NULL)
                    {
                    sprintf(tmpLine, "  BATCH_ALLOC_COOKIE=%s",
                            VPtr);

                    MUStrNCat(&BPtr, &BSpace, tmpLine);
                    }
                  }    /* END if (verbositylevel >= 4) */

                MUStrNCat(&BPtr, &BSpace, "\n");
                }  /* END for (pjob) */

              sprintf(tmpLine, "Assigned CPU Count:     %d\n",
                      numvnodes);

              MUStrNCat(&BPtr, &BSpace, tmpLine);
              }  /* END else ((pjob = (job *)GET_NEXT(svr_alljobs)) == NULL) */

            if ((pjob = (job *)GET_NEXT(svr_newjobs)) != NULL)
              {
              while (pjob != NULL)
                {
                sprintf(tmpLine, "job[%s]  state=NEW\n",
                        pjob->ji_qs.ji_jobid);

                MUStrNCat(&BPtr, &BSpace, tmpLine);

                pjob = (job *)GET_NEXT(pjob->ji_alljobs);
                }
              }

            if ((pva = (struct varattr *)GET_NEXT(mom_varattrs)) != NULL)
              {
              MUStrNCat(&BPtr, &BSpace, "Varattrs:\n");

              while (pva != NULL)
                {
                sprintf(tmpLine, "  ttl=%d  last=%s  cmd=%s\n  value=%s\n\n",
                        pva->va_ttl,
                        ctime(&pva->va_lasttime),
                        pva->va_cmd,
                        (pva->va_value != NULL) ? pva->va_value : "NULL");

                MUStrNCat(&BPtr, &BSpace, tmpLine);

                pva = (struct varattr *)GET_NEXT(pva->va_link);
                }
              }

            MUStrNCat(&BPtr, &BSpace, "\ndiagnostics complete\n");

            log_record(PBSEVENT_SYSTEM, 0, __func__, "internal diagnostics complete");
            }
          else
            {
            ap = rm_search(config_array, name);

            attr = momgetattr(curr);

            if (LOGLEVEL >= 3)
              log_record(PBSEVENT_SYSTEM, 0, __func__, "setting alarm in rm_request");

            alarm(alarm_time);

            if ((ap != NULL) && !restrictrm)
              {
              /* static */

              sprintf(output, "%s=%s",
                      cp,
                      conf_res(ap->c_u.c_value, attr));
              }
            else
              {
              /* check dependent code */

              log_buffer[0] = '\0';

              value = dependent(name, attr);

              if (value != NULL)
                {
                sprintf(output, "%s=%s",
                        cp,
                        value);
                }
              else
                {
                /* not found anywhere */

                sprintf(output, "%s=? %d",
                        cp,
                        rm_errno);
                }
              }

            alarm(0);
            }
          }  /* END (name[0] == '\0') */

        free(cp);

        ret = diswst(chan, output);


        if (ret != DIS_SUCCESS)
          {
          sprintf(log_buffer, "write string failed %s",
                  dis_emsg[ret]);

          goto bad;
          }

        if (DIS_tcp_wflush(chan) == -1)
          {
          log_err(errno, __func__, "flush");

          goto bad;
          }

        }    /* END for () */

      break;

    case RM_CMD_CONFIG:

      {
      char *ptr;

      if (MOMConfigRReconfig == FALSE)
        {
        log_err(-1, __func__,
          "remote reconfiguration disabled, ignoring request");

        goto bad;
        }

      if (restrictrm)
        {
        log_err(-1, __func__, "restricted configure attempt");

        goto bad;
        }

      log_record(PBSEVENT_SYSTEM, 0, __func__, "configure");

      body = disrst(chan, &ret);

      /* FORMAT:  FILE:<FILENAME> or <FILEDATA> (NYI) */

      if (ret == DIS_EOD)
        {
        /* no file specified, use default */

        body = NULL;
        }
      else if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "problem with config body %s",
                dis_emsg[ret]);

        goto bad;
        }
      else
        {
        FILE *fp;

        if ((ptr = strstr(body, "CONFIG:")) != NULL)
          {
          ptr += strlen("CONFIG:");

          /* overwrite config with data and clear body */

          if ((fp = fopen(config_file, "w+")) == NULL)
            {
            sprintf(log_buffer, "cannot open config file %s",
                    config_file);

            goto bad;
            }

          if (fwrite(ptr, sizeof(char), strlen(ptr) + 1, fp) < (strlen(ptr) + 1))
            {
            fclose(fp);

            sprintf(log_buffer, "cannot write config file %s",
                    config_file);

            goto bad;
            }

          fclose(fp);

          free(body);
          body = NULL;
          }
        }

      clear_servers();

      len = read_config(body);

      free(body);

      ret = diswsi(chan, len ? RM_RSP_ERROR : RM_RSP_OK);

      if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "write config response failed %s",
                dis_emsg[ret]);

        goto bad;
        }

      if (DIS_tcp_wflush(chan) == -1)
        {
        log_err(errno, __func__, "flush");

        goto bad;
        }

      }    /* END (case RM_CMD_CONFIG) */

    break;

    case RM_CMD_SHUTDOWN:

      if (restrictrm)
        {
        log_err(-1, __func__, "restricted shutdown attempt");

        goto bad;
        }

      log_record(PBSEVENT_SYSTEM, 0, __func__, "shutdown");

      ret = diswsi(chan, RM_RSP_OK);

      if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "write shutdown response failed %s",
                dis_emsg[ret]);

        log_err(-1, __func__, log_buffer);
        }

      DIS_tcp_wflush(chan);
 
      close_conn(chan->sock, FALSE);
      DIS_tcp_cleanup(chan);
      
      mom_lock(lockfds, F_UNLCK);
      close(lockfds);

      mom_run_state = MOM_RUN_STATE_EXIT;
      internal_state = INUSE_DOWN;

      for (sindex = 0; sindex < PBS_MAXSERVER; sindex++)
        {
        if (mom_servers[sindex].pbs_servername[0] != '\0')
          shutdown_to_server(sindex);
        }

      cleanup();

#if defined(NVIDIA_GPUS) && defined(NVML_API)
      shut_nvidia_nvml();
#endif  /* NVIDIA_GPUS and NVML_API */

      log_close(1);

      exit(0);

      /*NOTREACHED*/

      break;

    default:

      sprintf(log_buffer, "unknown command %d",
              command);

      log_err(-1, __func__, log_buffer);

      ret = diswsi(chan, RM_RSP_ERROR);

      if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "write default response failed %s",
                dis_emsg[ret]);

        goto bad;
        }

      ret = diswst(chan, log_buffer);

      if (ret != DIS_SUCCESS)
        {
        sprintf(log_buffer, "write string failed %s",
                dis_emsg[ret]);

        goto bad;
        }

      break;
    }  /* END switch(command) */


  return PBSE_NONE;

bad:
  tmp = netaddr_pbs_net_t(ipadd);

  sprintf(output, "\n\tmessage refused from port %d addr %s",
          port,
          tmp);

  sprintf(TMOMRejectConn, "%s:%d  %s",
          tmp,
          port,
          (NotTrusted == 1) ? "(server not authorized)" : "(request corrupt)");

  free(tmp);

  snprintf(log_buffer + strlen(log_buffer), sizeof(log_buffer) - strlen(log_buffer), "%s", output);

  log_err(errno, __func__, log_buffer);

  return DIS_EOD;
  }  /* END rm_request() */



int tcp_read_proto_version(

  struct tcp_chan *chan,
  int             *proto,
  int             *version)

  {
  int rc = DIS_SUCCESS;
  time_t tmpT;

  tmpT = pbs_tcp_timeout;

  pbs_tcp_timeout = 0;

  *proto = disrsi(chan, &rc);

  if (tmpT > 0)
    {
    /* restore */
    pbs_tcp_timeout = tmpT;
    }
  else
    {
    /* initialize */
    pbs_tcp_timeout = PMOMTCPTIMEOUT;
    }

  switch (rc)
    {
    case DIS_SUCCESS:  /* worked */
      break;

    case DIS_EOF:   /* closed */
    case DIS_EOD:   /* still open */
      break;

    default:
      sprintf(log_buffer, "no protocol number: %s (errno = %d)",
          dis_emsg[rc], errno);
      log_err(rc, __func__, log_buffer);
      break;
    }  /* END switch (rc) */

  if (rc == PBSE_NONE)
    {
    *version = disrsi(chan, &rc);
    if (rc != DIS_SUCCESS)
      {
      sprintf(log_buffer, "End of messages on socket %d", chan->sock);
      log_record(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST, __func__, log_buffer);
      }
    }

  return rc;

  }

int do_tcp(
    
  int socket)

  {
  int                       rc = PBSE_NONE;
  int                       proto = -1;
  int                       version = -1;
  struct tcp_chan          *chan = NULL;
  extern struct connection  svr_conn[];

  if ((chan = DIS_tcp_setup(socket)) == NULL)
    {
    sprintf(log_buffer, "Can not allocate memory for socket buffer");
    log_err(errno, __func__, log_buffer);
    return(PBSE_MEM_MALLOC);
    }

  if ((rc = tcp_read_proto_version(chan, &proto, &version)) != DIS_SUCCESS)
    {
    DIS_tcp_cleanup(chan);
    chan = NULL;
    goto do_tcp_cleanup;
    }

  switch (proto)
    {
    case RM_PROTOCOL:

      {
      time_t tmpT;

      DBPRT(("%s: got a resource monitor request\n", __func__))

      tmpT = pbs_tcp_timeout;

      rc = rm_request(chan, version);

      if (tmpT > 0)
        {
        /* restore */

        pbs_tcp_timeout = tmpT;
        }
      else
        {
        /* initialize */

        pbs_tcp_timeout = PMOMTCPTIMEOUT;
        }
      }    /* END BLOCK (case RM_PROTOCOL) */

    break;

    case TM_PROTOCOL:

      DBPRT(("%s: got an internal task manager request\n", __func__))
      svr_conn[chan->sock].cn_stay_open = TRUE;

      rc = tm_request(chan, version);

      while ((rc == PBSE_NONE) &&
             (tcp_chan_has_data(chan) == TRUE))
        {
        if ((rc = tcp_read_proto_version(chan, &proto, &version)) == DIS_SUCCESS)
          rc = tm_request(chan, version);
        }

      break;

    case IS_PROTOCOL:

      mom_is_request(chan,version,NULL);

      break;

    case IM_PROTOCOL:

      im_request(chan,version);

      break;

    default:

      {
      struct sockaddr_in *addr = NULL;
      struct sockaddr     s_addr;
      unsigned int        len = sizeof(s_addr);
      
      if (getpeername(chan->sock, &s_addr, &len) == 0)
        {
        addr = (struct sockaddr_in *)&s_addr;
        DBPRT(("%s: unknown request %d from %s",
          __func__, proto, netaddr(addr)))
        }
      else
        {
        DBPRT(("%s: unknown request %d\n", __func__, proto))
        }

      svr_conn[chan->sock].cn_stay_open = FALSE;
      }
    
    goto do_tcp_cleanup;

    break;
    }  /* END switch (proto) */

  /* don't close these connections -- the pointer is saved in 
   * the tasks for MPI jobs */
  if (svr_conn[chan->sock].cn_stay_open == FALSE)
    DIS_tcp_cleanup(chan);

  return rc;

do_tcp_cleanup:
  
  if ((chan != NULL) &&
      (svr_conn[chan->sock].cn_stay_open == FALSE))
    DIS_tcp_cleanup(chan);

  return DIS_INVALID;
  }  /* END do_tcp() */





void *tcp_request(

  void *new_sock)

  {
  int  c;
  long  ipadd;
  char  address[80];
  char *tmp;
  int rc = PBSE_NONE;
  int avail_bytes = -1;
  int socket = *(int *)new_sock;

  extern struct connection svr_conn[];
  if ((avail_bytes = socket_avail_bytes_on_descriptor(socket)) == 0)
    {
    close_conn(socket, FALSE);
    return NULL;
    }

  ipadd = svr_conn[socket].cn_addr;

  tmp = netaddr_pbs_net_t(ipadd);

  sprintf(address, "%s:%d",
          tmp,
          svr_conn[socket].cn_port);

  free(tmp);

  if (LOGLEVEL >= 6)
    {
    sprintf(log_buffer, "%s: fd %d addr %s", __func__, socket, address);

    log_record(PBSEVENT_JOB,PBS_EVENTCLASS_JOB,"tcp_request",log_buffer);
    }

  if (AVL_is_in_tree_no_port_compare(ipadd, 0, okclients) == 0)
    {
    sprintf(log_buffer, "bad connect from %s", address);
    log_err(errno, __func__, log_buffer);
    close_conn(socket, FALSE);
    return NULL;
    }

  log_buffer[0] = '\0';

  for (c = 0;;c++)
    {
    rc = do_tcp(socket);
    switch (rc)
      {
      case PBSE_NONE:
        continue;
        break;

      case DIS_EOF:
        DBPRT(("Closing socket %d twice...\n", socket))
      case PBSE_MEM_MALLOC:
      case DIS_EOD:
      case DIS_INVALID:
        if (svr_conn[socket].cn_stay_open == FALSE)
          close_conn(socket, FALSE);
        break;

      default:
        close_conn(socket, FALSE);
        DBPRT(("Error in connection. Closing %d\n", socket))
        break;
      }

      break;
    }  /* END for (c = 0) */

  DBPRT(("%s:(exit loop for socket %d) processed %d\n", __func__, socket, c))

  return NULL;
  }  /* END tcp_request() */



char *find_signal_name(

  int sig)

  {

  struct sig_tbl *psigt;

  extern struct sig_tbl sig_tbl[];

  for (psigt = sig_tbl; psigt->sig_name != NULL; psigt++)
    {
    if (psigt->sig_val == sig)
      {
      return(psigt->sig_name);
      }
    }

  return("unknown signal");
  }




/*
 *  Kill a job.
 * Call with the job pointer and a signal number.
 *
 * NOTE:  sends a signal to a job, does not purge job record
 *
 * @see kill_task() - child
 * @see scan_for_exiting() - parent
 */

int kill_job(

  job        *pjob,              /* I */
  int         sig,               /* I */
  const char *killer_id_name,    /* I - process name of calling routine */
  char       *why_killed_reason) /* I - reason for killing */

  {
  task *ptask;
  int   ct = 0;

  sprintf(log_buffer, "%s: sending signal %d, \"%s\" to job %s, reason: %s",
          killer_id_name,
          sig, find_signal_name(sig),
          pjob->ji_qs.ji_jobid,
          why_killed_reason);

  if (LOGLEVEL >= 2)
    {
    log_record(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, __func__, log_buffer);
    }

  DBPRT(("%s\n", log_buffer));
  DBPRT(("Job - %s Current State %s\n", pjob->ji_qs.ji_jobid, PJobSubState[MAX(0,pjob->ji_qs.ji_substate)]));

  /* NOTE:  should change be made to only execute precancel epilog if
   * job is active? (NYI) */

  /* NOTE:  epilog blocks until complete, which may cause issues if
   * shutdown grace time is enabled.  Change model to allow
   * epilog.precancel to run in background and have kill_task()
   * executed once it is complete (NYI) */

  /* NOTE:  this will allow kill_job to return immediately and will
   * require sigchild harvesting and the kill_task loop to be called
   * once this signal is received */

  /* NOTE:  if path_epilogpdel is not set, kill_task should be called
   * immediately (NYI) */

  if(sig == SIGTERM)
    {
    if (run_pelog(PE_EPILOGUSER, path_epilogpdel, pjob, PE_IO_TYPE_NULL) != 0)
      {
      log_err(-1, __func__, "precancel epilog failed");

      sprintf(PBSNodeMsgBuf, "ERROR:  precancel epilog failed");
      }
    }

  ptask = (task *)GET_NEXT(pjob->ji_tasks);

  while (ptask != NULL)
    {
    if (ptask->ti_qs.ti_status == TI_STATE_RUNNING)
      {
      if (LOGLEVEL >= 4)
        {
        log_record(
          PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          "kill_job found a task to kill");
        }

      ct += kill_task(ptask, sig, 0);
      }

    ptask = (task *)GET_NEXT(ptask->ti_jobtask);
    }  /* END while (ptask != NULL) */

  if (LOGLEVEL >= 6)
    {
    sprintf(log_buffer, "kill_job done (killed %d processes)",
            ct);

    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      pjob->ji_qs.ji_jobid,
      log_buffer);
    }

  return(ct);
  }  /* END kill_job() */


/*
 * size decoding routine.
 *
 * Accepts a resource pointer and a pointer to the unsigned long integer
 * to receive the decoded value.  It returns the decoded value in kb.
 *
 *  sizeof(word) = sizeof(int)
 */

unsigned long getsize(

  resource *pres)  /* I */

  {
  unsigned long value;
  unsigned long shift;

  if (pres->rs_value.at_type != ATR_TYPE_SIZE)
    {
    return(0);
    }

  value = pres->rs_value.at_val.at_size.atsv_num;

  shift = pres->rs_value.at_val.at_size.atsv_shift;

  if (pres->rs_value.at_val.at_size.atsv_units == ATR_SV_WORDSZ)
    {
    if (value > ULONG_MAX / sizeof(int))
      {
      return(0);
      }

    value *= sizeof(int);
    }

  if (shift > 10)
    {
    shift -= 10;

    return(value << shift);
    }

  shift = 10 - shift;

  return(value >> shift);
  }





/*
 * time decoding routine.
 *
 * Accepts a resource pointer and a pointer to the unsigned long integer
 * to receive the decoded value.  It returns the decoded value of time
 * in seconds.
 */

unsigned long gettime(

  resource *pres)

  {
  if (pres->rs_value.at_type != ATR_TYPE_LONG)
    {
    return(0);
    }

  if (pres->rs_value.at_val.at_long < 0)
    {
    return(0);
    }

  return((unsigned long)pres->rs_value.at_val.at_long);
  }  /* END getttime() */




/* log_buffer reports detailed failure reason */

/* return 0:  no issues detected */
/* return 1:  over limit/child termination request detected */

int job_over_limit(

  job *pjob)  /* I */

  {
  pbs_attribute       *attr;
  pbs_attribute       *used;
  resource            *limresc;
  resource            *useresc;

  struct resource_def *rd;
  unsigned long        total;
  int                  index;
  unsigned long        limit;
  char                *units;

#ifndef NUMA_SUPPORT
  int                  i;
#endif /* ndef NUMA_SUPPORT */

  if (mom_over_limit(pjob))
    {
    /* mom limits violated, log_buffer populated */

    /* no more POLL's */

    pjob->ji_nodekill = pjob->ji_nodeid;
    pjob->ji_qs.ji_un.ji_momt.ji_exitstat = JOB_EXEC_OVERLIMIT;

    return(1);
    }

#ifndef NUMA_SUPPORT 
  /* cannot perform this check with NUMA, numnodes always is 1 and
   * you'll never have job stats */
  if ((pjob->ji_numnodes == 1) ||
      ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0))
    {
    /* no other nodes or not mother superior */

    /* SUCCESS */

    return(0);
    }
#endif

  if (pjob->ji_nodekill != TM_ERROR_NODE)
    {
    /* one of the sister nodes reports a fatal error */

    hnodent *pnode = &pjob->ji_hosts[pjob->ji_nodekill];

    if (pnode->hn_sister != 0)
      {
      switch (pnode->hn_sister)
        {

        case SISTER_KILLDONE:

          sprintf(log_buffer, "node %d (%s) requested job terminate, '%s' (%d)",
                  pjob->ji_nodekill,
                  pnode->hn_host,
                  "killdone",
                  pnode->hn_sister);

          break;

        case SISTER_BADPOLL:

          sprintf(log_buffer, "node %d (%s) requested job terminate, '%s' (code %d)",
                  pjob->ji_nodekill,
                  pnode->hn_host,
                  "badpoll",
                  pnode->hn_sister);

          break;

        case SISTER_EOF:

          sprintf(log_buffer, "node %d (%s) requested job terminate, '%s' (code %d) - received SISTER_EOF attempting to communicate with sister MOM's",
                  pjob->ji_nodekill,
                  pnode->hn_host,
                  "EOF",
                  pnode->hn_sister);

          break;

        default:

          sprintf(log_buffer, "node %d (%s) requested job terminate, '%s' (code %d) - internal or network failure attempting to communicate with sister MOM's",
                  pjob->ji_nodekill,
                  pnode->hn_host,
                  "EOF",
                  pnode->hn_sister);

          break;
        }  /* END switch (pnode->hn_sister) */

      /* FAILURE */

      return(1);
      }  /* END if (pnode->hn_sister != 0) */
    }    /* END if (pjob->ji_nodekill != TM_ERROR_NODE) */

  attr = &pjob->ji_wattr[JOB_ATR_resource];

  used = &pjob->ji_wattr[JOB_ATR_resc_used];

  /* only enforce cpu time and memory usage */

  for (limresc = (resource *)GET_NEXT(attr->at_val.at_list);
       limresc != NULL;
       limresc = (resource *)GET_NEXT(limresc->rs_link))
    {
    if ((limresc->rs_value.at_flags & ATR_VFLAG_SET) == 0)
      continue;

    rd = limresc->rs_defin;

    if (!strcmp(rd->rs_name, "cput"))
      {
      if (igncput == TRUE)
        continue;
      else
        index = 0;
      }
    else if (!strcmp(rd->rs_name, "mem"))
      {
      if (ignmem == TRUE)
        continue;
      else
        index = 1;
      }
    else
      continue;

    useresc = find_resc_entry(used, rd);

    if (useresc == NULL)
      continue;

    if ((useresc->rs_value.at_flags & ATR_VFLAG_SET) == 0)
      continue;

    total = (index == 0) ? gettime(useresc) : getsize(useresc);

#ifndef NUMA_SUPPORT 
    for (i = 0;i < pjob->ji_numnodes - 1;i++)
      {
      noderes *nr = &pjob->ji_resources[i];

      total += ((index == 0) ? nr->nr_cput : nr->nr_mem);
      }
#endif /* ndef NUMA_SUPPORT */

    limit = (index == 0) ? gettime(limresc) : getsize(limresc);

    if (limit <= total)
      break;
    }  /* END for (limresc) */

  if (limresc == NULL)
    {
    /* no limit violation detected, job ok */

    return(0);
    }

  units = index == 0 ? "secs" : "kb";

  sprintf(log_buffer, "%s job total %lu %s exceeded limit %lu %s",
          rd->rs_name,
          total,
          units,
          limit,
          units);

  pjob->ji_nodekill = pjob->ji_nodeid;

  return(1);
  }  /* END job_over_limit() */




void usage(

  char *prog)  /* I */

  {
  fprintf(stderr, "Usage: %s\n",
          prog);

  fprintf(stderr, "  -a <INT>  \\\\ Alarm Time\n");
  fprintf(stderr, "  -c <PATH> \\\\ Config File\n");
  fprintf(stderr, "  -C <PATH> \\\\ Checkpoint Dir\n");
  fprintf(stderr, "  -d <PATH> \\\\ Home Dir\n");
  fprintf(stderr, "  -C <PATH> \\\\ Checkpoint Dir\n");
  fprintf(stderr, "  -D        \\\\ DEBUG - do not background\n");
  fprintf(stderr, "  -h        \\\\ Print Usage\n");
  fprintf(stderr, "  -H <HOST> \\\\ Hostname\n");
  fprintf(stderr, "  -l        \\\\ MOM Log Dir Path\n");
  fprintf(stderr, "  -L <PATH> \\\\ Logfile\n");
  fprintf(stderr, "  -M <INT>  \\\\ MOM Port\n");
  fprintf(stderr, "  -p        \\\\ Recover Jobs (Default)\n");
  fprintf(stderr, "  -P        \\\\ Purge Jobs\n");
  fprintf(stderr, "  -q        \\\\ Do Not Recover Jobs\n");
  fprintf(stderr, "  -r        \\\\ Recover Jobs (2)\n");
  fprintf(stderr, "  -R <INT>  \\\\ RM Port\n");
  fprintf(stderr, "  -s        \\\\ Logfile Suffix\n");
  fprintf(stderr, "  -S <INT>  \\\\ Server Port\n");
  fprintf(stderr, "  -v        \\\\ Version\n");
  fprintf(stderr, "  -x        \\\\ Do Not Use Privileged Ports\n");
  fprintf(stderr, "  --about   \\\\ Print Build Information\n");
  fprintf(stderr, "  --help    \\\\ Print Usage\n");
  fprintf(stderr, "  --version \\\\ Version\n");

  }  /* END usage() */




/*
 * MOMFindMyExe - attempt to find my running executable file.
 *                returns alloc'd memory that is never freed.
 */

static char *orig_path;

char *MOMFindMyExe(

  char *argv0)  /* I */

  {
  char *link;
  int  has_slash = 0;
  char *p;
  char *p_next;
  char *path;


  link = calloc(MAXPATHLEN + 1, sizeof(char));

  if (link == NULL)
    {
    /* FAILURE */

    return(NULL);
    }

  /* Linux has a handy symlink, so try that first */

  if (readlink("/proc/self/exe", link, MAXPATHLEN) > 0)
    {
    if (link[0] != '\0' && link[0] != '[')
      {
      return(link);
      }
    }

  /* if argv0 has a /, then it should exist relative to $PWD */

  for (p = argv0; *p; p++)
    {
    if (*p == '/')
      {
      has_slash = 1;

      break;
      }
    }

  if (has_slash)
    {
    char resolvedpath[MAXPATHLEN+1];

    if (argv0[0] == '/')
      {
      strcpy(link, argv0);
      }
    else
      {
      if (getcwd(link, MAXPATHLEN) == NULL)
        {
        free(link);

        return(NULL);
        }

      strcat(link, "/");

      strcat(link, argv0);
      }

    if (realpath(link, resolvedpath) == NULL)
      {
      free(link);

      return(NULL);
      }

    strcpy(link, resolvedpath);

    if (access(link, X_OK) == 0)
      {
      return(link);
      }

    free(link);

    return(NULL);
    }

  /* argv0 doesn't have a /, so search $PATH */

  path = getenv("PATH");

  if (path != NULL)
    {
    for (p = path; *p; p = p_next)
      {
      char *q;
      size_t p_len;

      for (q = p;*q;q++)
        {
        if (*q == ':')
          break;
        }

      p_len = q - p;

      p_next = (*q == '\0' ? q : q + 1);

      /* We have a path item at p, of length p_len.
         Now concatenate the path item and argv0.  */

      if (p_len == 0)
        {
        /* An empty PATH element designates the current directory.  */

        if (getcwd(link, MAXPATHLEN) == NULL)
          {
          free(link);

          return(NULL);
          }

        strcat(link, "/");

        strcat(link, argv0);
        }
      else
        {
        snprintf(link, MAXPATHLEN + 1, "%s", p);
        *(link + p_len) = '\0';
        strcat(link, "/");
        strcat(link, argv0);
        }

      if (access(link, X_OK) == 0)
        {
        return(link);
        }
      }  /* END for (p = path; *p; p = p_next) */
    }

  free(link);

  return(NULL);
  }  /* END MOMFindMyExe() */




/*
 * MOMGetFileMtime - return the mtime of a file
 */

time_t MOMGetFileMtime(

  const char *fpath)

  {

  struct stat sbuf;
  int ret;

  if ((fpath == NULL) || (*fpath == '\0'))
    {
    return(0);
    }

  ret = stat(fpath, &sbuf);

  if (ret == 0)
    {
    return(sbuf.st_mtime);
    }

  return(0);
  }  /* END MOMGetFileMtime */





/*
 * MOMCheckRestart() - set mom_run_state to restart if appropriate.
 *                     this is called when no jobs are running (below
 *                     in the main loop, and in mom_job_purge().)
 */

void MOMCheckRestart(void)

  {
  time_t newmtime;

  /* make sure we're not making a mess in the aux dir */
  cleanup_aux();

  if ((MOMConfigRestart <= 0) || (MOMExeTime <= 0))
    {
    return;
    }

  newmtime = MOMGetFileMtime(MOMExePath);

  if ((newmtime > 0) && 
      (newmtime != MOMExeTime))
    {
    if (mom_run_state == MOM_RUN_STATE_RUNNING)
      mom_run_state = MOM_RUN_STATE_RESTART;

    sprintf(
      log_buffer,
      "%s has changed, initiating re-exec (now: %ld, was: %ld)",
      MOMExePath,
      (long int)newmtime,
      (long int)MOMExeTime);

    if (LOGLEVEL > 6)
      {
      log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, msg_daemonname, log_buffer);
      }

    DBPRT(("%s\n", log_buffer));
    }
  }  /* END MOMCheckRestart() */




/*
 * initialize_globals
 */

void initialize_globals(void)

  {
  char  *ptr;                   /* local tmp variable */

  strcpy(pbs_current_user, "pbs_mom");
  msg_daemonname = pbs_current_user;

  time(&MOMStartTime);

  CLEAR_HEAD(svr_newjobs);
  CLEAR_HEAD(svr_alljobs);
  CLEAR_HEAD(mom_polljobs);
  CLEAR_HEAD(svr_requests);
  CLEAR_HEAD(mom_varattrs);


  if (getenv("PBSMOMHOME") != NULL)
    {
    path_home = getenv("PBSMOMHOME");
    }

  MOMConfigVersion[0] = '\0';

  mom_server_all_init();

  pbsgroup = getgid();
  pbsuser  = getuid();
  loopcnt  = time(NULL);

  MOMExePath = MOMFindMyExe(program_name);
  MOMExeTime = MOMGetFileMtime(MOMExePath);

  strcpy(xauth_path, XAUTH_PATH);
  strcpy(rcp_path, RCP_PATH);
  strcpy(rcp_args, RCP_ARGS);
#ifdef DEFAULT_MOMLOGDIR
  path_log = strdup(DEFAULT_MOMLOGDIR);
#endif
#ifdef DEFAULT_MOMLOGSUFFIX
  log_init(DEFAULT_MOMLOGSUFFIX, NULL);
#endif

  /* PATH is restored before a restart */

  if (getenv("PATH") != NULL)
    {
    orig_path = strdup(getenv("PATH"));
    }

  /* get default service port */

  ptr = getenv("PBS_MOM_SERVICE_PORT");

  if (ptr != NULL)
    {
    pbs_mom_port = (int)strtol(ptr, NULL, 10);
    }

  if (pbs_mom_port <= 0)
    {
    pbs_mom_port = get_svrport(
                     PBS_MOM_SERVICE_NAME,
                     "tcp",
                     PBS_MOM_SERVICE_PORT);
    }

  ptr = getenv("PBS_BATCH_SERVICE_PORT");

  if (ptr != NULL)
    {
    default_server_port = (int)strtol(ptr, NULL, 10);
    }

  if (default_server_port <= 0)
    {
    default_server_port = get_svrport(
                            PBS_BATCH_SERVICE_NAME,
                            "tcp",
                            PBS_BATCH_SERVICE_PORT);
    }

  ptr = getenv("PBS_MANAGER_SERVICE_PORT");

  if (ptr != NULL)
    {
    pbs_rm_port = (int)strtol(ptr, NULL, 10);
    }

  if (pbs_rm_port <= 0)
    {
    pbs_rm_port = get_svrport(
                    PBS_MANAGER_SERVICE_NAME,
                    "tcp",
                    PBS_MANAGER_SERVICE_PORT);
    }

  /* set timeout values for MOM */

  MaxConnectTimeout = 10000;  /* in microseconds */

  memset(JobsToResend,0,sizeof(JobsToResend));

  /* set the mom alias name to nothing */
  mom_alias[0] = '\0';
  lock_init();
  }  /* END initialize_globals() */



/*
 * stop_me = signal handler for SIGTERM
 */

static void stop_me(

  int sig)  /* I */

  {
  const char *dowhat;

  /* just exit, leaving jobs running */

  mom_run_state = MOM_RUN_STATE_EXIT;

  dowhat = "leaving jobs running, just exiting";

  sprintf(log_buffer, "caught signal %d: %s",
          sig,
          dowhat);

  log_record(
    PBSEVENT_SYSTEM | PBSEVENT_FORCE,
    PBS_EVENTCLASS_SERVER,
    msg_daemonname,
    log_buffer);

  return;
  }  /* END void stop_me() */




/*
 * PBSAdjustLogLevel
 */

static void PBSAdjustLogLevel(

  int sig)  /* I */

  {
  if (sig == SIGUSR1)
    {
    /* increase log level */

    LOGLEVEL = MIN(LOGLEVEL + 1, 10);
    }
  else if (sig == SIGUSR2)
    {
    /* increase log level */

    LOGLEVEL = MAX(LOGLEVEL - 1, 0);
    }

  sprintf(log_buffer, "received signal %d: adjusting loglevel to %d",

          sig,
          LOGLEVEL);

  log_record(
    PBSEVENT_SYSTEM | PBSEVENT_FORCE,
    PBS_EVENTCLASS_SERVER,
    msg_daemonname,
    log_buffer);

  return;
  }  /* END PBSAdjustLogLevel() */





/*
 * mk_dirs - make the directory names used by MOM
 */

char *mk_dirs(

  char *base)  /* I */

  {
  char *pn;
  int   ltop = strlen(path_home);

  pn = calloc(1, ltop + strlen(base) + 2);

  if (pn == NULL)
    {
    /* cannot allocate memory */

    exit(2);
    }

  strcpy(pn, path_home);

  if (*(path_home + ltop - 1) != '/')
    strcat(pn, "/");

  strcat(pn, base);

  return(pn);
  }  /* END mk_dirs() */

/*
 * parse_command_line
 */

void parse_command_line(

  int   argc,    /* I */
  char *argv[])  /* I */

  {
  extern char *optarg;
  extern int   optind;
  int          errflg;
  int          c;
  char        *ptr;                   /* local tmp variable */

  errflg = 0;

  while ((c = getopt(argc, argv, "a:A:c:C:d:DhH:l:L:mM:pPqrR:s:S:vwx-:")) != -1)
    {
    switch (c)
      {

      case '-':

        if (optarg == NULL)
          break;

        if (!strcmp(optarg, "about"))
          {
          printf("package:     %s\n", PACKAGE_STRING);
          printf("sourcedir:   %s\n", PBS_SOURCE_DIR);
          printf("configure:   %s\n", PBS_CONFIG_ARGS);
          printf("buildcflags: %s\n", PBS_CFLAGS);
          printf("buildhost:   %s\n", PBS_BUILD_HOST);
          printf("builddate:   %s\n", PBS_BUILD_DATE);
          printf("builddir:    %s\n", PBS_BUILD_DIR);
          printf("builduser:   %s\n", PBS_BUILD_USER);
          printf("installdir:  %s\n", PBS_INSTALL_DIR);
          printf("serverhome:  %s\n", PBS_SERVER_HOME);
          printf("version:     %s\n", PACKAGE_VERSION);

          exit(0);
          }
        else if (!strcmp(optarg, "version"))
          {
          printf("Version: %s\nRevision: %s\n",
            PACKAGE_VERSION, GIT_HASH);

          exit(0);
          }
        else if (!strcmp(optarg, "help"))
          {
          usage(argv[0]);

          exit(0);
          }
        else
          {
          errflg = 1;
          }

        break;

      case 'a':

        alarm_time = (int)strtol(optarg, &ptr, 10);

        if ((alarm_time <= 0) || (*ptr != '\0'))
          {
          fprintf(stderr, "%s: bad alarm time\n",
                  optarg);

          errflg = 1;
          }

        break;

      case 'A':

        /* mom's alias name - used for multi-mom */

        snprintf(mom_alias, sizeof(mom_alias), "%s", optarg);

        break;

      case 'c': /* config file */

        config_file_specified = 1;

        strcpy(config_file, optarg); /* remember name */

        break;

      case 'h':

        usage(argv[0]);  /* exits */

        exit(0);

        break;

      case 'H': /* multihomed host */

        hostname_specified = 1;

        snprintf(mom_host, sizeof(mom_host), "%s", optarg);

        break;

      case 'C':
        mom_checkpoint_set_directory_path(optarg);
        break;

      case 'd': /* directory */

        path_home = optarg;

        break;

      case 'D':  /* debug */

        DOBACKGROUND = 0;

        break;

      case 'l':

        path_log = strdup(optarg);

        break;

      case 'L':

        log_file = optarg;

        break;

      case 'M':

        pbs_mom_port = (unsigned int)atoi(optarg);

        if (pbs_mom_port == 0)
          {
          fprintf(stderr, "Bad MOM port value %s\n",
                  optarg);

          exit(1);
          }

        break;

      case 'm':

        multi_mom = 1;
        break;

      case 'p':

        if (!recover_set)
          {
          recover = JOB_RECOV_RUNNING;
          recover_set = TRUE;
          }
        else
          {
          errflg = 1;
          }

        break;

      case 'P':

        if ( !recover_set )
          {
          recover = JOB_RECOV_DELETE;
          recover_set = TRUE;
          }
        else
          {
          errflg = 1;
          }

        break;

      case 'r':

        if (!recover_set)
          {
          recover = JOB_RECOV_TERM_REQUE;
          recover_set = TRUE;
          }
        else
          {
          errflg = 1;
          }

        break;

      case 'q':

        if (!recover_set)
          {
          recover = JOB_RECOV_REQUE;
          recover_set = TRUE;
          }
        else
          {
          errflg = 1;
          }

        break;

      case 'R':

        pbs_rm_port = (unsigned int)atoi(optarg);

        if (pbs_rm_port == 0)
          {
          fprintf(stderr, "Bad RM port value %s\n",
                  optarg);

          exit(1);
          }

        break;

      case 's':

        log_init(optarg, NULL);

        break;

      case 'S':

        default_server_port = (unsigned int)atoi(optarg);

        if (default_server_port == 0)
          {
          fprintf(stderr, "Bad Server port value %s\n",
                  optarg);

          exit(1);
          }

        break;

      case 'v':

        fprintf(stderr, "version: %s\n",
                PACKAGE_VERSION);

        exit(0);

        break;

      case 'w':

        /* wait 10 minutes or until you get the mom hierarchy file from the 
         * server to send your first update */
        first_update_time = time(NULL) + 600;

        break;

      case 'x':

        port_care = FALSE;

        break;

      case '?':

      default:

        errflg = 1;

        break;
      }  /* END switch(c) */
    }    /* END while ((c = getopt(argc,argv,"a:c:C:d:Dh:L:M:prR:S:vx-:")) != -1) */

  if ((errflg > 0) || (optind != argc))
    {
    usage(argv[0]);  /* exits */

    exit(1);
    }

  return;
  }  /* END parse_command_line() */






/**
 * setup_program_environment
 */

int setup_program_environment(void)

  {
  int           c;
  int           hostc = 1;
#if !defined(DEBUG) && !defined(DISABLE_DAEMONS)
  FILE         *dummyfile;
#endif
  char logSuffix[MAX_PORT_STRING_LEN];
  char momLock[MAX_LOCK_FILE_NAME_LEN];
#ifdef PENABLE_LINUX26_CPUSETS
  int  rc;
#endif

  struct sigaction act;
  char         *ptr;            /* local tmp variable */

  /* must be started with real and effective uid of 0 */
  if (IamRoot() == 0)
    {
    return(1);
    }

  initialize_network_info();

  /* The following is code to reduce security risks                */
  /* start out with standard umask, system resource limit infinite */

  umask(022);

  if (getenv("PBSLOGLEVEL") != NULL)
    {
    LOGLEVEL = (int)strtol(getenv("PBSLOGLEVEL"), NULL, 0);
    }

  if (getenv("PBSDEBUG") != NULL)
    {
    DEBUGMODE = 1;

    DOBACKGROUND = 0;
    }

  memset(TMOMStartInfo, 0, sizeof(pjobexec_t)*TMAX_JE);

  /* modify program environment */

  if ((num_var_env = setup_env(PBS_ENVIRON)) == -1)
    {
    exit(1);
    }

  c = getgid();

  /* secure suppl. groups */
  if (setgroups(1,(gid_t *)&c) != 0)
    {
    snprintf(log_buffer, sizeof(log_buffer),
      "Unable to drop secondary groups. Some MAC framework is active?\n");
    log_err(errno, __func__, log_buffer);
    snprintf(log_buffer, sizeof(log_buffer),
      "setgroups(group = %lu) failed: %s\n",
      (unsigned long)c, strerror(errno));
    log_err(errno, __func__, log_buffer);

    return(1);
    }

#ifndef DEBUG
#ifdef _CRAY

  limit(C_JOB,      0, L_CPROC, 0);
  limit(C_JOB,      0, L_CPU,   0);
  limit(C_JOBPROCS, 0, L_CPU,   0);
  limit(C_PROC,     0, L_FD,  255);
  limit(C_JOB,      0, L_FSBLK, 0);
  limit(C_JOBPROCS, 0, L_FSBLK, 0);
  limit(C_JOB,      0, L_MEM  , 0);
  limit(C_JOBPROCS, 0, L_MEM  , 0);

#else /* _CRAY */

    {

    struct rlimit rlimit;

    rlimit.rlim_cur = RLIM_INFINITY;
    rlimit.rlim_max = RLIM_INFINITY;
    setrlimit(RLIMIT_CPU,   &rlimit);
    setrlimit(RLIMIT_FSIZE, &rlimit);
    setrlimit(RLIMIT_DATA,  &rlimit);
#ifdef RLIMIT_RSS
    setrlimit(RLIMIT_RSS,   &rlimit);
#endif /* RLIMIT_RSS */
#ifdef RLIMIT_VMEM
    setrlimit(RLIMIT_VMEM, &rlimit);
#endif /* RLIMIT_VMEM */
    }  /* END BLOCK */
#endif /* else _CRAY */
#endif /* DEBUG */

  /* set up and validate home paths */

  c = 0;

  mom_home              = mk_dirs("mom_priv");
  path_jobs             = mk_dirs("mom_priv/jobs/");
  path_epilog           = mk_dirs("mom_priv/epilogue");
  path_prolog           = mk_dirs("mom_priv/prologue");
  path_epiloguser       = mk_dirs("mom_priv/epilogue.user");
  path_prologuser       = mk_dirs("mom_priv/prologue.user");
  path_epilogp          = mk_dirs("mom_priv/epilogue.parallel");
  path_prologp          = mk_dirs("mom_priv/prologue.parallel");
  path_epiloguserp      = mk_dirs("mom_priv/epilogue.user.parallel");
  path_prologuserp      = mk_dirs("mom_priv/prologue.user.parallel");
  path_epilogpdel       = mk_dirs("mom_priv/epilogue.precancel");
  path_layout           = mk_dirs("mom_priv/mom.layout");
  path_mom_hierarchy    = mk_dirs("mom_priv/mom_hierarchy");

#ifndef DEFAULT_MOMLOGDIR

  if (path_log == NULL)
    path_log       = mk_dirs("mom_logs");

#endif

  path_spool       = mk_dirs("spool/");

  path_undeliv     = mk_dirs("undelivered/");

#ifdef __CYGWIN__
/*  AUX is reserved word in Windows  */
  path_aux         = mk_dirs("auxx/");
#else
  path_aux         = mk_dirs("aux/");
#endif  /* __CYGWIN__ */

  /* initialize the mom_status */
  mom_status = get_dynamic_string(16 * 1024, NULL);

  init_resc_defs();

  c |= mom_checkpoint_init();

  /* change working directory to mom_priv */

  if (chdir(mom_home) == -1)
    {
    char tmpLine[1024];

    sprintf(tmpLine, "cannot change directory to home '%s'",
            mom_home);

    perror(tmpLine);

    return(1);
    }

#if !defined(DEBUG) && !defined(NO_SECURITY_CHECK)

  c |= chk_file_sec(path_jobs,    1, 0, S_IWGRP | S_IWOTH, 1, NULL);

  c |= chk_file_sec(path_aux,     1, 0, S_IWGRP | S_IWOTH, 1, NULL);

  c |= chk_file_sec(path_spool,   1, 1, S_IWOTH,         0, NULL);

  c |= chk_file_sec(path_undeliv, 1, 1, S_IWOTH,         0, NULL);

  c |= chk_file_sec(PBS_ENVIRON,  0, 0, S_IWGRP | S_IWOTH, 0, NULL);

  if (c)
    {
    return(3);
    }

#endif  /* not DEBUG and not NO_SECURITY_CHECK */

  if (hostname_specified == 0)
    {
    hostc = gethostname(mom_host, PBS_MAXHOSTNAME);
    }

  if (!multi_mom)
    {
    log_init(NULL, mom_host);
    }
  else
    {
    sprintf(logSuffix, "%d", pbs_mom_port);
    log_init(logSuffix, mom_host);
    }
 
  /* open log file while std in,out,err still open, forces to fd 4 */
  pthread_mutex_lock(log_mutex);
  if ((c = log_open(log_file, path_log)) != 0)
    {
    pthread_mutex_unlock(log_mutex);
    /* use given name */

    fprintf(stderr, "pbs_mom: Unable to open logfile\n");

    return(1);
    }
  pthread_mutex_unlock(log_mutex);

  check_log(); /* see if this log should be rolled */

  if (!multi_mom)
    sprintf(momLock,"mom.lock");
  else
    sprintf(momLock, "mom%d.lock", pbs_mom_port);

  lockfds = open(momLock, O_CREAT | O_WRONLY, 0644);

  if (lockfds < 0)
    {
    sprintf(log_buffer, "pbs_mom: unable to open lock file - errno=%d '%s'\n",
      errno,
      strerror(errno));

    fprintf(stderr, "%s",
      log_buffer);

    return(1);
    }

  mom_lock(lockfds, F_WRLCK); /* See if other MOMs are running */

  /* initialize the network interface */

  if (init_network(pbs_mom_port, mom_process_request) != 0)
    {
    c = errno;

    sprintf(log_buffer, "server port = %u, errno = %d (%s)",
      pbs_mom_port,
      c,
      strerror(c));

    if (c == EADDRINUSE)
      strcat(log_buffer, ", already in use");

    log_err(-1, msg_daemonname, log_buffer);

    strcat(log_buffer, "\n");

    fprintf(stderr, "%s", log_buffer);

    return(3);
    }

  if (init_network(pbs_rm_port, tcp_request) != 0)
    {
    c = errno;

    sprintf(log_buffer, "resource (tcp) port = %u, errno = %d (%s)",
      pbs_rm_port,
      c,
      strerror(c));

    if (c == EADDRINUSE)
      strcat(log_buffer, ", already in use");

    log_err(-1, msg_daemonname, log_buffer);

    strcat(log_buffer, "\n");

    fprintf(stderr, "%s", log_buffer);

    return(3);
    }

#ifdef PENABLE_LINUX26_CPUSETS
  /* load system topology */
  if ((hwloc_topology_init(&topology) == -1))
    {
    log_err(-1, msg_daemonname, "Unable to init machine topology");
    return(-1);
    }

  if ((hwloc_topology_set_flags(topology, HWLOC_TOPOLOGY_FLAG_WHOLE_SYSTEM) != 0))
    {
    log_err(-1, msg_daemonname, "Unable to configure machine topology");
    return(-1);
    }

  if ((hwloc_topology_load(topology) == -1))
    {
    log_err(-1, msg_daemonname, "Unable to load machine topology");
    return(-1);
    }

  sprintf(log_buffer, "machine topology contains %d memory nodes, %d cpus",
    hwloc_get_nbobjs_by_type(topology, HWLOC_OBJ_NODE),
    hwloc_get_nbobjs_by_type(topology, HWLOC_OBJ_PU));
  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, log_buffer);

#endif

#ifdef NUMA_SUPPORT
  if ((rc = setup_nodeboards()) != 0)
    return(rc);
#else
  snprintf(path_meminfo,sizeof(path_meminfo),"%s", "/proc/meminfo");
#endif /* END NUMA_SUPPORT */

#if defined(PENABLE_LINUX26_CPUSETS)
  /* NOTE: moved to before we go into background so that we can print an error
   * message if we can't mount the cpuset directory and it isn't mounted */
  /* Create the top level torque cpuset if it doesn't already exist. */
  if ((rc = init_torque_cpuset()) != 0)
    return(rc);
#endif

  /* go into the background and become own session/process group */

#if !defined(DEBUG) && !defined(DISABLE_DAEMONS)

  mom_lock(lockfds, F_UNLCK); /* unlock so child can relock */

  if (DOBACKGROUND == 1)
    {
    if (fork() > 0)
      {
      exit(0); /* parent goes away */
      }

    if (setsid() == -1)
      {
      log_err(errno, msg_daemonname, "setsid failed");

      return(2);
      }

    fclose(stdin);

    fclose(stdout);
    fclose(stderr);

    dummyfile = fopen("/dev/null", "r");
    assert((dummyfile != 0) && (fileno(dummyfile) == 0));

    dummyfile = fopen("/dev/null", "w");
    assert((dummyfile != 0) && (fileno(dummyfile) == 1));

    dummyfile = fopen("/dev/null", "w");
    assert((dummyfile != 0) && (fileno(dummyfile) == 2));
    }  /* END if (DOBACKGROUND == 1) */

  mom_lock(lockfds, F_WRLCK); /* lock out other MOMs */

#else /* DEBUG */
#if defined(_CRAY)

  /* CRAY cannot restart checkpointed job if MOM has controlling tty */

  sprintf(log_buffer, "/tmp/pbs_mom.%d",
          getpid());

  printf("Debug output will be in %s\n",
         log_buffer);

  freopen(log_buffer, "w", stdout);

  freopen(log_buffer, "w", stderr);

  ioctl(0, TCCLRCTTY, 0);

  close(0);

#endif /* _CRAY */
  setvbuf(stdout, NULL, _IOLBF, 0);

  setvbuf(stderr, NULL, _IOLBF, 0);

#endif /* DEBUG */

  /* write MOM's pid into lockfile */

  if (ftruncate(lockfds, (off_t)0) != 0)
    {
    log_err(errno, msg_daemonname, "failed to truncate lockfile");

    return(2);
    }

  sprintf(log_buffer, "%ld\n", (long)getpid());

  if (write(lockfds, log_buffer, strlen(log_buffer) + 1) !=
      (ssize_t)(strlen(log_buffer) + 1))
    {
    log_err(errno, msg_daemonname, "failed to write to lockfile");

    return(2);
    }

#if (PLOCK_DAEMONS & 4)
  /* lock daemon into memory */

  /* NOTE:  should reduce maximum stack limit using ulimit() before calling plock */

  if (plock(PROCLOCK) == -1)
    {
    log_err(errno, msg_daemonname, "failed to lock mom into memory with plock");
    }
  else
    {
    MOMIsPLocked = 1;
    }

#endif /* PLOCK_DAEMONS */

  sigemptyset(&allsigs);

  act.sa_mask = allsigs;

  act.sa_flags = 0;

  /*
  ** Signals to be ignored.
  */

  act.sa_handler = SIG_IGN;

  sigaction(SIGPIPE, &act, NULL);

#ifdef SIGINFO
  sigaction(SIGINFO, &act, NULL);

#endif /* SIGINFO */

  sigaddset(&allsigs, SIGHUP); /* remember to block these */

  sigaddset(&allsigs, SIGINT); /* during critical sections */

  sigaddset(&allsigs, SIGTERM); /* so we don't get confused */

  sigaddset(&allsigs, SIGCHLD);

#ifdef _CRAY
  sigaddset(&allsigs, WJSIGNAL);

#endif
  act.sa_mask = allsigs;

  /*
  ** We want to abort system calls
  ** and call a function.
  */
#ifdef SA_INTERRUPT
  act.sa_flags |= SA_INTERRUPT; /* don't restart system calls */

#endif

#ifdef NOSIGCHLDMOM
  act.sa_handler = SIG_DFL;
#else
  act.sa_handler = catch_child;	/* set up to catch death of child */
#endif

  sigaction(SIGCHLD, &act, NULL);

#ifdef _CRAY
  sigaction(WJSIGNAL, &act, NULL);

#endif

  /*
   * Catch these signals to ensure we core dump even if
   * our rlimit for core dumps is set to 0 initially.
   *
   * Chris Samuel - VPAC
   * csamuel@vpac.org - 29th July 2003
   *
   * Now conditional on the PBSCOREDUMP environment variable.
   */

  if (getenv("PBSCOREDUMP"))
    {
    act.sa_handler = catch_abort;   /* make sure we core dump */

    sigaction(SIGSEGV, &act, NULL);
    sigaction(SIGBUS, &act, NULL);
    sigaction(SIGFPE, &act, NULL);
    sigaction(SIGILL, &act, NULL);
    sigaction(SIGTRAP, &act, NULL);
    sigaction(SIGSYS, &act, NULL);
    }

  act.sa_handler = catch_hup; /* do a restart on SIGHUP */

  sigaction(SIGHUP, &act, NULL);

  act.sa_handler = toolong; /* handle an alarm call */
  sigaction(SIGALRM, &act, NULL);

  act.sa_handler = stop_me; /* shutdown for these */
  sigaction(SIGINT, &act, NULL);
  sigaction(SIGTERM, &act, NULL);

  act.sa_handler = PBSAdjustLogLevel;
  sigaction(SIGUSR1, &act, NULL);
  sigaction(SIGUSR2, &act, NULL);

#ifdef SIGXCPU
  sigaction(SIGXCPU, &act, NULL);
#endif

#ifdef SIGXFSZ
  sigaction(SIGXFSZ, &act, NULL);
#endif

#ifdef SIGCPULIM
  sigaction(SIGCPULIM, &act, NULL);
#endif

#ifdef SIGSHUTDN
  sigaction(SIGSHUTDN, &act, NULL);
#endif

#ifdef _CRAY

  /* Special code for CRAY MLS Systems */

  if (sysconf(_SC_CRAY_SECURE_SYS))
    {

    struct usrv usrv;

    if (getusrv(&usrv) < 0)
      {
      fprintf(stderr, "cannot get security info\n");

      return(1);
      }

    usrv.sv_permit = 0;

    usrv.sv_intcat = 0;
    usrv.sv_valcat = 0;

    if (setusrv(&usrv) < 0)
      {
      fprintf(stderr, "cannot put security info\n");

      return(1);
      }

    if (setucat(0) < 0)
      {
      fprintf(stderr, "cannot put security cat\n");

      return(2);
      }
    }

#endif /* _CRAY */

  /* initialize variables */


  if ((hostname_specified != 0) || (hostc == 0))
    {
    strcpy(mom_short_name, mom_host);

    c = get_fullhostname(mom_host, mom_host, PBS_MAXHOSTNAME, NULL);

    if (c != 0)
      {
      char logbuf[1024];

      snprintf(logbuf, 1024, "Unable to get my full hostname for %s error %d",
               mom_host,
               c);

      log_err(-1, msg_daemonname, logbuf);

      return(-1);
      }
    }

  if (c == -1)
    {
    log_err(-1, msg_daemonname, "Unable to get my host name");

    return(-1);
    }

  time_now = time((time_t *)0);

  ret_size = RETURN_STRING_SIZE;

  if ((ret_string = calloc(1, ret_size)) == NULL)
    {
    perror("calloc");

    exit(1);
    }

  localaddr = ntohl(inet_addr("127.0.0.1"));

  okclients = AVL_insert(localaddr, 0, NULL, okclients);

  addclient(mom_host);

  if (gethostname(ret_string, ret_size) == 0)
    addclient(ret_string);

  tmpdir_basename[0] = '\0';

  if (read_config(NULL))
    {
    fprintf(stderr, "%s: cannot load config file '%s'\n",
            program_name,
            config_file);

    exit(1);
    }

  /* if no alias is specified, make mom_alias the same as mom_host */
  if (mom_alias[0] == '\0')
    strcpy(mom_alias,mom_short_name);

  initialize();  /* init RM code */

  mh = initialize_mom_hierarchy();

  /* initialize machine-dependent polling routines */
  if ((c = mom_open_poll()) != PBSE_NONE)
    {
    log_err(c, msg_daemonname, "pre_poll failed");

    return(3);
    }

  if (mom_get_sample() != PBSE_NONE)
    {
    log_err(c, msg_daemonname, "mom_get_sample failed after mom_open_poll");

    return(3);
    }
  
  things_to_resend = initialize_resizable_array(10);

  exiting_job_list = initialize_resizable_array(10);

  /* recover & abort jobs which were under MOM's control */
  log_record(
    PBSEVENT_DEBUG,
    PBS_EVENTCLASS_SERVER,
    msg_daemonname,
    "before init_abort_jobs");

  init_abort_jobs(recover);

#if PENABLE_LINUX26_CPUSETS
  /* Nuke cpusets that do not correspond to existing jobs */
  cleanup_torque_cpuset();
#endif

#ifdef _POSIX_MEMLOCK
  /* call mlockall() only 1 time, since it seems to leak mem */
  if (MOMIsLocked == 0)
    {
    int mlockall_return;

    /* make sure pbs_mom stays in RAM and doesn't get paged out */
    mlockall_return = mlockall(MCL_CURRENT | MCL_FUTURE);

    /* exit iff mlock failed, but ignore function not implemented error */
    if ((mlockall_return == -1) && (errno != ENOSYS))
      {
      perror("pbs_mom:mom_main.c:mlockall()");

      exit(1);
      }

    MOMIsLocked = 1;
    }
#endif /* _POSIX_MEMLOCK */

  /* record the fact that we are up and running */
  log_record(
    PBSEVENT_SYSTEM | PBSEVENT_FORCE,
    PBS_EVENTCLASS_SERVER,
    msg_daemonname,
    "Is up");

  DBPRT(("MOM is up\n"));

  sprintf(log_buffer, "MOM executable path and mtime at launch: %s %ld",
    MOMExePath == NULL ? "NULL" : MOMExePath,
    (long int)MOMExeTime);

  log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, __func__, log_buffer);

  ptr = getenv("MOMSLEEPTIME");

  if (ptr != NULL)
    {
    long tmpL;

    tmpL = strtol(ptr, NULL, 10);

    sleep(tmpL % (rand() + 1));
    }  /* END if (ptr != NULL) */

  initialize_threadpool(&request_pool,MOM_THREADS,MOM_THREADS,THREAD_INFINITE);

  received_cluster_addrs = FALSE;
  requested_cluster_addrs = 0;

  /* allocate status strings if needed */
  received_statuses = initialize_resizable_array(2);
  received_table = create_hash(101);


  if ((received_statuses == NULL) ||
      (received_table == NULL))
    {
    log_err(ENOMEM, __func__, "No memory!!!");
    return(-1);
    }

  srand(get_random_number());

  return(PBSE_NONE);
  }  /* END setup_program_environment() */






/*
 * TMOMJobGetStartInfo
 *
 * NOTE: if pjob is NULL, return empty slot, otherwise return slot containing job.
 */

int TMOMJobGetStartInfo(

  job         *pjob, /* I */
  pjobexec_t **TJEP) /* O */

  {
  int index;

  for (index = 0;index < TMAX_JE;index++)
    {
    if (TMOMStartInfo[index].pjob == pjob)
      {
      *TJEP = &TMOMStartInfo[index];

      return(SUCCESS);
      }
    } /* END for (index) */

  return(FAILURE);
  }  /* END TMOMJobGetStartInfo() */



/*
 * TMOMScanForStarting
 */

int TMOMScanForStarting(void)

  {
  job *pjob;
  job *nextjob;

  int  Count;
  int  RC;
  int  SC;

#ifdef MSIC
  list_link *tmpL;
#endif

#ifdef MSIC
  /* NOTE:  solaris system is choking on GET_NEXT - isolate */

  tmpL = GET_NEXT(svr_alljobs);

  tmpL = svr_alljobs.ll_next->ll_struct;

  pjob = (job *)tmpL;
#endif /* MSIC */

  pjob = (job *)GET_NEXT(svr_alljobs);

  while (pjob != NULL)
    {
    nextjob = (job *)GET_NEXT(pjob->ji_alljobs);

    if (pjob->ji_qs.ji_substate == JOB_SUBSTATE_STARTING)
      {
      pjobexec_t *TJE = NULL;

      if (LOGLEVEL >= 2)
        {
        snprintf(log_buffer, sizeof(log_buffer),
          "checking job start in %s - examining pipe from child",
          __func__);

        log_record(
          PBSEVENT_JOB | PBSEVENT_FORCE,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          log_buffer);
        }

      if (TMOMJobGetStartInfo(pjob, &TJE) == FAILURE)
        {
        sprintf(log_buffer, "job %s start data lost, server will retry",
                pjob->ji_qs.ji_jobid);

        log_record(
          PBSEVENT_ERROR,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          log_buffer);

        exec_bail(pjob, JOB_EXEC_RETRY);

        pjob = nextjob;

        continue;
        }

      /* check if job is ready */

      if (TMomCheckJobChild(TJE, 1, &Count, &RC) == FAILURE)
        {
        long STime;

        if (LOGLEVEL >= 3)
          {
          sprintf(log_buffer, "job %s child not started, will check later",
                  pjob->ji_qs.ji_jobid);

          log_record(
            PBSEVENT_ERROR,
            PBS_EVENTCLASS_JOB,
            pjob->ji_qs.ji_jobid,
            log_buffer);
          }

        /* if job has been in prerun > TJobStartTimeout, purge job */

        STime = pjob->ji_wattr[JOB_ATR_mtime].at_val.at_long;

        if ((STime > 0) && ((time_now - STime) > TJobStartTimeout))
          {
          sprintf(log_buffer, "job %s child not started after %ld seconds, server will retry",
                  pjob->ji_qs.ji_jobid,
                  TJobStartTimeout);

          log_record(
            PBSEVENT_ERROR,
            PBS_EVENTCLASS_JOB,
            pjob->ji_qs.ji_jobid,
            log_buffer);

          memset(TJE, 0, sizeof(pjobexec_t));

          exec_bail(pjob, JOB_EXEC_RETRY);
          }
        }
      else
        {
        /* NOTE:  TMomFinalizeJob3() populates SC */

        if (TMomFinalizeJob3(TJE, Count, RC, &SC) == FAILURE)
          {
          /* no need to log this, TMomFinalizeJob3() already did */

          memset(TJE, 0, sizeof(pjobexec_t));

          exec_bail(pjob, SC);
          }
        else
          {
          /* job successfully started */

          memset(TJE, 0, sizeof(pjobexec_t));

          if (LOGLEVEL >= 3)
            {
            sprintf(log_buffer, "%s:job %s reported successful start",
              __func__,
              pjob->ji_qs.ji_jobid);

            log_event(PBSEVENT_JOB,PBS_EVENTCLASS_JOB,pjob->ji_qs.ji_jobid,log_buffer);
            }
          }
        }    /* END else (TMomCheckJobChild() == FAILURE) */
      }      /* END if (pjob->ji_qs.ji_substate == JOB_SUBSTATE_STARTING) */

    pjob = nextjob;
    }        /* END while (pjob != NULL) */

  return(SUCCESS);
  }  /* END TMOMScanForStarting() */





/**
 * examine_all_polled_jobs
 *
 * check on over limit condition for polled jobs
 */

void examine_all_polled_jobs(void)

  {
  job         *pjob;
  int         c;


  for (pjob = (job *)GET_NEXT(mom_polljobs);pjob;
       pjob = (job *)GET_NEXT(pjob->ji_jobque))
    {
    if (pjob->ji_qs.ji_substate != JOB_SUBSTATE_RUNNING)
      continue;

    /*
    ** Send message to get info from other MOM's
    ** if I am Mother Superior for the job and
    ** it is not being killed.
    */

    if ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) &&
        (pjob->ji_nodekill == TM_ERROR_NODE))
      {
      /*
      ** If can't send poll to everybody, the
      ** time has come to die.
      */

      if (send_sisters(pjob, IM_POLL_JOB, FALSE) != pjob->ji_numnodes - 1)
        {
        sprintf(log_buffer, "cannot contact all sisters");

        log_record(PBSEVENT_JOB | PBSEVENT_FORCE,
                   PBS_EVENTCLASS_JOB,
                   pjob->ji_qs.ji_jobid,
                   log_buffer);
        }
      }

    c = pjob->ji_qs.ji_svrflags;

    if (c & JOB_SVFLG_OVERLMT2)
      {
      kill_job(pjob, SIGKILL, __func__, "job is over-limit-2");

      continue;
      }

    if (c & JOB_SVFLG_OVERLMT1)
      {
      kill_job(pjob, SIGTERM, __func__, "job is over-limit-1");

      pjob->ji_qs.ji_svrflags |= JOB_SVFLG_OVERLMT2;

      continue;
      }

    log_buffer[0] = '\0';

    if (job_over_limit(pjob) != 0)
      {
      log_record(
        PBSEVENT_JOB | PBSEVENT_FORCE,
        PBS_EVENTCLASS_JOB,
        pjob->ji_qs.ji_jobid,
        log_buffer);

      if (c & JOB_SVFLG_HERE)
        {
        char *kill_msg;

        kill_msg = calloc(1, 80 + strlen(log_buffer));

        if (kill_msg != NULL)
          {
          sprintf(kill_msg,"=>> PBS: job killed: %s\n",
            log_buffer);

          message_job(pjob,StdErr,kill_msg);

          free(kill_msg);
          }
        }

      kill_job(pjob, SIGTERM, __func__, "job is over-limit-0");

      pjob->ji_qs.ji_svrflags |= JOB_SVFLG_OVERLMT1;
      }
    }    /* END for (pjob) */

  return;
  }      /* END examine_all_polled_jobs() */





/*
 * examine_all_running_jobs
 */

void examine_all_running_jobs(void)

  {
  job         *pjob;
#ifdef _CRAY
  int         c;
#endif
  task         *ptask;

  for (pjob = (job *)GET_NEXT(svr_alljobs);
       pjob != NULL;
       pjob = (job *)GET_NEXT(pjob->ji_alljobs))
    {
    if (pjob->ji_qs.ji_substate != JOB_SUBSTATE_RUNNING)
      continue; /* This job is not running, skip it. */

    if ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0)
      continue; /* We are not the Mother Superior for this job, skip it. */

    /* update information for my tasks */

    mom_set_use(pjob); /* Machine dependent function to compute and set attributes like cput, vmem, etc. */

    /* Have all job processes vanished undetected?       */
    /* double check by sig0 to session pid for each task */
    /* But why not use the proc_array? */

    if (pjob->ji_flags & MOM_NO_PROC)
      {
      pjob->ji_flags &= ~MOM_NO_PROC;

      for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
           ptask != NULL;
           ptask = (task *)GET_NEXT(ptask->ti_jobtask))
        {
#ifdef _CRAY

        if (pjob->ji_globid[0] == '\0')
          break;

        c = atoi(pjob->ji_globid);

        if ((kill((pid_t)c, 0) == -1) && (errno == ESRCH))
#else /* not cray */
        if ((kill(ptask->ti_qs.ti_sid, 0) == -1) && (errno == ESRCH))
#endif /* not cray */
          {
          if (LOGLEVEL >= 3)
            {
            log_event(
              PBSEVENT_JOB,
              PBS_EVENTCLASS_JOB,
              pjob->ji_qs.ji_jobid,
              "no active process found");
            }

          ptask->ti_qs.ti_exitstat = 0;

          ptask->ti_qs.ti_status = TI_STATE_EXITED;
          pjob->ji_qs.ji_un.ji_momt.ji_exitstat = 0;

          if (LOGLEVEL >= 6)
            {
            log_record(
              PBSEVENT_ERROR,
              PBS_EVENTCLASS_JOB,
              pjob->ji_qs.ji_jobid,
              "saving task (main loop)");
            }

          task_save(ptask);

          exiting_tasks = 1;
          }  /* END if ((kill == -1) && ...) */
        }    /* END while (ptask != NULL) */
      }      /* END if (pjob->ji_flags & MOM_NO_PROC) */


    mom_checkpoint_check_periodic_timer(pjob);
    }  /* END for (pjob) */

  return;
  }  /* END examine_all_running_jobs() */





/**
 * examine_all_jobs_to_resend
 *
 * tries to resend each of the jobs that hasn't been sent yet
 */
void examine_all_jobs_to_resend(void)

  {
  int jindex;

  for (jindex=0;jindex < MAX_RESEND_JOBS;jindex++)
    {
    /* no job ptrs are stored after a NULL value */
    if (JobsToResend[jindex] == NULL)
      break;

    /* skip dummy job */
    if (JobsToResend[jindex] == (job *)DUMMY_JOB_PTR)
      continue;

    if (!post_epilogue(JobsToResend[jindex], MOM_OBIT_RETRY))
      {

      if (LOGLEVEL >= 7)
        {
        log_record(
          PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          JobsToResend[jindex]->ji_qs.ji_jobid,
          "job obit resent");
        }

      /* sent successfully, make this slot the dummy pointer */

      JobsToResend[jindex] = (job *)DUMMY_JOB_PTR;
      }
    }
  }  /* END examine_all_jobs_to_resend() */




void check_exiting_jobs()

  {
  exiting_job_info *eji;
  job              *pjob;
  time_t            time_now = time(NULL);

  while ((eji = (exiting_job_info *)pop_thing(exiting_job_list)) != NULL)
    {
    if (time_now - eji->obit_sent < 300)
      {
      /* insert this back at the front */
      insert_thing_after(exiting_job_list, eji, ALWAYS_EMPTY_INDEX);
      break;
      }

    pjob = mom_find_job(eji->jobid);

    if (pjob == NULL)
      {
      free(eji);
      }
    else
      {
      post_epilogue(pjob, 0);
      eji->obit_sent = time_now;
      insert_thing(exiting_job_list, eji);
      }
    }

  } /* END check_exiting_jobs() */




/*
 * kill_all_running_jobs
 */

void kill_all_running_jobs(void)

  {
  job *pjob;
  unsigned int momport = 0;

  for (pjob = (job *)GET_NEXT(svr_alljobs);
       pjob != NULL;
       pjob = (job *)GET_NEXT(pjob->ji_alljobs))
    {
    if (pjob->ji_qs.ji_substate == JOB_SUBSTATE_RUNNING)
      {
      kill_job(pjob, SIGKILL, "kill_all_running_jobs", "mom is terminating with kill jobs flag");

      pjob->ji_qs.ji_substate = JOB_SUBSTATE_EXITING;

      if (multi_mom)
        {
        momport = pbs_rm_port;
        }
      job_save(pjob, SAVEJOB_QUICK, momport);
      }
    else
      {
      term_job(pjob);
      }
    }  /* END for (pjob) */

#ifndef NOSIGCHLDMOM
  if (termin_child != 0)
#endif
    scan_for_terminated();

  if (exiting_tasks)
    scan_for_exiting();

  return;
  }  /* END kill_all_running_jobs() */



/**
 * mark_for_resend
 *
 * used to keep track of jobs whose obits weren't sent correctly
 * marks them so they can be resent
 *
 * @param pjob - the job that should be resent
 */
int mark_for_resend(

  job *pjob) /* I */

  {
  int jindex;
  int rc = FAILURE;

  if (pjob == NULL)
    return(rc);

  for (jindex = 0;jindex < MAX_RESEND_JOBS;jindex++)
    {
    if ((JobsToResend[jindex] == NULL) ||
        (JobsToResend[jindex] == (job *)DUMMY_JOB_PTR))
      {
      JobsToResend[jindex] = pjob;

      if (LOGLEVEL >= 7)
        {
        log_record(
          PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          "marking job for resend");
        }

      rc = SUCCESS;

      break;
      }
    }

  return(rc);
  }




/*
 * This is for a mom starting with the -P option. Set all existing
 * tasks to TI_STATE_EXITED so they can be cleanup up on the mom
 * and at the server
 */
void prepare_child_tasks_for_delete()

  {
  job *job;


  for (job = GET_NEXT(svr_alljobs);job != NULL;job = GET_NEXT(job->ji_alljobs))
    {
    task *task;

    for (task = GET_NEXT(job->ji_tasks);task != NULL;task = GET_NEXT(task->ti_jobtask))
      {

      char buf[128];

      extern int exiting_tasks;

      sprintf(buf, "preparing exited session %d for task %d in job %s for deletion",
              (int)task->ti_qs.ti_sid,
              task->ti_qs.ti_task,
              job->ji_qs.ji_jobid);

      log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, __func__, buf);

      task->ti_qs.ti_exitstat = 0;  /* actually unknown */
      task->ti_qs.ti_status = TI_STATE_EXITED;

      task_save(task);

      exiting_tasks = 1;
      }
    }
  }




/**
 * main_loop
 *
 * @see main() - parent
 */

void main_loop(void)

  {
  extern time_t wait_time;
  double        myla;
  job          *pjob;
  time_t        tmpTime;
#ifdef USESAVEDRESOURCES
  int           check_dead = TRUE;
#endif    /* USESAVEDRESOURCES */

  mom_run_state = MOM_RUN_STATE_RUNNING;  /* mom_run_state is altered by stop_me() or MOMCheckRestart() */

  while (mom_run_state == MOM_RUN_STATE_RUNNING)
    {
    if (call_hup)
      {
      process_hup();  /* Do a restart of resmom */
      }

    dep_main_loop_cycle();  /* Call machine dependent code periodically */

    time_now = time(NULL);

    /* check if loadave means we should be "busy" */

    if (max_load_val > 0.0)
      {
      get_la(&myla);  /* Machine dependent load average computation (for linux read contents of /proc/loadavg) */

      /* check if need to update busy state */

      check_busy(myla);
      }

    /* should we check the log file ?*/

    if (time_now >= (last_log_check + PBS_LOG_CHECK_RATE))
      {
      check_log();  /* Possibly do a log_roll */
      }

    /* check for what needs to be sent is now done within */
    mom_server_all_update_stat();

    /* if needed, update server with my state change */
    /* can be changed in check_busy(), query_adp(), and update_stat() */
    mom_server_all_send_state();

#ifdef USESAVEDRESOURCES
    /* if -p, must poll tasks inside jobs to look for completion */
    if ((check_dead) &&
        (recover == JOB_RECOV_RUNNING))
      scan_non_child_tasks();
#endif

    if (time_now >= (last_poll_time + CheckPollTime))
      {
      last_poll_time = time_now;
      
      if (GET_NEXT(svr_alljobs))
        {
        /* There are jobs, update process status from the OS */
        
        if (mom_get_sample() == PBSE_NONE)
          {
          /* no errors in getting process status information */
          
          examine_all_running_jobs();
          
          examine_all_polled_jobs();
          
          examine_all_jobs_to_resend();
          }
        }
      }

#ifdef USESAVEDRESOURCES
    check_dead = FALSE;
#endif    /* USESAVEDRESOURCES */

#ifndef NOSIGCHLDMOM
    if (termin_child != 0)  /* termin_child is a flag set by the catch_child signal handler */
#endif
      scan_for_terminated();  /* machine dependent (calls mom_get_sample()???) */

    /* if -p, must poll tasks inside jobs to look for completion */

    if (recover == JOB_RECOV_RUNNING)
      scan_non_child_tasks();

    if (recover == JOB_RECOV_DELETE)
      {
      prepare_child_tasks_for_delete();
      /* we can only do this once so set recover back to the default */
      recover = JOB_RECOV_RUNNING;
      }

    if (exiting_tasks)
      scan_for_exiting();

    check_exiting_jobs();

    TMOMScanForStarting();

    /* unblock signals */

    if (sigprocmask(SIG_UNBLOCK, &allsigs, NULL) == -1)
      log_err(errno, __func__, "sigprocmask(UNBLOCK)");

    time_now = time((time_t *)0);

    tmpTime = MIN(wait_time, time_now - (LastServerUpdateTime + ServerStatUpdateInterval));

    tmpTime = MIN(tmpTime, time_now - (last_poll_time + CheckPollTime));

    tmpTime = MAX(1, tmpTime);

    if (LastServerUpdateTime == 0)
      tmpTime = 1;

    resend_things();

    /* wait_request does a select and then calls the connection's cn_func for sockets with data */

    if (wait_request(tmpTime, NULL) != 0)
      {
      if (errno == EBADF)
        {

        init_network(pbs_mom_port, mom_process_request);

        init_network(pbs_rm_port, tcp_request);
        }

      log_err(-1, msg_daemonname, "wait_request failed");
      }

    /* block signals while we do things */

    if (sigprocmask(SIG_BLOCK, &allsigs, NULL) == -1)
      log_err(errno, __func__, "sigprocmask(BLOCK)");


    if ((pjob = (job *)GET_NEXT(svr_alljobs)) == NULL)
      {
      MOMCheckRestart();  /* There are no jobs, see if the server needs to be restarted. */
      }
    }      /* END while (mom_run_state == MOM_RUN_STATE_RUNNING) */

  return;
  }        /* END main_loop() */





/*
 * restart_mom
 */

void restart_mom(

  int   argc,
  char *argv[])

  {
  char *envstr;

  envstr = calloc(1, (strlen("PATH") + strlen(orig_path) + 2) * sizeof(char));

  if (!envstr)
    {
    sprintf(log_buffer, "calloc failed prior to execing myself: %s (%d)",
            strerror(errno),
            errno);

    log_err(errno, __func__, log_buffer);

    return;
    }

  strcpy(envstr, "PATH=");
  strcat(envstr, orig_path);
  putenv(envstr);

  DBPRT(("Re-execing myself now...\n"));

  execvp(MOMExePath, argv);

  sprintf(log_buffer, "execing myself failed: %s (%d)",
          strerror(errno),
          errno);

  log_err(errno, __func__, log_buffer);

  return;
  }  /* END restart_mom() */




#ifdef NUMA_SUPPORT
/*
 * Parses mom.layout for layout of nodeboards.
 * Initializes nodeset of each nodeboard.
 * num_node_boards is set to the number of nodeboards.
 * Return 0 on success, other on error.
 */
int read_layout_file()

  {
  FILE          *layout;
  char           line[MAX_LINE];
  char          *delims = " \t\n\r=";
  char          *tok = NULL;
  char          *val = NULL;
  int            i = 0;
  int            empty_line;
  hwloc_bitmap_t nodeset = NULL;
  
  if ((layout = fopen(path_layout, "r")) == NULL)
    {
    snprintf(log_buffer,sizeof(log_buffer),
      "Unable to read the layout file in %s\n",
      path_layout);
    log_err(errno, __func__, log_buffer);

    return(NO_LAYOUT_FILE);
    }

  /* parse lines. format is:
   * nodes=<X>  
   * extra key=value pairs are ignored by TORQUE but don't cause a failure */
  while (fgets(line, sizeof(line), layout) != NULL)
    {
    empty_line = TRUE;

    /* Strip off comments */
    if ((tok = strchr(line, '#')) != NULL)
      *tok = '\0';

    tok = strtok(line,delims);

    /* find the specifications */
    while (tok != NULL)
      {
      /* do general error checking on each pair, should be in 
       * the format name=val, spacing optional */
      val = strtok(NULL,delims);

      if (val == NULL)
        {
        snprintf(log_buffer,sizeof(log_buffer),
          "Malformed mom.layout file, line:\n%s\n",
          line);
        goto failure;
        }
     
      empty_line = FALSE;

      if (strcmp(tok,"nodes") == 0)
        {
        /* Allocate temp nodeset */
        if ((nodeset = hwloc_bitmap_alloc()) == NULL)
          {
          log_err(errno, __func__, "failed to allocate nodeset");
          return(-1);
          }

        /* Parse val into nodeset, abort if parsing fails */
        if ((node_boards[i].num_nodes = hwloc_bitmap_parselist(val, nodeset)) < 0)
          {
          sprintf(log_buffer, "failed to parse mom.layout file token: nodes=%s", val);
          goto failure;
          }

        }

      else if (strcmp(tok,"memsize") == 0)
        {
        node_boards[i].memsize = atoi(val);
        /* default to KB, keeping with TORQUE tradition */
        if ((strstr(val,"gb")) ||
            (strstr(val,"GB")))
          {
          node_boards[i].memsize *= 1024 * 1024;
          }
        else if ((strstr(val,"mb")) ||
                 (strstr(val,"MB")))
          {
          node_boards[i].memsize *= 1024;
          }
        }

      else
        {
        /* ignore other stuff for now */
        }

      /* advance token */
      tok = strtok(NULL,delims);
      } /* END while (parsing line) */

    if (empty_line == FALSE)
      {
      /* Check if we have a nodeset, abort if not */
      if (nodeset != NULL)
        {
        /* Store nodeset of node_boards[i], may be empty */
        if ((node_boards[i].nodeset = hwloc_bitmap_dup(nodeset)) == NULL)
          {
          log_err(errno, __func__, "failed to duplicate nodeset");      
          return(-1);
          }

        /* Free temp nodeset */
        hwloc_bitmap_free(nodeset);
        nodeset = NULL;

        /* Show what we have */
        sprintf(log_buffer, "nodeboard %2d: %d NUMA nodes: ", i, node_boards[i].num_nodes);
        hwloc_bitmap_displaylist(log_buffer + strlen(log_buffer),
                                 sizeof(log_buffer) - strlen(log_buffer),
                                 node_boards[i].nodeset);
        log_event(PBSEVENT_SYSTEM, PBS_EVENTCLASS_NODE, __func__, log_buffer);
        }
      else
        {
        sprintf(log_buffer, "nodeboard %d has no nodeset", i);
        goto failure;
        }

      /* Done with this nodeboard */
      i++;
      }
    } /* END while (parsing file) */

  fclose(layout);

  num_node_boards = i;

  snprintf(log_buffer,sizeof(log_buffer),
    "Setting up this mom to function as %d numa nodes\n",
    num_node_boards);

  log_event(PBSEVENT_SYSTEM, PBS_EVENTCLASS_NODE, __func__, log_buffer);

  return(PBSE_NONE);

failure:
 
  log_err(-1, __func__, log_buffer);

  if (nodeset)
    hwloc_bitmap_free(nodeset);

  return(BAD_LAYOUT_FILE);
  } /* END read_layout_file() */






/* 
 * Parses mom.layout, registers procs/mem. Dies if there's a problem.
 * Currently we allow nodeset and/or cpuset to be empty.
 * We also do not check if nodesets/cpusets of nodeboards overlap.
 *
 * parses mom.layout, registers procs/mem
 * @return nonzero if there's a problem
 */
int setup_nodeboards()

  {
  hwloc_obj_t     obj;
  hwloc_obj_t     pu;
  int             i;
  int             j;
  int             k;
  int             rc;
  int             mempath_len = strlen("/sys/devices/system/node/node999999/meminfo");

  /* Read mom.layout, init nodesets */
  if ((rc = read_layout_file()) != PBSE_NONE)
    {
    log_err(-1, __func__, "Could not read layout file!\n");
    
    exit(rc);
    } 
  
  /*
   * Walk through nodeboards
   * - set up cpuset and nodeset
   * - set up meminfo
   */
  for (i = 0; i < num_node_boards; i++)
    {
    /* Allocate cpuset for this nodeboard */
    if ((node_boards[i].cpuset = hwloc_bitmap_alloc()) == NULL)
      {
      log_err(errno, __func__, "failed to allocate cpuset");

      exit(-1);
      }

    /* Derive cpuset from nodeset */
    hwloc_cpuset_from_nodeset_strict(topology, node_boards[i].cpuset, node_boards[i].nodeset);
    
    /*
     * Handle SMT CPUs.
     * If a system has SMT enabled, there are more than one logical CPU per physical core.
     * If MOMConfigUseSMT is off, we only want the first logical CPU of a core in the cpuset.
     * Thus we map the additional logical CPUs out of the cpuset.
     * To be portable among architectures as much as possible, the only assumption that
     * is made here is that the CPUs to become mapped out are HWLOC_OBJ_PU objects that
     * are children of a HWLOC_OBJ_CORE object.
     * If there are no HWLOC_OBJ_CORE objects in the cpuset, we cannot detect if cpuset members
     * are physical or logical. Then the cpuset is left as-is.
     */
    if (!MOMConfigUseSMT)
      {
      for (obj = hwloc_get_next_obj_inside_cpuset_by_type(topology, node_boards[i].cpuset, HWLOC_OBJ_CORE, NULL);
          obj;
          obj = hwloc_get_next_obj_inside_cpuset_by_type(topology, node_boards[i].cpuset, HWLOC_OBJ_CORE, obj))
        {
        j = 1;
        while ((pu = hwloc_get_obj_inside_cpuset_by_type(topology, obj->cpuset, HWLOC_OBJ_PU, j++)) != NULL)
          hwloc_bitmap_andnot(node_boards[i].cpuset, node_boards[i].cpuset, pu->cpuset);
        }
      }
    
    /* Number of CPUs in cpuset */
    node_boards[i].num_cpus = hwloc_bitmap_weight(node_boards[i].cpuset);

    /*
     * Convert cpuset back to nodeset.
     * This maps out NUMA nodes, which were given in mom.layout,
     * but which do not exist for whatever reason.
     */
    hwloc_cpuset_to_nodeset_strict(topology, node_boards[i].cpuset, node_boards[i].nodeset);
    node_boards[i].num_nodes = hwloc_bitmap_weight(node_boards[i].nodeset);
    
    /* Set up meminfo paths */
    if (node_boards[i].num_nodes)
      {
      if ((node_boards[i].path_meminfo = (char **)calloc(node_boards[i].num_nodes, sizeof(char *))) == NULL)
        {
        log_err(errno, __func__, "failed to allocate memory");   
        exit(-1);
        }

      k = 0;
      hwloc_bitmap_foreach_begin(j, node_boards[i].nodeset)
        {
        if ((node_boards[i].path_meminfo[k] = (char *)calloc(1, mempath_len)) == NULL)
          {
          log_err(errno, __func__, "failed to allocate memory");   
          exit(-1);
          }

        snprintf(node_boards[i].path_meminfo[k], mempath_len,
          "/sys/devices/system/node/node%d/meminfo",
          j);

        k++;
        }

      hwloc_bitmap_foreach_end();
      }
    
    /* Show what we have */
    snprintf(log_buffer, sizeof(log_buffer),
      "nodeboard %2d: %d cpus (",
      i,
      node_boards[i].num_cpus);

    hwloc_bitmap_displaylist(log_buffer + strlen(log_buffer), sizeof(log_buffer) - strlen(log_buffer),
      node_boards[i].cpuset);
    snprintf(log_buffer + strlen(log_buffer), sizeof(log_buffer) - strlen(log_buffer),
      "), %d mems (",
      node_boards[i].num_nodes);

    hwloc_bitmap_displaylist(log_buffer + strlen(log_buffer), sizeof(log_buffer) - strlen(log_buffer),
      node_boards[i].nodeset);
    snprintf(log_buffer + strlen(log_buffer), sizeof(log_buffer) - strlen(log_buffer),
      ")");
    log_event(PBSEVENT_SYSTEM, PBS_EVENTCLASS_NODE, __func__, log_buffer);

    } /* END for each nodeboard */

  return(PBSE_NONE);
  } /* END setup_nodeboards() */
#endif /* ifdef NUMA_SUPPORT */




/* 
 *
 * @see main_loop() - child
 */

int main(

  int   argc,    /* I */
  char *argv[])  /* I */

  {
  int       rc;
  int       tmpFD;

  tmpFD = sysconf(_SC_OPEN_MAX);

  /* close any inherited extra files, leaving stdin, stdout, and stderr open */

  while (--tmpFD > 2)
    close(tmpFD);

  program_name = argv[0];

  initialize_globals();

  parse_command_line(argc, argv); /* Calls exit on command line error */

  if ((rc = setup_program_environment()) != 0)
    {
    return(rc);
    }

#ifdef NVIDIA_GPUS
#ifdef NVML_API
  if (!init_nvidia_nvml())
    {
    use_nvidia_gpu = FALSE;
    }
#endif  /* NVML_API */
  if (!check_nvidia_setup())
    {
    use_nvidia_gpu = FALSE;
    }

  if (!use_nvidia_gpu)
    {
    sprintf(log_buffer, "Not using Nvidia gpu support even though built with --enable-nvidia-gpus");

    log_ext(-1, "main", log_buffer, LOG_DEBUG);
    }

#endif  /* NVIDIA_GPUS */

  main_loop();

  if (mom_run_state == MOM_RUN_STATE_KILLALL)
    {
    kill_all_running_jobs();
    }

  /* shutdown mom */

#if defined(NVIDIA_GPUS) && defined(NVML_API)
  shut_nvidia_nvml();
#endif  /* NVIDIA_GPUS and NVML_API */

  mom_close_poll();

  net_close(-1);  /* close all network connections */

  if (mom_run_state == MOM_RUN_STATE_RESTART)
    {
    sprintf(log_buffer, "Will be restarting: %s",
            MOMExePath);

    log_record(PBSEVENT_SYSTEM | PBSEVENT_FORCE, PBS_EVENTCLASS_SERVER,
               msg_daemonname,
               log_buffer);
    }

  log_record(PBSEVENT_SYSTEM | PBSEVENT_FORCE, PBS_EVENTCLASS_SERVER,

             msg_daemonname,
             "Is down");

  log_close(1);

  if (mom_run_state == MOM_RUN_STATE_RESTART)
    {
    restart_mom(argc, argv);
    }

  return(0);
  }  /* END main() */




im_compose_info *create_compose_reply_info(
    
  char       *jobid,
  char       *cookie,
  hnodent    *np,
  int         command,
  tm_event_t  event,
  tm_task_id  taskid)

  {
  im_compose_info *ici = calloc(1, sizeof(im_compose_info));

  if (ici != NULL)
    {
    strcpy(ici->jobid, jobid);
    strcpy(ici->cookie, cookie);
    memcpy(&(ici->np), np, sizeof(ici->np));
    ici->command = command;
    ici->event   = event;
    ici->taskid  = taskid;
    }
  else
    log_err(ENOMEM, __func__, "Cannot allocate memory!");

  return(ici);
  } /* END create_compose_reply_info() */




int im_compose_send_info(

  struct tcp_chan *chan,
  im_compose_info *ici)

  {
  return(im_compose(chan, ici->jobid, ici->cookie, ici->command, ici->event, ici->taskid));
  } /* END im_compose_send_info() */





int resend_compose_reply(

  im_compose_info *ici)

  {
  int      ret = -1;
  hnodent *np;
  int      stream;
  char     log_buf[LOCAL_LOG_BUF_SIZE];
  struct tcp_chan *chan = NULL;

  np = &ici->np;
  stream = tcp_connect_sockaddr((struct sockaddr *)&np->sock_addr, sizeof(np->sock_addr));
  
  if (IS_VALID_STREAM(stream))
    {
    if ((chan = DIS_tcp_setup(stream)) == NULL)
      {
      }
    else if ((ret = im_compose_send_info(chan, ici)) == DIS_SUCCESS)
      {
      if ((ret = DIS_tcp_wflush(chan)) == DIS_SUCCESS)
        {
        sprintf(log_buf, "re-sent im_compose reply to %s", netaddr(&np->sock_addr));
        log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, ici->jobid, log_buf);
        free(ici);
        }
      }
    
    close(stream);
    if (chan != NULL)
      DIS_tcp_cleanup(chan);
    }

  return(ret);
  } /* END resend_compose_reply() */




int resend_kill_job_reply(

  killjob_reply_info *kj)

  {
  int      stream;
  int      ret = -1;
  hnodent *np;
  struct tcp_chan *chan = NULL;
        
  np = &kj->ici->np;
  stream = tcp_connect_sockaddr((struct sockaddr *)&np->sock_addr, sizeof(np->sock_addr));
  
  if (IS_VALID_STREAM(stream))
    {
    if ((chan = DIS_tcp_setup(stream)) == NULL)
      {
      }
    else if ((ret = im_compose_send_info(chan, kj->ici)) == DIS_SUCCESS)
      {
      if ((ret = diswul(chan, kj->cputime)) == DIS_SUCCESS)
        {
        if ((ret = diswul(chan, kj->mem)) == DIS_SUCCESS)
          {
          if ((ret = diswul(chan, kj->vmem)) == DIS_SUCCESS)
            {
            if (kj->node_id >= 0)
              ret = diswsi(chan, kj->node_id);
            
            if (ret == DIS_SUCCESS)
              {
              if ((ret = DIS_tcp_wflush(chan)) == DIS_SUCCESS)
                {
                log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, kj->ici->jobid, "Successfully re-sent kill job reply");
                free(kj->ici);
                free(kj);
                }
              }
            }
          }
        }
      }

    close(stream);
    if (chan != NULL)
      DIS_tcp_cleanup(chan);
    }

  return(ret);
  } /* END resend_kill_job_reply() */





int resend_spawn_task_reply(
    
  spawn_task_info *st)

  {
  int      ret = -1;
  hnodent *np = &st->ici->np;
  struct tcp_chan *chan = NULL;
  int      stream = tcp_connect_sockaddr((struct sockaddr *)&np->sock_addr, sizeof(np->sock_addr));

  if (IS_VALID_STREAM(stream))
    {
    if ((chan = DIS_tcp_setup(stream)) == NULL)
      {
      }
    else if ((ret = im_compose_send_info(chan, st->ici)) == DIS_SUCCESS)
      {
      if ((ret = diswsi(chan, st->ti_task)) == DIS_SUCCESS)
        {
        if ((ret = DIS_tcp_wflush(chan)) == DIS_SUCCESS)
          {
/*          read_tcp_reply(chan, IM_PROTOCOL, IM_PROTOCOL_VER, st->ici->command, &ret);
 *          */

          if (ret == DIS_SUCCESS)
            {
            log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, st->ici->jobid, "Successfully re-sent spawn task reply");
            free(st->ici);
            free(st);
            }
          }
        }
      }

    close(stream);
    if (chan != NULL)
      DIS_tcp_cleanup(chan);
    }

  return(ret);
  } /* END resend_spawn_task_reply() */




int resend_obit_task_reply(

  obit_task_info *ot)

  {
  int      ret = -1;
  hnodent *np = &ot->ici->np;
  struct tcp_chan *chan = NULL;
  int      stream = tcp_connect_sockaddr((struct sockaddr *)&np->sock_addr, sizeof(np->sock_addr));

  if (IS_VALID_STREAM(stream))
    {
    if ((chan = DIS_tcp_setup(stream)) == NULL)
      {
      }
    else if ((ret = im_compose_send_info(chan, ot->ici)) == DIS_SUCCESS)
      {
      if ((ret = diswsi(chan, ot->ti_exitstat)) == DIS_SUCCESS)
        {
        if ((ret = DIS_tcp_wflush(chan)) == DIS_SUCCESS)
          {
/*          read_tcp_reply(chan, IM_PROTOCOL, IM_PROTOCOL_VER, IM_OBIT_TASK, &ret); */

          if (ret == DIS_SUCCESS)
            {
            log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, ot->ici->jobid, "Successfully re-sent obit task reply");
            free(ot->ici);
            free(ot);
            }
          }
        }
      }

    close(stream);
    if (chan != NULL)
      DIS_tcp_cleanup(chan);
    }
  
  return(ret);
  } /* END resend_obit_task_reply() */





void resend_things()

  {
  int                 iter = -1;
  int                 ret;
  resend_momcomm     *mc;
  im_compose_info    *ici;
  killjob_reply_info *kj;
  spawn_task_info    *st;

  while ((mc = (resend_momcomm *)next_thing(things_to_resend, &iter)) != NULL)
    {
    ret = -1;
    mc->resend_attempts += 1;

    switch (mc->mc_type)
      {
      case COMPOSE_REPLY:

        ici = (im_compose_info *)mc->mc_struct;
        ret = resend_compose_reply(ici);

        break;

      case KILLJOB_REPLY:

        kj = (killjob_reply_info *)mc->mc_struct;
        ret = resend_kill_job_reply(kj);

        break;

      case SPAWN_TASK_REPLY:

        st = (spawn_task_info *)mc->mc_struct;
        ret = resend_spawn_task_reply(st);

        break;

      case OBIT_TASK_REPLY:

        ret = resend_obit_task_reply((obit_task_info *)mc->mc_struct);

        break;

      default:

        snprintf(log_buffer, sizeof(log_buffer), 
          "I don't recognize send mom communication of type %d",
          mc->mc_type);
        log_err(-1, __func__, log_buffer);

        /* remove the garbage */
        ret = DIS_SUCCESS;

        break;
      }

    if ((ret == DIS_SUCCESS) ||
        (mc->resend_attempts > 3))
      {
      remove_thing(things_to_resend, mc);
      free(mc);
      }
    } /* END for each resendable thing */
  } /* END resend_things() */



int add_to_resend_things(

  resend_momcomm *mc)

  {
  return(insert_thing(things_to_resend, mc));
  } /* END add_to_resend_things() */



/* END mom_main.c */


