/*
*         OpenPBS (Portable Batch System) v2.3 Software License
*
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
*
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
*
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
*
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
*
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
*
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
*
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
*
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
*
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
*
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
*
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
*
* 7. DISCLAIMER OF WARRANTY
*
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
*
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_mail.c - send mail to mail list or owner of job on
 * job begin, job end, and/or job abort
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include "pbs_ifl.h"
#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "list_link.h"
#include "attribute.h"
#include "server_limits.h"
#include "pbs_job.h"
#include "log.h"
#include "../lib/Liblog/pbs_log.h"
#include "../lib/Liblog/log_event.h"
#include "server.h"
#include "utils.h"
#include "threadpool.h"
#include "svr_func.h" /* get_svr_attr_* */

/* External Functions Called */

extern void net_close (int);
extern void svr_format_job (FILE *, mail_info *, char *);

/* Global Data */

extern struct server server;

extern int LOGLEVEL;



void free_mail_info(

  mail_info *mi)

  {
  if (mi->exec_host)
    free(mi->exec_host);

  if (mi->jobname)
    free(mi->jobname);

  if (mi->text)
    free(mi->text);

  free(mi->jobid);
  free(mi->mailto);

  free(mi);
  } /* END free_mail_info() */





void *send_the_mail(

  void *vp)

  {
  mail_info *mi = (mail_info *)vp;

  int        i;
  char      *mailfrom = NULL;
  char      *subjectfmt = NULL;
  char      *bodyfmt = NULL;
  char      *cmdbuf = NULL;
  char       bodyfmtbuf[MAXLINE];
  FILE      *outmail;
  
  /* Who is mail from, if SRV_ATR_mailfrom not set use default */
  get_svr_attr_str(SRV_ATR_mailfrom, &mailfrom);
  if (mailfrom == NULL)
    {
    if (LOGLEVEL >= 5)
      {
      char tmpBuf[LOG_BUF_SIZE];

      snprintf(tmpBuf,sizeof(tmpBuf),
        "Updated mailto from user list: '%s'\n",
        mi->mailto);
      log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        mi->jobid,
        tmpBuf);
      }

    mailfrom = PBS_DEFAULT_MAIL;
    }

  /* mail subject line formating statement */
  get_svr_attr_str(SRV_ATR_MailSubjectFmt, &subjectfmt);
  if (subjectfmt == NULL)
    {
    subjectfmt = "PBS JOB %i";
    }

  /* mail body formating statement */
  get_svr_attr_str(SRV_ATR_MailBodyFmt, &bodyfmt);
  if (bodyfmt == NULL)
    {
    bodyfmt =  strcpy(bodyfmtbuf, "PBS Job Id: %i\n"
                                  "Job Name:   %j\n");
    if (mi->exec_host != NULL)
      {
      strcat(bodyfmt, "Exec host:  %h\n");
      }

    strcat(bodyfmt, "%m\n");

    if (mi->text != NULL)
      {
      strcat(bodyfmt, "%d\n");
      }
    }

  /* setup sendmail command line with -f from_whom */
  i = strlen(SENDMAIL_CMD) + strlen(mailfrom) + strlen(mi->mailto) + 6;

  if ((cmdbuf = calloc(1, i + 1)) == NULL)
    {
    char tmpBuf[LOG_BUF_SIZE];

    snprintf(tmpBuf,sizeof(tmpBuf),
      "Unable to popen() command '%s' for writing: '%s' (error %d)\n",
      SENDMAIL_CMD,
      strerror(errno),
      errno);
    log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      mi->jobid,
      tmpBuf);
  
    free_mail_info(mi);

    return(NULL);
    }

  sprintf(cmdbuf, "%s -f %s %s",
    SENDMAIL_CMD,
    mailfrom,
    mi->mailto);

  outmail = popen(cmdbuf, "w");

  if (outmail == NULL)
    {
    char tmpBuf[LOG_BUF_SIZE];

    snprintf(tmpBuf,sizeof(tmpBuf),
      "Unable to popen() command '%s' for writing: '%s' (error %d)\n",
      cmdbuf,
      strerror(errno),
      errno);
    log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      mi->jobid,
      tmpBuf);

    free_mail_info(mi);
    free(cmdbuf);

    return(NULL);
    }

  /* Pipe in mail headers: To: and Subject: */
  fprintf(outmail, "To: %s\n", mi->mailto);

  fprintf(outmail, "Subject: ");
  svr_format_job(outmail, mi, subjectfmt);
  fprintf(outmail, "\n");

  /* Set "Precedence: bulk" to avoid vacation messages, etc */
  fprintf(outmail, "Precedence: bulk\n\n");

  /* Now pipe in the email body */
  svr_format_job(outmail, mi, bodyfmt);

  errno = 0;
  if ((i = pclose(outmail)) != 0)
    {
    char tmpBuf[LOG_BUF_SIZE];

    snprintf(tmpBuf,sizeof(tmpBuf),
      "Email '%c' to %s failed: Child process '%s' %s %d (errno %d:%s)\n",
      mi->mail_point,
      mi->mailto,
      cmdbuf,
      ((WIFEXITED(i)) ? ("returned") : ((WIFSIGNALED(i)) ? ("killed by signal") : ("croaked"))),
      ((WIFEXITED(i)) ? (WEXITSTATUS(i)) : ((WIFSIGNALED(i)) ? (WTERMSIG(i)) : (i))),
      errno,
      strerror(errno));
    log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      mi->jobid,
      tmpBuf);
    }
  else if (LOGLEVEL >= 4)
    {
    log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      mi->jobid,
      "Email sent successfully\n");
    }

  free_mail_info(mi);
  free(cmdbuf);
    
  return(NULL);
  } /* END send_the_mail() */




void svr_mailowner(

  job   *pjob,      /* I */
  int    mailpoint, /* note, single character  */
  int    force,     /* if set to MAIL_FORCE, force mail delivery */
  char  *text)      /* (optional) additional message text */

  {
  static char          *memory_err = "Cannot allocate memory to send email";

  char                  mailto[1024];
  char                 *domain = NULL;
  int                   i;
  mail_info            *mi;
  long                  no_force = FALSE;

  struct array_strings *pas;
  memset(mailto, 0, sizeof(mailto));

  get_svr_attr_str(SRV_ATR_MailDomain, &domain);
  if ((domain != NULL) &&
      (!strcasecmp("never", domain)))
    {
    /* never send user mail under any conditions */
    if (LOGLEVEL >= 3) 
      {
      log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        pjob->ji_qs.ji_jobid,
        "Not sending email: Mail domain set to 'never'\n");
      }

    return;
    }

  if (LOGLEVEL >= 3)
    {
    char tmpBuf[LOG_BUF_SIZE];

    snprintf(tmpBuf, LOG_BUF_SIZE, "preparing to send '%c' mail for job %s to %s (%.64s)\n",
             (char)mailpoint,
             pjob->ji_qs.ji_jobid,
             pjob->ji_wattr[JOB_ATR_job_owner].at_val.at_str,
             (text != NULL) ? text : "---");

    log_event(
      PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      pjob->ji_qs.ji_jobid,
      tmpBuf);
    }

  /*
   * if force is true, force the mail out regardless of mailpoint
   * unless server no_mail_force attribute is set to true
   */
  get_svr_attr_l(SRV_ATR_NoMailForce, &no_force);

  if ((force != MAIL_FORCE) ||
      (no_force == TRUE))
    {

    if (pjob->ji_wattr[JOB_ATR_mailpnts].at_flags & ATR_VFLAG_SET)
      {
      if (*(pjob->ji_wattr[JOB_ATR_mailpnts].at_val.at_str) ==  MAIL_NONE)
        {
        /* do not send mail. No mail requested on job */
        log_event(PBSEVENT_JOB,
                  PBS_EVENTCLASS_JOB,
                  pjob->ji_qs.ji_jobid,
                  "Not sending email: job requested no e-mail");
        return;
        }
      /* see if user specified mail of this type */
      if (strchr(
            pjob->ji_wattr[JOB_ATR_mailpnts].at_val.at_str,
            mailpoint) == NULL)
        {
        /* do not send mail */
        log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          "Not sending email: User does not want mail of this type.\n");

        return;
        }
      }
    else if (mailpoint != MAIL_ABORT) /* not set, default to abort */
      {
      log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        pjob->ji_qs.ji_jobid,
        "Not sending email: Default mailpoint does not include this type.\n");

      return;
      }
    }

  mi = calloc(1, sizeof(mail_info));

  if (mi == NULL)
    {
    log_err(ENOMEM, __func__, memory_err);
    return;
    }

  /* Who does the mail go to?  If mail-list, them; else owner */
  mailto[0] = '\0';

  if (pjob->ji_wattr[JOB_ATR_mailuser].at_flags & ATR_VFLAG_SET)
    {
    /* has mail user list, send to them rather than owner */

    pas = pjob->ji_wattr[JOB_ATR_mailuser].at_val.at_arst;

    if (pas != NULL)
      {
      for (i = 0;i < pas->as_usedptr;i++)
        {
        if ((strlen(mailto) + strlen(pas->as_string[i]) + 2) < sizeof(mailto))
          {
          strcat(mailto, pas->as_string[i]);
          strcat(mailto, " ");
          }
        }
      }
    }
  else
    {
    /* no mail user list, just send to owner */

    if (domain != NULL)
      {
      snprintf(mailto, sizeof(mailto), "%s@%s",
        pjob->ji_wattr[JOB_ATR_euser].at_val.at_str, domain);

      if (LOGLEVEL >= 5) 
        {
        char tmpBuf[LOG_BUF_SIZE];

        snprintf(tmpBuf,sizeof(tmpBuf),
          "Updated mailto from job owner and mail domain: '%s'\n",
          mailto);
        log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          tmpBuf);
        }
      }
    else
      {
#ifdef TMAILDOMAIN
      snprintf(mailto, sizeof(mailto), "%s@%s",
        pjob->ji_wattr[JOB_ATR_euser].at_val.at_str, TMAILDOMAIN);
#else /* TMAILDOMAIN */
      snprintf(mailto, sizeof(mailto), "%s", pjob->ji_wattr[JOB_ATR_job_owner].at_val.at_str);
#endif /* TMAILDOMAIN */

      if (LOGLEVEL >= 5)
        {
        char tmpBuf[LOG_BUF_SIZE];

        snprintf(tmpBuf,sizeof(tmpBuf),
          "Updated mailto from job owner: '%s'\n",
          mailto);
        log_event(PBSEVENT_ERROR | PBSEVENT_ADMIN | PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          pjob->ji_qs.ji_jobid,
          tmpBuf);
        }
      }
    }

  /* initialize the mail information */

  if ((mi->mailto = strdup(mailto)) == NULL)
    {
    log_err(ENOMEM, __func__, memory_err);
    return;
    }

  mi->mail_point = mailpoint;

  if (pjob->ji_wattr[JOB_ATR_exec_host].at_val.at_str != NULL)
    {
    mi->exec_host = strdup(pjob->ji_wattr[JOB_ATR_exec_host].at_val.at_str);

    if (mi->exec_host == NULL)
      {
      log_err(ENOMEM, __func__, memory_err);
      return;
      }
    }
  else
    mi->exec_host = NULL;

  if ((mi->jobid = strdup(pjob->ji_qs.ji_jobid)) == NULL)
    {
    log_err(ENOMEM, __func__, memory_err);
    return;
    }

  if (pjob->ji_wattr[JOB_ATR_jobname].at_val.at_str != NULL)
    {
    mi->jobname = strdup(pjob->ji_wattr[JOB_ATR_jobname].at_val.at_str);

    if (mi->jobname == NULL)
      {
      log_err(ENOMEM, __func__, memory_err);
      return;
      }
    }
  else
    mi->jobname = NULL;

  if (text)
    {
    if ((mi->text = strdup(text)) == NULL)
      {
      log_err(ENOMEM, __func__, memory_err);
      return;
      }
    }
  else
    mi->text = NULL;

  /* have a thread do the work of sending the mail */
  enqueue_threadpool_request(send_the_mail,mi);

  return;
  }  /* END svr_mailowner() */

/* END svr_mail.c */
