/*
*         OpenPBS (Portable Batch System) v2.3 Software License
* 
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
* 
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
* 
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
* 
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
* 
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
* 
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
* 
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
* 
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
* 
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
* 
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
* 
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information 
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
* 
* 7. DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * req_quejob.c 
 *
 * Functions relating to the Queue Job Batch Request sequence, including
 * Queue Job, Job Script, Ready to Commit, and Commit.
 *
 * Included functions are:
 *	req_quejob()
 *	req_jobcredential()
 *	req_jobscript()
 *	req_rdycommit()
 *	req_commit()
 */

/* #define __TNW */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#include <errno.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include "libpbs.h"
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "queue.h"
#include "net_connect.h"
#include "pbs_error.h"
#include "log.h"
#include "svrfunc.h"

#ifdef PBS_MOM
#include <pwd.h>
#include "mom_func.h"
#endif	/* PBS_MOM */

/* External Functions Called: */

extern int  reply_jid A_((char *jobid));
extern void start_exec A_((job *));
extern int  svr_authorize_jobreq A_((struct batch_request *,job *));
extern int  svr_chkque A_((job *,pbs_queue *,char *,int));
extern int  job_route A_((job *));

/* Global Data Items: */

#ifndef PBS_MOM
extern char *path_spool;
extern struct server server;
extern char  server_name[];
extern int   queue_rank;
#endif	/* !PBS_MOM */

extern const char *PJobSubState[];

/* sync w/enum job_file TJobFileType[]) */

const char *TJobFileType[] = {
  "jobscript",
  "stdin",
  "stdout",
  "stderr",
  "ckpt",
  NULL };

extern int	 resc_access_perm;
extern list_head svr_alljobs;
extern list_head svr_newjobs;
extern attribute_def job_attr_def[];
extern char *path_jobs;
extern char *pbs_o_host;
extern char *msg_script_open;
extern char *msg_script_write;
extern char *msg_init_abt;

extern char *msg_jobnew;
extern time_t time_now;
extern int    LOGLEVEL;

extern  char *msg_daemonname;



/* Private Functions in this file */

static job *locate_new_job A_((int sock, char *jobid));

#ifdef PNOT
static int user_account_verify A_((char *arguser, char* argaccount));
static char *user_account_default A_((char *arguser));
static int user_account_read_user A_((char *arguser));
#endif /* PNOT */

#ifndef PBS_MOM
static char *pbs_o_que = "PBS_O_QUEUE=";
#endif




/*
 * req_quejob - Queue Job Batch Request processing routine
 */

void req_quejob(

  struct batch_request *preq)	/* ptr to the decoded request   */

  {
  char          *id = "req_quejob";

  char		 basename[PBS_JOBBASE + 1];
  int		 created_here = 0;
  int		 index;
  char		*jid;
  char		 namebuf[MAXPATHLEN + 1];
  attribute_def	*pdef;
  job		*pj;
  svrattrl	*psatl;
  int		 rc;
  int		 sock = preq->rq_conn;

#ifndef PBS_MOM
  int		 i;
  char		 buf[256];
  int		 fds;
  char		 jidbuf[PBS_MAXSVRJOBID + 1];
  char		*pc;
  pbs_queue	*pque;
  char		*qname;
  attribute	 tempattr;
#endif /* server */

  /* set basic (user) level access permission */

  resc_access_perm = ATR_DFLAG_USWR|ATR_DFLAG_Creat;

#ifndef PBS_MOM		/* server */

  /*
   * if the job id is supplied, the request had better be 
   * from another server
   */

  if (preq->rq_fromsvr) 
    {
    /* from another server - accept the extra attributes */

    resc_access_perm |= ATR_DFLAG_MGWR|ATR_DFLAG_SvWR;

    jid = preq->rq_ind.rq_queuejob.rq_jid;
    } 
  else if (preq->rq_ind.rq_queuejob.rq_jid[0] != '\0') 
    {
    /* FAILURE */

    /* a job id is not allowed from a client */

    log_err(errno,id,"job id not allowed from client");

    req_reject(PBSE_IVALREQ,0,preq,NULL,"job not allowed from client");

    return;
    } 
  else 
    {
    /* assign it a job id */

    created_here = JOB_SVFLG_HERE;

    sprintf(jidbuf,"%d.",
      server.sv_qs.sv_jobidnumber);

    strcat(jidbuf,server_name);

    jid = jidbuf;
		
    /* having updated sv_jobidnumber, must save server struct */

    if (++server.sv_qs.sv_jobidnumber > PBS_SEQNUMTOP)
      server.sv_qs.sv_jobidnumber = 0;	/* wrap it */

    if (svr_save(&server,SVR_SAVE_QUICK)) 
      {
      /* FAILURE */

      req_reject(PBSE_INTERNAL,0,preq,NULL,NULL);

      return;
      }
    }

#else /* PBS_MOM */

  if (preq->rq_fromsvr) 
    {		
    /* from another server - accept the extra attributes */

    resc_access_perm |= ATR_DFLAG_MGWR|ATR_DFLAG_SvWR|ATR_DFLAG_MOM;

    jid = preq->rq_ind.rq_queuejob.rq_jid;
    } 
  else 
    {
    /* request must be from server */

    log_err(errno,id,"request not from server");

    req_reject(PBSE_IVALREQ,0,preq,NULL,"request not received from server");

    return;
    }

#endif /* PBS_MOM */

  /* does job already exist, check both old and new jobs */

  if ((pj = find_job(jid)) == NULL) 
    {
    pj = (job *)GET_NEXT(svr_newjobs);

    while (pj != NULL) 
      {
      if (!strcmp(pj->ji_qs.ji_jobid,jid))
        break;

      pj = (job *)GET_NEXT(pj->ji_alljobs);
      }
    }

#ifndef PBS_MOM	/* server */

  if (pj != NULL) 
    {
    /* server will reject queue request if job already exists */

    log_err(errno,id,"cannot queue new job, job already exists");

    req_reject(PBSE_JOBEXIST,0,preq,NULL,NULL);

    return;
    }

  /* find requested queue, is it there? */

  qname = preq->rq_ind.rq_queuejob.rq_destin;  

  if ((*qname == '\0') || (*qname == '@')) 
    {  
    /* use default queue */

    pque = get_dfltque();

    rc   = PBSE_QUENODFLT;
    } 
  else 
    { 	
    /* else find the named queue */

    pque = find_queuebyname(preq->rq_ind.rq_queuejob.rq_destin);

    rc   = PBSE_UNKQUE;
    }

  if (pque == NULL) 
    {
    /* FAILURE */

    log_err(errno,id,"requested queue not found");

    req_reject(rc,0,preq,NULL,"cannot locate queue"); /* not there   */

    return;
    }

  /*
   * make up job file name, it is based on the jobid, however the
   * minimun file name is only 14 character in POSIX, so we may have
   * to "hash" the name slightly
   */

  strncpy(basename,jid,PBS_JOBBASE);
  basename[PBS_JOBBASE] = '\0';

  do {
    strcpy(namebuf,path_jobs);
    strcat(namebuf,basename);
    strcat(namebuf,JOB_FILE_SUFFIX);

    fds = open(namebuf,O_CREAT|O_EXCL|O_WRONLY,0600);

    if (fds < 0) 
      {
      if (errno == EEXIST) 
        {
        pc = basename + strlen(basename) - 1;

        while (!isprint((int)*pc)) 
          {
          pc--;

          if (pc <= basename) 
            {
            /* FAILURE */

            log_err(errno,id,"job file is corrupt");

            req_reject(PBSE_INTERNAL,0,preq,NULL,"job file is corrupt");

            return;
            }
          }

        (*pc)++;
        } 
      else 
        {
        /* FAILURE */

        log_err(errno,id,"cannot create job file");

        req_reject(PBSE_SYSTEM,0,preq,NULL,"cannot open new job file");

        return;
        }
      }
    } while (fds < 0);

  close(fds);

  /* create the job structure */

  if ((pj = job_alloc()) == NULL) 
    {
    /* FAILURE */

    log_err(errno,id,"cannot alloc new job");

    unlink(namebuf);

    req_reject(PBSE_SYSTEM,0,preq,NULL,"cannot alloc new job structure");

    return;
    }

#else /* PBS_MOM */

  /*
   * New job ...
   *
   * for MOM - rather than make up a hashname, we use the name sent
   * to us by the server as an attribute.
   */

  psatl = (svrattrl *)GET_NEXT(preq->rq_ind.rq_queuejob.rq_attr);

  while (psatl != NULL) 
    {
    if (!strcmp(psatl->al_name,ATTR_hashname)) 
      {
      strcpy(basename,psatl->al_value);

      break;
      }

    psatl = (svrattrl *)GET_NEXT(psatl->al_link);
    }

  if (pj != NULL) 
    {
    /* newly queued job already exists */

    if (pj->ji_qs.ji_substate == JOB_SUBSTATE_RUNNING) 
      {
      /* FAILURE - job exists and is running */

      log_err(errno,id,"cannot queue new job, job exists and is running");

      req_reject(PBSE_JOBEXIST,0,preq,NULL,"job is running");

      return;
      }

    /* if checkpointed, then keep old and skip rest of process */

    if (pj->ji_qs.ji_svrflags & JOB_SVFLG_CHKPT) 
      {
      pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSIN;

      if (reply_jobid(preq,pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_Queue) == 0) 
        {
        delete_link(&pj->ji_alljobs);

        append_link(&svr_newjobs,&pj->ji_alljobs,pj);

        pj->ji_qs.ji_un_type = JOB_UNION_TYPE_NEW;
        pj->ji_qs.ji_un.ji_newt.ji_fromsock = sock;
        pj->ji_qs.ji_un.ji_newt.ji_fromaddr = get_connectaddr(sock);
        pj->ji_qs.ji_un.ji_newt.ji_scriptsz = 0;
        } 
      else 
        {
        close_conn(sock);
        }

      return;
      }  /* END if (pj->ji_qs.ji_svrflags & JOB_SVFLG_CHKPT) */

    /* unlink job from svr_alljobs since it will be placed on newjobs */

    delete_link(&pj->ji_alljobs);
    }  /* END if (pj != NULL) */ 
  else 
    {
    /* if not already here, allocate job struct */

    if ((pj = job_alloc()) == NULL) 
      {
      /* FAILURE */

      req_reject(PBSE_SYSTEM,0,preq,NULL,"cannot allocate new job structure");

      return;
      }

    strcpy(namebuf,path_jobs);      /* job directory path */
    strcat(namebuf,basename);
    strcat(namebuf,JOB_TASKDIR_SUFFIX);

    if ((mkdir(namebuf,0700) == -1) && (errno != EEXIST))
      {
      /* FAILURE */

      char tmpLine[1024];

      sprintf(tmpLine,"cannot create directory '%s'",
        namebuf);

      log_err(errno,tmpLine,msg_init_abt);

      job_purge(pj);

      req_reject(PBSE_SYSTEM,0,preq,NULL,tmpLine);

      return;
      }
    }    /* END else (pj != NULL) */

#endif  /* PBS_MOM */

  strcpy(pj->ji_qs.ji_jobid,jid);
  strcpy(pj->ji_qs.ji_fileprefix,basename);

  pj->ji_modified       = 1;
  pj->ji_qs.ji_svrflags = created_here;
  pj->ji_qs.ji_un_type  = JOB_UNION_TYPE_NEW;

  /* decode attributes from request into job structure */

  psatl = (svrattrl *)GET_NEXT(preq->rq_ind.rq_queuejob.rq_attr);
 
  while (psatl != NULL) 
    {
    /* identify the attribute by name */

    index = find_attr(job_attr_def,psatl->al_name,JOB_ATR_LAST);

    if (index < 0)
      {
      /* FAILURE */

      /* didn`t recognize the name */

#ifndef PBS_MOM
      index = JOB_ATR_UNKN;	/* keep as "unknown" for now */
#else	/* is PBS_MOM */
      /* FAILURE */

      job_purge(pj);   /* CRI - 12/20/2004 */

      reply_badattr(PBSE_NOATTR,1,psatl,preq);

      return;
#endif	/* PBS_MOM */
      }

    pdef = &job_attr_def[index];

    /* Is attribute not writeable by manager or by a server? */

    if ((pdef->at_flags & resc_access_perm) == 0) 
      {
      /* FAILURE */

      job_purge(pj);

      reply_badattr(PBSE_ATTRRO,1,psatl,preq);

      return;
      }

    /* decode attribute */

    rc = pdef->at_decode(
      &pj->ji_wattr[index],
      psatl->al_name, 
      psatl->al_resc, 
      psatl->al_value);

#ifndef PBS_MOM

    if (rc != 0) 
      {
      if (rc == PBSE_UNKRESC) 
        {
        /* check for RM extension */

        /* NYI */

        /* unknown resources not allowed in Exec queue */

        if (pque->qu_qs.qu_type == QTYPE_Execution) 
          {
          /* FAILURE */

          job_purge(pj);

          reply_badattr(rc,1,psatl,preq);

          return;
          }
        } 
      else 
        {
        /* FAILURE */

        /* any other error is fatal */

        job_purge(pj);

        reply_badattr(rc,1,psatl,preq);

        return;
        }
      }

#else	/* PBS_MOM */

    if (rc != 0) 
      {
      /* FAILURE */

      /* all errors are fatal for MOM */

      job_purge(pj);

      reply_badattr(rc,1,psatl,preq);

      return;
      }

    if (psatl->al_op == DFLT) 
      {
      if (psatl->al_resc) 
        {
        resource     *presc;
        resource_def *prdef;

        prdef = find_resc_def(svr_resc_def,psatl->al_resc,svr_resc_size);

        if (prdef == NULL) 
          {
          job_purge(pj);

          reply_badattr(rc,1,psatl,preq);

          return;
          }

        presc = find_resc_entry(&pj->ji_wattr[index],prdef);

        if (presc != NULL)
          presc->rs_value.at_flags |= ATR_VFLAG_DEFLT;
        } 
      else 
        {
        pj->ji_wattr[index].at_flags |= ATR_VFLAG_DEFLT;
        }
      }    /* END if (psatl->al_op == DFLT) */
#endif     /* PBS_MOM */

    psatl = (svrattrl *)GET_NEXT(psatl->al_link);
    }      /* END while (psatl != NULL) */

#ifndef PBS_MOM

  /* perform any at_action routine declared for the attributes */

  for (i = 0;i < JOB_ATR_LAST;++i) 
    {
    pdef = &job_attr_def[i];

    if ((pj->ji_wattr[i].at_flags & ATR_VFLAG_SET) && (pdef->at_action)) 
      {
      rc = pdef->at_action(&pj->ji_wattr[i],pj,ATR_ACTION_NEW);

      if (rc) 
        {
        job_purge(pj);

        req_reject(rc,i,preq,NULL,"cannot execute attribute action");

        return;
        }
      }
    }    /* END for (i) */
	
  /*
   * Now that the attributes have been decoded, we can setup some
   * additional parameters and perform a few more checks.
   *
   * First, set some items based on who created the job...
   */

  if (created_here) 
    {	
    /* created here */

    /* check that job has a jobname */

    if ((pj->ji_wattr[(int)JOB_ATR_jobname].at_flags & ATR_VFLAG_SET) == 0) 
      {
      job_attr_def[(int)JOB_ATR_jobname].at_decode(
        &pj->ji_wattr[(int)JOB_ATR_jobname],
        NULL, 
        NULL, 
        "none");
      }

    /* check value of priority */

    if (pj->ji_wattr[(int)JOB_ATR_priority].at_flags & ATR_VFLAG_SET) 
      {
      if ((pj->ji_wattr[(int)JOB_ATR_priority].at_val.at_long < -1024) || 
          (pj->ji_wattr[(int)JOB_ATR_priority].at_val.at_long > 1024)) 
        {
        job_purge(pj);

        req_reject(PBSE_BADATVAL,0,preq,NULL,"invalid job priority");

        return;
        }
      }

    /* set job owner attribute to user@host */

    job_attr_def[(int)JOB_ATR_job_owner].at_free(
      &pj->ji_wattr[(int)JOB_ATR_job_owner]);	

    strcpy(buf,preq->rq_user);
    strcat(buf,"@");
    strcat(buf,preq->rq_host);

    job_attr_def[(int)JOB_ATR_job_owner].at_decode(
      &pj->ji_wattr[(int)JOB_ATR_job_owner],
      NULL,
      NULL,
      buf);

    /* set create time */

    pj->ji_wattr[(int)JOB_ATR_ctime].at_val.at_long =(long)time_now;
    pj->ji_wattr[(int)JOB_ATR_ctime].at_flags |= ATR_VFLAG_SET;

    /* set hop count = 1 */

    pj->ji_wattr[(int)JOB_ATR_hopcount].at_val.at_long = 1;
    pj->ji_wattr[(int)JOB_ATR_hopcount].at_flags |= ATR_VFLAG_SET;

    /* Interactive jobs are necessarily not rerunable */

    if (pj->ji_wattr[(int)JOB_ATR_interactive].at_val.at_long)
      {
      pj->ji_wattr[(int)JOB_ATR_rerunable].at_val.at_long = 0;
      pj->ji_wattr[(int)JOB_ATR_rerunable].at_flags |= ATR_VFLAG_SET;
      }

    /* need to set certain environmental variables per POSIX */

    clear_attr(&tempattr,&job_attr_def[(int)JOB_ATR_variables]);

    strcpy(buf,pbs_o_que);
    strcat(buf,pque->qu_qs.qu_name);

    if (get_variable(pj,pbs_o_host) == NULL) 
      {
      strcat(buf,",");
      strcat(buf,pbs_o_host);
      strcat(buf,"=");
      strcat(buf,preq->rq_host);
      }

    job_attr_def[(int)JOB_ATR_variables].at_decode(&tempattr,
      NULL, 
      NULL, 
      buf);

    job_attr_def[(int)JOB_ATR_variables].at_set(
      &pj->ji_wattr[(int)JOB_ATR_variables],
      &tempattr, 
      INCR);

    job_attr_def[(int)JOB_ATR_variables].at_free(&tempattr);

    /* if JOB_ATR_outpath/JOB_ATR_errpath not set, set default */

    if (!(pj->ji_wattr[(int)JOB_ATR_outpath].at_flags & ATR_VFLAG_SET)) 
      {
      pj->ji_wattr[(int)JOB_ATR_outpath].at_val.at_str =
        prefix_std_file(pj,(int)'o');

      pj->ji_wattr[(int)JOB_ATR_outpath].at_flags |= ATR_VFLAG_SET;
      }

    if (!(pj->ji_wattr[(int)JOB_ATR_errpath].at_flags & ATR_VFLAG_SET)) 
      {
      pj->ji_wattr[(int)JOB_ATR_errpath].at_val.at_str =
        prefix_std_file(pj,(int)'e');

      pj->ji_wattr[(int)JOB_ATR_errpath].at_flags |=ATR_VFLAG_SET;
      }

    if ((pj->ji_wattr[(int)JOB_ATR_outpath].at_val.at_str == NULL) ||
        (pj->ji_wattr[(int)JOB_ATR_errpath].at_val.at_str == NULL)) 
      {
      job_purge(pj);

      req_reject(PBSE_NOATTR,0,preq,NULL,"no output/error file specified");

      return;
      }

#ifdef PNOT
    /*************************************************************
     * Argonne National Laboratory Account/Project enforcement patch
     *     Dec 2000, Sep 2004, JP Navarro navarro@mcs.anl.gov
     * Verify the specified account (project) is valid for the user
     * If none is specified, assign the user's default if allowed
     * Else return error: valid user account is required
     *************************************************************/

    if (pj->ji_wattr[(int)JOB_ATR_account].at_flags & ATR_VFLAG_SET) 
      {
      /* account specified, reject if it's not valid for user */

      if (user_account_verify(
           preq->rq_user,
           pj->ji_wattr[(int)JOB_ATR_account].at_val.at_str) == 0) 
        {
        job_purge(pj);

        req_reject(PBSE_BADACCT,0,preq,NULL,"invalid account");

        return;
        }
      } 
    else 
      {
      /* account not specified, get default value */

      job_attr_def[(int)JOB_ATR_account].at_decode(
        &pj->ji_wattr[(int)JOB_ATR_account],
        NULL, 
        NULL,
        (char *)user_account_default(preq->rq_user));

      if (pj->ji_wattr[(int)JOB_ATR_account].at_val.at_str == 0) 
        {
        /* no default found */

        job_purge(pj);

        req_reject(PBSE_BADACCT,0,preq,NULL,"no default account available");

        return;
        }
      }

#endif /* PNOT */

    } 
  else 
    {		
    /* job was created elsewhere and moved here */
		
    /* make sure job_owner is set, error if not */

    if (!(pj->ji_wattr[(int)JOB_ATR_job_owner].at_flags & ATR_VFLAG_SET)) 
      {
      job_purge(pj);		

      log_err(errno,"req_quejob","job owner not set");

      req_reject(PBSE_IVALREQ,0,preq,NULL,"no job owner specified");

      return;
      }

    /* increment hop count */

    if (++pj->ji_wattr[(int)JOB_ATR_hopcount].at_val.at_long > PBS_MAX_HOPCOUNT) 
      {
      job_purge(pj);		

      req_reject(PBSE_HOPCOUNT,0,preq,NULL,"max job hop reached");

      return;
      }
    }

  /* set up at_server attribute for status */

  job_attr_def[(int)JOB_ATR_at_server].at_decode(
    &pj->ji_wattr[(int)JOB_ATR_at_server],
    NULL,
    NULL, 
    server_name);

  /*
   * See if the job is qualified to go into the requested queue.
   * Note, if an execution queue, then ji_qs.ji_un.ji_exect is set up
   *
   * svr_chkque is called way down here because it needs to have the
   * job structure and attributes already set up.
   */

  if ((rc = svr_chkque(pj,pque,preq->rq_host,MOVE_TYPE_Move))) 
    {
    job_purge(pj);

    req_reject(rc,0,preq,NULL,NULL);

    return;
    }

  strcpy(pj->ji_qs.ji_queue,pque->qu_qs.qu_name);

  pj->ji_wattr[(int)JOB_ATR_substate].at_val.at_long = JOB_SUBSTATE_TRANSIN;
  pj->ji_wattr[(int)JOB_ATR_substate].at_flags |= ATR_VFLAG_SET;

#endif  /* !PBS_MOM */

  /* set remaining job structure elements */

  pj->ji_qs.ji_state =    JOB_STATE_TRANSIT;
  pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSIN;

  pj->ji_wattr[(int)JOB_ATR_mtime].at_val.at_long = (long)time_now;
  pj->ji_wattr[(int)JOB_ATR_mtime].at_flags |= ATR_VFLAG_SET;

  pj->ji_qs.ji_un_type = JOB_UNION_TYPE_NEW;
  pj->ji_qs.ji_un.ji_newt.ji_fromsock = sock;
  pj->ji_qs.ji_un.ji_newt.ji_fromaddr = get_connectaddr(sock);
  pj->ji_qs.ji_un.ji_newt.ji_scriptsz = 0;

  /* acknowledge the request with the job id */

  if (reply_jobid(preq,pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_Queue) != 0) 
    {
    /* reply failed, purge the job and close the connection */

    close_conn(sock);

    job_purge(pj);

    return;
    }

  /* link job into server's new jobs list request  */

  append_link(&svr_newjobs,&pj->ji_alljobs,pj);

  return;
  }  /* END req_quejob() */





/*
 * req_jobcredential - receive a set of credentials to be used by the job
 *
 * THIS IS JUST A PLACE HOLDER FOR NOW
 * It does nothing but acknowledge the request 
 */

void req_jobcredential(

  struct batch_request *preq)  /* ptr to the decoded request   */

  {
  job *pj;

  pj = locate_new_job(preq->rq_conn,NULL);

  if (pj == NULL) 
    {
    req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

    return;
    }

#ifndef PBS_MOM

  if (svr_authorize_jobreq(preq, pj) == -1) 
    {
    req_reject(PBSE_PERM,0,preq,NULL,NULL);

    return;
    }

#endif	/* PBS_MOM */

  reply_ack(preq);

  return;
  }  /* END req_jobcredential() */




/*
 * req_jobscript - receive job script section
 *
 * Each section is appended to the file
 */

void req_jobscript(

  struct batch_request *preq)	/* ptr to the decoded request*/

  {
  char *id = "req_jobscript";

  int	 fds;
  char	 namebuf[MAXPATHLEN];
  job	*pj;
#ifdef PBS_MOM
  int	 filemode = 0700;
  extern char mom_host[];
#else	/* server */
  int	 filemode = 0600;
#endif

#ifdef __TNW
  int    DoRetry = 1;
#endif /* __TNW */

  errno = 0;

  pj = locate_new_job(preq->rq_conn,NULL);

  if (pj == NULL) 
    {
    log_err(errno,id,"cannot locate new job");

    req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

    return;
    }

  /* what is the difference between JOB_SUBSTATE_TRANSIN and TRANSICM? */

  if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSIN) 
    {
    if (errno == 0)
      {
      sprintf(log_buffer,"job %s in unexpected state '%s'",
        pj->ji_qs.ji_jobid,
        PJobSubState[pj->ji_qs.ji_substate]);
      }
    else
      {
      sprintf(log_buffer,"job %s in unexpected state '%s' (errno=%d - %s)",
        pj->ji_qs.ji_jobid,
        PJobSubState[pj->ji_qs.ji_substate],
        errno,
        strerror(errno));
      }

    log_err(errno,id,log_buffer);

#ifdef PBS_MOM

    /* NOTE:  state is out of sync between pbs_server and pbs_mom */
    /*        pbs_server believes job is idle, pbs_mom does not */
    /*        MOM should check local job state, and if not idle, purge the job */
    /*        w/job_purge(pj) */

    req_reject(PBSE_IVALREQ,0,preq,mom_host,log_buffer);
#else
    req_reject(PBSE_IVALREQ,0,preq,NULL,log_buffer);
#endif /* PBS_MOM */

    return;
    }

#ifndef PBS_MOM

  if (svr_authorize_jobreq(preq,pj) == -1) 
    {
    /* FAILURE */

    log_err(errno,id,"cannot authorize request");

    req_reject(PBSE_PERM,0,preq,NULL,NULL);

    return;
    }

#else

  /* mom - if job has been checkpointed, discard script,already have it */

  if (pj->ji_qs.ji_svrflags & JOB_SVFLG_CHKPT) 
    {
    /* SUCCESS - do nothing, ignore script */

    reply_ack(preq);

    return;
    }

#endif	/* PBS_MOM */

  strcpy(namebuf,path_jobs);  
  strcat(namebuf,pj->ji_qs.ji_fileprefix);
  strcat(namebuf,JOB_SCRIPT_SUFFIX);

#ifdef __TNW
retry:
#endif

  if (pj->ji_qs.ji_un.ji_newt.ji_scriptsz == 0) 
    {
    /* NOTE:  fail is job script already exists */

    fds = open(namebuf,O_WRONLY|O_CREAT|O_EXCL|O_Sync,filemode);
    } 
  else 
    {
    fds = open(namebuf,O_WRONLY|O_APPEND|O_Sync,filemode);
    }

  if (fds < 0)
    {
    char tmpLine[1024];

    snprintf(tmpLine,sizeof(tmpLine),"cannot open '%s' errno=%d - %s",
      namebuf,
      errno,
      strerror(errno));

#ifdef __TNW

    if (errno == EEXIST)
      {
      if (DoRetry == 1)
        {
        /* stale script detected - remove it and retry */

        remove(namebuf);
  
        log_err(errno,id,"duplicate script file located and removed");

        DoRetry = 0;

        goto retry;
        }
      }

#endif /* __TNW */

    /* FAILURE */

    /* NOTE: log_err may modify errno */

    log_err(errno,id,msg_script_open);

#ifdef PBS_MOM
    req_reject(PBSE_INTERNAL,0,preq,mom_host,tmpLine);
#else
    req_reject(PBSE_INTERNAL,0,preq,NULL,tmpLine);
#endif /* PBS_MOM */

    return;
    }

  if (write(
        fds, 
        preq->rq_ind.rq_jobfile.rq_data, 
        (unsigned)preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size) 
    {
    /* FAILURE */

    log_err(errno,id,msg_script_write);

#ifdef PBS_MOM
    req_reject(PBSE_INTERNAL,0,preq,mom_host,"cannot write job command file");
#else
    req_reject(PBSE_INTERNAL,0,preq,NULL,"cannot write job command file");
#endif /* PBS_MOM */

    close(fds);

    return;
    }

  close(fds);

  pj->ji_qs.ji_un.ji_newt.ji_scriptsz += preq->rq_ind.rq_jobfile.rq_size;

  /* job has a script file */

  pj->ji_qs.ji_svrflags = 
    (pj->ji_qs.ji_svrflags & ~JOB_SVFLG_CHKPT)|JOB_SVFLG_SCRIPT; 

  /* SUCCESS */

  reply_ack(preq);

  return;
  }  /* END req_jobscript() */




#ifndef PBS_MOM	
/* the following is for the server only, MOM has her own version below */

/*
 * req_mvjobfile - receive a job file
 *	This request is used to move a file associated with a job, typically
 *	the standard output or error, between a server and a server or from
 *	a mom back to a server.  For a server, the destination is always 
 *	within the spool directory.
 */

void req_mvjobfile(  /* NOTE:  routine for server only - mom code follows this routine */

  struct batch_request *preq)	/* ptr to the decoded request   */

  {
  int	 fds;
  char	 namebuf[MAXPATHLEN];
  job	*pj;

  pj = locate_new_job(preq->rq_conn,NULL);

  if (pj == NULL)
    pj = find_job(preq->rq_ind.rq_jobfile.rq_jobid);

  if ((preq->rq_fromsvr == 0) || (pj == NULL)) 
    {
    snprintf(log_buffer,1024,"cannot find job %s",
      preq->rq_ind.rq_jobfile.rq_jobid);

    log_err(errno,"req_mvjobfile",log_buffer);

    req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

    return;
    }

  strcpy(namebuf,path_spool);  

  strcat(namebuf,pj->ji_qs.ji_fileprefix);

  switch ((enum job_file)preq->rq_ind.rq_jobfile.rq_type) 
    {
    case StdOut:

      strcat(namebuf,JOB_STDOUT_SUFFIX);

      break;

    case StdErr:

      strcat(namebuf,JOB_STDERR_SUFFIX);

      break;

    case Chkpt:

      strcat(namebuf,JOB_CKPT_SUFFIX);

      break;

    default:

      log_err(errno,"req_mvjobfile","unexpected move type");

      req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

      return;

      /*NOTREACHED*/

      break;
    }

  if (preq->rq_ind.rq_jobfile.rq_sequence == 0)
    {
    unlink(namebuf);
    fds = open(namebuf,O_WRONLY|O_CREAT|O_EXCL|O_Sync,0600);
    }
  else
    fds = open(namebuf,O_WRONLY|O_APPEND|O_Sync,0600);

  if (fds < 0) 
    {
    log_err(errno,"req_mvjobfile",msg_script_open);

    req_reject(PBSE_INTERNAL,0,preq,NULL,NULL);

    return;
    }

  if (write(
       fds, 
       preq->rq_ind.rq_jobfile.rq_data, 
       (unsigned)preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size) 
    {
    log_err(errno,"req_jobfile",msg_script_write);

    req_reject(PBSE_SYSTEM,0,preq,NULL,NULL);

    close(fds);

    return;
    }

  close(fds);

  if (LOGLEVEL >= 6)
    {
    sprintf(log_buffer,"successfully moved file '%s' for job '%s'",
      namebuf,
      preq->rq_ind.rq_jobfile.rq_jobid);

    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      log_buffer);
    }

  reply_ack(preq);

  return;
  }  /* END req_mvjobfile() */

#else /* PBS_MOM */

/*
 * req_mvjobfile - move the specifled job standard files 
 *	This is MOM's version.  The files are owned by the user and placed
 *	in either the spool area or the user's home directory depending
 *	on the compile option, see std_file_name().
 */

void req_mvjobfile(  /* routine for MOM only - server routine listed above */

  struct batch_request *preq)  /* I */

  {
  int	         fds;
  enum job_file  jft;
  int	         oflag;
  job	        *pj;
  struct passwd *pwd;

  jft = (enum job_file)preq->rq_ind.rq_jobfile.rq_type;

  if (preq->rq_ind.rq_jobfile.rq_sequence == 0)
    oflag = O_CREAT | O_WRONLY | O_TRUNC;
  else
    oflag = O_CREAT | O_WRONLY | O_APPEND;

  pj = locate_new_job(preq->rq_conn,NULL);

  if (pj == NULL)
    pj = find_job(preq->rq_ind.rq_jobfile.rq_jobid);

  if (pj == NULL) 
    {
    snprintf(log_buffer,1024,"cannot find job %s for move of %s file",
      preq->rq_ind.rq_jobfile.rq_jobid,
      TJobFileType[jft]);

    log_err(-1,"req_mvjobfile",log_buffer);

    req_reject(PBSE_UNKJOBID,0,preq,NULL,NULL);

    return;
    }

  if ((pj->ji_grpcache == NULL) && (check_pwd(pj) == NULL))
    {
    req_reject(PBSE_UNKJOBID,0,preq,NULL,NULL);

    return;
    }
    
  if (((pwd = getpwnam(pj->ji_wattr[(int)JOB_ATR_euser].at_val.at_str)) == NULL) || 
      ((fds = open_std_file(pj,jft,oflag,pwd->pw_gid)) < 0)) 
    {
    req_reject(PBSE_MOMREJECT,0,preq,NULL,NULL);

    return;
    }

  if (write(
       fds,
       preq->rq_ind.rq_jobfile.rq_data, 
       preq->rq_ind.rq_jobfile.rq_size) != preq->rq_ind.rq_jobfile.rq_size)
    {
    req_reject(PBSE_SYSTEM,0,preq,NULL,NULL);
    }
  else
    {
    reply_ack(preq);	
    }

  close(fds);

  if (LOGLEVEL >= 6)
    {
    sprintf(log_buffer,"successfully moved %s file for job '%s'",
      TJobFileType[jft],
      preq->rq_ind.rq_jobfile.rq_jobid);

    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      log_buffer);
    }

  return;
  }  /* END req_mvjobfile() */


#endif /* PBS_MOM */




/*
 * req_rdytocommit - Ready To Commit Batch Request
 *
 *	Set substate to JOB_SUBSTATE_TRANSICM and
 *	record job to permanent storage, i.e. written to the job save file
 *      (called by both pbs_server and pbs_mom) 
 */

void req_rdytocommit(

  struct batch_request *preq)

  {
  job *pj;
  int  sock = preq->rq_conn;
  
  int  OrigState;
  int  OrigSState;
  char OrigSChar;
  long OrigFlags;

  pj = locate_new_job(sock,preq->rq_ind.rq_rdytocommit);

  if (LOGLEVEL >= 6)
    {
    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      "ready to commit job");
    }

  if (pj == NULL) 
    {
    log_err(errno,"req_rdytocommit","unknown job id");

    req_reject(PBSE_UNKJOBID,0,preq,NULL,NULL);

    /* FAILURE */

    return;
    }

  if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSIN) 
    {
    log_err(errno,"req_rdytocommit","cannot commit job in unexpected state");

    req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

    /* FAILURE */

    return;
    }

#ifndef PBS_MOM

  if (svr_authorize_jobreq(preq,pj) == -1) 
    {
    req_reject(PBSE_PERM,0,preq,NULL,"cannot authorize jobreq");

    /* FAILURE */

    return;
    }

#endif /* !PBS_MOM */

  OrigState  = pj->ji_qs.ji_state;
  OrigSState = pj->ji_qs.ji_substate;
  OrigSChar  = pj->ji_wattr[(int)JOB_ATR_state].at_val.at_char;
  OrigFlags  = pj->ji_wattr[(int)JOB_ATR_state].at_flags;
 
  pj->ji_qs.ji_state    = JOB_STATE_TRANSIT;
  pj->ji_qs.ji_substate = JOB_SUBSTATE_TRANSICM;
  pj->ji_wattr[(int)JOB_ATR_state].at_val.at_char = 'T';
  pj->ji_wattr[(int)JOB_ATR_state].at_flags |= ATR_VFLAG_SET;

  if (job_save(pj,SAVEJOB_NEW) == -1) 
    {
    char tmpLine[1024];

    sprintf(tmpLine,"cannot save job - errno=%d - %s",
      errno,
      strerror(errno));

    log_err(errno,"req_rdytocommit",tmpLine);

    /* commit failed, backoff state changes */

    pj->ji_qs.ji_state    = OrigState;
    pj->ji_qs.ji_substate = OrigSState;
    pj->ji_wattr[(int)JOB_ATR_state].at_val.at_char = OrigSChar;
    pj->ji_wattr[(int)JOB_ATR_state].at_flags = OrigFlags;

    req_reject(PBSE_SYSTEM,0,preq,NULL,tmpLine);

    /* FAILURE */

    return;
    }

  /* acknowledge the request with the job id */

  if (reply_jobid(preq,pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_RdytoCom) != 0) 
    {
    /* reply failed, purge the job and close the connection */

    sprintf(log_buffer,"cannot report jobid - errno=%d - %s",
      errno,
      strerror(errno));

    log_err(errno,"req_rdytocommit",log_buffer);

    close_conn(sock);

    job_purge(pj);

    /* FAILURE */

    return;
    }

  if (LOGLEVEL >= 6)
    {
    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      "ready to commit job completed");
    }

  return;
  }  /* END req_rdytocommit() */




/*
 * req_commit - commit ownership of job
 *
 *	Set state of job to JOB_STATE_QUEUED (or Held or Waiting) and
 *	enqueue the job into its destination queue.
 */

void req_commit(

  struct batch_request *preq)  /* I */

  {
  job	  *pj;

#ifndef PBS_MOM	/* SERVER only */
  int	   newstate;
  int	   newsub;
  pbs_queue *pque;
  int	   rc;
#endif /* SERVER only */

  pj = locate_new_job(preq->rq_conn,preq->rq_ind.rq_commit);

  if (LOGLEVEL >= 6)
    {
    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      "committing job");
    }

  if (pj == NULL) 
    {
    req_reject(PBSE_UNKJOBID,0,preq,NULL,NULL);

    return;
    }

  if (pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSICM) 
    {
    log_err(errno,"req_commit","cannot commit job in unexpected state");

    req_reject(PBSE_IVALREQ,0,preq,NULL,NULL);

    return;
    }

#ifdef PBS_MOM	/* MOM only */

  /* move job from new job list to "all" job list, set to running state */

  delete_link(&pj->ji_alljobs);
  append_link(&svr_alljobs,&pj->ji_alljobs,pj);

  /*
  ** Set JOB_SVFLG_HERE to indicate that this is Mother Superior.
  */

  pj->ji_qs.ji_svrflags |= JOB_SVFLG_HERE;

  pj->ji_qs.ji_state = JOB_STATE_RUNNING;
  pj->ji_qs.ji_substate = JOB_SUBSTATE_PRERUN;
  pj->ji_qs.ji_un_type = JOB_UNION_TYPE_MOM;
  pj->ji_qs.ji_un.ji_momt.ji_svraddr = get_connectaddr(preq->rq_conn);
  pj->ji_qs.ji_un.ji_momt.ji_exitstat = 0;

  /* For MOM - start up the job (blocks) */

  if (LOGLEVEL >= 6)
    {
    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      "starting job execution");
    }

  start_exec(pj);

  if (LOGLEVEL >= 6)
    {
    log_record(
      PBSEVENT_JOB,
      PBS_EVENTCLASS_JOB,
      (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
      "job execution started");
    }
	
  /* if start request fails, reply with failure string */

  if (pj->ji_qs.ji_substate == JOB_SUBSTATE_EXITING)
    {
    char tmpLine[1024];

    if ((pj->ji_hosts != NULL) && 
        (pj->ji_nodekill >= 0) && 
        (pj->ji_hosts[pj->ji_nodekill].hn_host != NULL))
      {
      sprintf(tmpLine,"start failed on node %s",
        pj->ji_hosts[pj->ji_nodekill].hn_host);
      }
    else
      {
      sprintf(tmpLine,"start failed on unknown node");
      }

    if (LOGLEVEL >= 6)
      {
      log_record(
        PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
        tmpLine);
      }

    reply_text(preq,0,tmpLine);
    }
  else
    {	
    reply_jobid(preq,pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_Commit);
    }

  job_save(pj,SAVEJOB_FULL);

  /* NOTE: we used to flag JOB_ATR_errpath, JOB_ATR_outpath,
   * JOB_ATR_session_id, and JOB_ATR_altid as modified at this point to make sure
   * pbs_server got these attr values.  This worked fine before TORQUE modified
   * job launched into an async process.  At 2.0.0p6, a new attribute "SEND" flag
   * was added to handle this process. */

#else	/* PBS_SERVER */

  if (svr_authorize_jobreq(preq,pj) == -1) 
    {
    req_reject(PBSE_PERM,0,preq,NULL,NULL);

    if (LOGLEVEL >= 6)
      {
      log_record(
        PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
        "no permission to start job");
      }

    return;
    }

  /* remove job from the server new job list, set state, and enqueue it */

  delete_link(&pj->ji_alljobs);

  svr_evaljobstate(pj,&newstate,&newsub,1);

  svr_setjobstate(pj,newstate,newsub);

  /* set the queue rank attribute */

  pj->ji_wattr[(int)JOB_ATR_qrank].at_val.at_long = ++queue_rank;
  pj->ji_wattr[(int)JOB_ATR_qrank].at_flags |= ATR_VFLAG_SET;

  if ((rc = svr_enquejob(pj))) 
    {
    job_purge(pj);

    req_reject(rc,0,preq,NULL,NULL);

    if (LOGLEVEL >= 6)
      {
      log_record(
        PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
        "cannot queue job");
      }

    return;
    }

  if (job_save(pj,SAVEJOB_FULL) != 0) 
    {
    job_purge(pj);

    req_reject(PBSE_SYSTEM,0,preq,NULL,NULL);

    if (LOGLEVEL >= 6)
      {
      log_record(
        PBSEVENT_JOB,
        PBS_EVENTCLASS_JOB,
        (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
        "cannot save job");
      }

    return;
    }

  /*
   * if the job went into a Route (push) queue that has been started,
   * try once to route it to give immediate feedback as a courtsey
   * to the user.
   */

  pque = pj->ji_qhdr;

  if ((preq->rq_fromsvr == 0) &&
      (pque->qu_qs.qu_type == QTYPE_RoutePush) &&
      (pque->qu_attr[(int)QA_ATR_Started].at_val.at_long != 0)) 
    {
    if ((rc = job_route(pj))) 
      {
      job_purge(pj);

      req_reject(rc,0,preq,NULL,NULL);

      if (LOGLEVEL >= 6)
        {
        log_record(
          PBSEVENT_JOB,
          PBS_EVENTCLASS_JOB,
          (pj != NULL) ? pj->ji_qs.ji_jobid : "NULL",
          "job route job");
        }

      /* FAILURE */

      return;
      }
    }

  /* need to format message first, before request goes away */

  sprintf(log_buffer,msg_jobnew, 
    preq->rq_user, preq->rq_host,
    pj->ji_wattr[(int)JOB_ATR_job_owner].at_val.at_str,
    pj->ji_wattr[(int)JOB_ATR_jobname].at_val.at_str,
    pj->ji_qhdr->qu_qs.qu_name);

  /* acknowledge the request with the job id */

  reply_jobid(preq,pj->ji_qs.ji_jobid,BATCH_REPLY_CHOICE_Commit);

  LOG_EVENT(
    PBSEVENT_JOB,
    PBS_EVENTCLASS_JOB,
    pj->ji_qs.ji_jobid,
    log_buffer);

  if ((pj->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0)
    {
    /* notify creator where job is */

    issue_track(pj);
    }
#endif		/* PBS_SERVER */

  return;
  }  /* END req_commit() */




/*
 * locate_new_job - locate a "new" job which has been set up req_quejob on
 *	the servers new job list.
 *	
 *	This function is used by the sub-requests which make up the global
 *	"Queue Job Request" to locate the job structure.
 *
 *	If the jobid is specified (will be for rdytocommit and commit, but not
 *	for script), we search for a matching jobid.  
 *	
 *	The job must (also) match the socket specified and the host associated
 *	with the socket unless ji_fromsock == -1, then its a recovery situation.
 */

static job *locate_new_job(

  int   sock,   /* I */
  char *jobid)  /* I (optional) */

  {
  job *pj;

  pj = (job *)GET_NEXT(svr_newjobs);

  while (pj != NULL) 
    {
    if ((pj->ji_qs.ji_un.ji_newt.ji_fromsock == -1) ||
       ((pj->ji_qs.ji_un.ji_newt.ji_fromsock == sock) &&
        (pj->ji_qs.ji_un.ji_newt.ji_fromaddr == get_connectaddr(sock)))) 
      {
      if (jobid != NULL) 
        {
        if (!strncmp(pj->ji_qs.ji_jobid,jobid,PBS_MAXSVRJOBID))
          {
          /* requested job located */

          break;
          }
        } 
      else if (pj->ji_qs.ji_un.ji_newt.ji_fromsock == -1)
        {
        /* empty job slot located */

        break;
        }
#ifdef __TNW
      else if ((pj->ji_qs.ji_substate != JOB_SUBSTATE_TRANSIN) || 
               (time_now - pj->ji_wattr[(int)JOB_ATR_mtime].at_val.at_long > 9000))
        {
        /* ignore 'stale' recovered job */

        /* NO-OP */
        }
#endif /* __TNW */
      else
        {
        /* matching job slot located */

        break;
        }
      }    /* END if ((pj->ji_qs.ji_un.ji_newt.ji_fromsock == -1) || ...) */

    pj = (job *)GET_NEXT(pj->ji_alljobs);
    }  /* END while(pj != NULL) */

  /* return job slot located (NULL on FAILURE) */

  return(pj);
  }  /* END locate_new_job() */




/*
 * City Tools patch
 */

#define UserAcctMax  12

struct {
  int  ActCnt;                          /* How many projects */
  int  ActMax;                          /* Max allowed in this struct */
  char ActRaw[80];                      /* The raw project data */
  char ActDat[80];                      /* ActRaw with \0 as necessary */
  char *ActAdr[UserAcctMax];            /* Pointers to ActDat items */
  } UserAcct = {
      0, UserAcctMax,
      "", "",
      {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0} };   /* UserAcctMax values */



#ifdef PNOT
int user_account_verify(

  char *arguser, 
  char *argaccount)

  {
  int  i;
  int  rc = 0; /* 0 = failure, 1 = success */

  if (!user_account_read_user(arguser)) 
    {
    sprintf(log_buffer,"user_account_verify(%s, %s) -> USER NOT FOUND",
      arguser, 
      argaccount);

    goto user_account_verify_done;
    }

  for (i = 0;i < UserAcct.ActCnt;++i) 
    {
    if (strcmp(argaccount,UserAcct.ActAdr[i]) == 0) 
      {
      sprintf(log_buffer,"user_account_verify(%s, %s) -> SUCCESS",
        arguser, 
        argaccount);

      rc = 1;

      goto user_account_verify_done;
      }
    }    /* END for (i) */

  sprintf(log_buffer,"user_account_verify(%s, %s) -> FAILED",
    arguser, 
    argaccount);

user_account_verify_done:

  log_event(
    PBSEVENT_JOB|PBSEVENT_SECURITY, 
    PBS_EVENTCLASS_SERVER,
    msg_daemonname, 
    log_buffer);

  return(rc);
  }




char *user_account_default(

  char *arguser)

  {
  char *rc = 0; /* 0 = failure, <address> = success */

  if (!user_account_read_user(arguser))
    {
    sprintf(log_buffer,"user_account_default(%s) = USER NOT FOUND",
      arguser);

    goto user_account_default_done;
    }

  if (UserAcct.ActCnt < 1) 
    {
    sprintf(log_buffer,"user_account_default(%s) = NO PROJECT FOUND",
      arguser);

    goto user_account_default_done;
    }

  rc = UserAcct.ActAdr[0];

  sprintf(log_buffer,"user_account_default(%s) = %s",
    arguser, 
    rc);

user_account_default_done:

  log_event(
    PBSEVENT_JOB|PBSEVENT_SECURITY, 
    PBS_EVENTCLASS_SERVER,
    msg_daemonname, 
    log_buffer);

  return(rc);
  }




/*
 * Given a username, returns that user's accounts from the user->account file
 * Returns 0 if username isn't found
 */

#define AcctScanUser  0
#define AcctScanAcct  1
#define AcctScanLine -1
#define AcctFile "/usr/local/etc/usertg/project-by-user-torque.map"

int user_account_read_user(

  char *arguser)

  {
  char  proj_file[] = AcctFile;
  int   fd;
  char  s_buf[64*1024];
  int   readsize;
  int   scanmode = AcctScanUser;
  int   i, j;
  char  *ci;
  char  *cj;

  int   arguserlen;

  arguserlen = strlen(arguser);

  if ((fd = open(proj_file,O_RDONLY,0)) < 0) 
    {
    return(0);
    }

  if (lseek(fd,(off_t)0,SEEK_SET) != 0)  
    {
    close(fd);

    return(0);
    }

        readsize = read(fd, s_buf, sizeof(s_buf));
        close(fd);
       if (readsize < 1 || readsize > sizeof(s_buf))
           return(0);                       /* Bail if not sane */

       for (i=0; i<readsize; ++i) {
           /* First,  handle comments and whitespace */
           if (scanmode == AcctScanLine) {  /* Looking for new line */
               if (s_buf[i] == '\n')        /* Found it */
                   scanmode = AcctScanUser;
               continue;
           }
           if (isspace(s_buf[i]))            /* Skip spaces */
                continue;
           if (s_buf[i] == '#') {            /* Comment found */
               scanmode = AcctScanLine;
                continue;
            }

           /* Next, handle user and account scanning */
            if (scanmode == AcctScanUser) {
               if ((i+arguserlen) > readsize) /* Past the end */
                   return(0);
               if (strncmp(&s_buf[i], arguser, arguserlen)) {
                   scanmode = AcctScanLine;   /* Not arguser, next line */
                    continue;
                }
               if (isspace(s_buf[i+arguserlen]) ||
                           s_buf[i+arguserlen] == ':') {
                  i += arguserlen;            /* Is arguser */
                  scanmode = AcctScanAcct;
                } else {                       /* Whatever, ignore it */
                  scanmode = AcctScanLine;
                  continue;
               }
           } else {                           /* scanmode == AcctScanAcct */
              if (s_buf[i] == ':' || isspace(s_buf[i]))
                  continue;
               for (j=i; j<readsize; j++) {
                  if (isspace(s_buf[j])) {
                      strncpy(UserAcct.ActRaw, &s_buf[i], j-i);
                      UserAcct.ActRaw[j-i] = '\0';
                      goto have_account;
                   }
             }
              return(0);
           }
       }

  return(0);                             /* Never found it */

have_account:

  if (strlen(UserAcct.ActRaw) < 1)       /* Nothing found */
    {
    return(0);
    }

  strcpy(UserAcct.ActDat,UserAcct.ActRaw);

  UserAcct.ActCnt = 0;
  UserAcct.ActMax = UserAcctMax;

  for (ci = &UserAcct.ActDat[0];*ci != '\0';ci++) 
    {
    if (isspace(*ci) || (*ci == ','))
      continue;

    for (cj = ci + 1;!isspace(*cj) && (*cj != ',') && (*cj != '\0');cj++) 
      {
      /* NO-OP */
      }

    *cj = '\0';
 
    UserAcct.ActAdr[UserAcct.ActCnt++] = ci;

    ci = cj;
    }

  return(1);
  }  /* END user_account_read_user() */

#endif /* PNOT */


/* END req_quejob.c() */

