/*
*         OpenPBS (Portable Batch System) v2.3 Software License
* 
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
* 
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
* 
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
* 
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
* 
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
* 
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
* 
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
* 
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
* 
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
* 
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
* 
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information 
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
* 
* 7. DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include 	<sys/types.h>
#include	<netinet/in.h>

#include	"portability.h"
#include	"libpbs.h"
#include	"server_limits.h"
#include	"list_link.h"
#include	"attribute.h"
#include	"resource.h"
#include	"server.h"
#include	"net_connect.h"
#include	"batch_request.h"
#include	"work_task.h"
#include 	"svrfunc.h"
#include	"job.h"
#include	"log.h"
#include	"pbs_nodes.h"
#include	"rpp.h"
#include	"dis.h"
#include	"dis_init.h"
#include	"resmon.h"
#include  "query_configs.h"

extern void DIS_rpp_reset A_((void));

extern int LOGLEVEL;

#if !defined(H_ERRNO_DECLARED)
extern int h_errno;
#endif

int	 	 svr_totnodes = 0;	/* total number nodes defined       */
int		 svr_clnodes  = 0;	/* number of cluster nodes	    */
int		 svr_tsnodes  = 0;	/* number of time shared nodes	    */
int		 svr_chngNodesfile = 0;	/* 1 signals want nodes file update */
					/* on server shutdown, (qmgr mods)  */
struct pbsnode **pbsndlist = NULL;
struct pbsnode **pbsndmast = NULL;
static int	 svr_numnodes = 0;	/* number nodes currently available */
static int	 exclusive;		/* node allocation type */

static FILE 	*nstatef = NULL;

extern int	 server_init_type;
extern int	 has_nodes;

extern time_t    time_now;
  
extern int	ctnodes A_((char *));
extern char	*path_home;
extern char	*path_nodes;
extern char	*path_nodes_new;
extern char	*path_nodestate;
extern unsigned int pbs_mom_port;
extern char  server_name[];

extern struct server server;

#define		SKIP_NONE	0
#define		SKIP_EXCLUSIVE	1
#define		SKIP_ANYINUSE	2

static int hasprop(struct pbsnode *, struct prop *);

/*

 GBS - I put this in since it's used in the server to mom
 communication for resource manager information.  The server
 opens rpp sockets for pinging.  I just used those for the
 resource manager queries.

*/

static void funcs_dis() /* The equivalent of DIS_tcp_funcs() */

  {
  if (dis_getc != rpp_getc) 
    {
    dis_getc = (int (*)(int))rpp_getc;
    dis_puts = (int (*)(int, const char *, size_t))rpp_write;
    dis_gets = (int (*)(int, char *, size_t))rpp_read;
    disr_commit = (int (*)(int, int))rpp_rcommit;
    disw_commit = (int (*)(int, int))rpp_wcommit;
    }

  return;
  }

/*#define      setup_dis(x)    funcs_dis()  */    /* RPP doesn't need reset */
/*#define      close_dis(x)    rpp_close(x) */
/*#define      flush_dis(x)    rpp_flush(x) */

/*
 * Tree search generalized from Knuth (6.2.2) Algorithm T just like
 * the AT&T man page says.
 *
 * The tree_t structure is for internal use only, lint doesn't grok it.
 *
 * Written by reading the System V Interface Definition, not the code.
 *
 * Totally public domain.
 */
/*LINTLIBRARY*/

/*
**      Modified by Tom Proett <proett@nas.nasa.gov> for PBS.
*/

typedef struct tree_t {
  u_long		key;
  struct pbsnode	*nodep;
  struct tree_t	*left, *right;
  } tree;

tree	*ipaddrs = NULL;	/* tree of ip addrs */
tree	*streams = NULL;	/* tree of stream numbers */



/* find value in tree, return NULL if not found */

struct pbsnode *tfind(

  const u_long   key,	/* key to be located */
  tree         **rootp)	/* address of tree root */

  {
  if (rootp == NULL)
    {
    return NULL;
    }

  while (*rootp != NULL) 
    {		/* Knuth's T1: */
    if (key == (*rootp)->key)	/* T2: */
      {
      return(*rootp)->nodep;	/* we found it! */
      }

    rootp = (key < (*rootp)->key) ?
      &(*rootp)->left :	/* T3: follow left branch */
      &(*rootp)->right;	/* T4: follow right branch */
    }

  return(NULL);
  }  /* END tfind() */






struct pbsnode *tfind_addr(

  const u_long key)

  {
  return tfind(key,&ipaddrs);
  }





void tinsert(

  const	u_long     key,		/* key to be located */
  struct pbsnode  *nodep,
  tree           **rootp)	/* address of tree root */

  {
  register tree	*q;

  if (nodep != NULL)
    {
    /*
    DBPRT(("tinsert: %lu %s stream %d\n", 
      key,
      nodep->nd_name, 
      nodep->nd_stream))
    */
    }

  if (rootp == NULL)
    {
    return;
    }

  while (*rootp != NULL) 
    {	/* Knuth's T1: */
    if (key == (*rootp)->key)	/* T2: */
      {
      return;			/* we found it! */
      }

    rootp = (key < (*rootp)->key) ?
      &(*rootp)->left :	/* T3: follow left branch */
      &(*rootp)->right;	/* T4: follow right branch */
    }

   q = (tree *) malloc(sizeof(tree));	/* T5: key not found */

   if (q != NULL) 
     {			/* make new node */
     *rootp = q;			/* link new node to old */
     q->key = key;			/* initialize new node */
     q->nodep = nodep;
     q->left = q->right = NULL;
     }

  return;
  }  /* END tinsert() */




/* delete node with given key */

void *tdelete(

  const u_long   key,	/* key to be deleted */
  tree         **rootp)	/* address of the root of tree */

  {
  tree		*p;
  register tree	*q;
  register tree	*r;

  if (LOGLEVEL >= 6)
    {
    DBPRT(("tdelete: %lu\n", 
      key))
    }

  if (rootp == (struct tree_t **)0 || (p = *rootp) == (struct tree_t *)0)
    {
    return(NULL);
    }

  while (key != (*rootp)->key) 
    {
    p = *rootp;

    rootp = (key < (*rootp)->key) ?
      &(*rootp)->left :		/* left branch */
      &(*rootp)->right;		/* right branch */

    if (*rootp == NULL)
      {
      return(NULL);		/* key not found */
      }
    }

  r = (*rootp)->right;				/* D1: */

  if ((q = (*rootp)->left) == NULL)		/* Left */
    {
    q = r;
    }
  else if (r != NULL) 
    {		/* Right is null? */
    if (r->left == NULL) 
      {	/* D2: Find successor */
      r->left = q;

      q = r;
      }
    else 
      {		/* D3: Find (struct tree_t *)0 link */
      for (q = r->left;q->left != NULL;q = r->left)
        r = q;

      r->left = q->right;

      q->left = (*rootp)->left;
      q->right = (*rootp)->right;
      }
    }

  free((struct tree_t *)*rootp);     /* D4: Free node */

  *rootp = q;                         /* link parent to new node */

  return(p);
  }  /* END tdelete() */




void tfree(

  tree **rootp)

  {
  if (rootp == NULL || *rootp == NULL)
    {
    return;
    }
  
  tfree(&(*rootp)->left);
  tfree(&(*rootp)->right);

  free(*rootp);

  *rootp = NULL;

  return;
  }  /* END tfree() */




/* update_node_state - central location for updating node state */

void update_node_state(

  struct pbsnode *np,         /* I (modified) */
  int             newstate)   /* I */

  {
  char      *id = "update_node_state";

  struct pbssubn *sp;

  /*
   * LOGLEVEL >= 4 logs all state changes
   *          >= 2 logs down->(busy|free) changes
   *          (busy|free)->down changes are always logged
   */          

  log_buffer[0] = '\0';

  if (newstate & INUSE_DOWN)
    {
    if (!(np->nd_state & INUSE_DOWN))
      {
      sprintf(log_buffer,"node %s marked down",
        (np->nd_name != NULL) ? np->nd_name : "NULL");

      np->nd_state |= INUSE_DOWN; 
      np->nd_state &= ~INUSE_UNKNOWN; 

      /* mark all subnodes down */

      for (sp = np->nd_psn;sp != NULL;sp = sp->next)
        {
        sp->inuse |= INUSE_DOWN;
        }
      }
          
    /* ignoring the obvious possiblity of a "down,busy" node */
    }
  else if (newstate & INUSE_BUSY)
    {                                                                                      
    if ((!(np->nd_state & INUSE_BUSY) && (LOGLEVEL >= 4)) ||                               
        ( (np->nd_state & INUSE_DOWN) && (LOGLEVEL >= 2)))
      {
      sprintf(log_buffer,"node %s marked busy",
        (np->nd_name != NULL) ? np->nd_name : "NULL");
      }

    np->nd_state |= INUSE_BUSY; 
    np->nd_state &= ~INUSE_UNKNOWN; 

    if (np->nd_state & INUSE_DOWN)
      {
      np->nd_state &= ~INUSE_DOWN; 

      /* clear down on all subnodes */

      for (sp = np->nd_psn;sp != NULL;sp = sp->next)
        {
        sp->inuse &= ~INUSE_DOWN;
        }
      }

    }
  else if (newstate == INUSE_FREE)
    {
    if (((np->nd_state & INUSE_DOWN) && (LOGLEVEL >= 2)) ||
        ((np->nd_state & INUSE_BUSY) && (LOGLEVEL >= 4)))
      {
      sprintf(log_buffer,"node %s marked free",
        (np->nd_name != NULL) ? np->nd_name : "NULL");
      }

    np->nd_state &= ~INUSE_BUSY; 
    np->nd_state &= ~INUSE_UNKNOWN; 

    if (np->nd_state & INUSE_DOWN)
      {
      np->nd_state &= ~INUSE_DOWN; 

      /* clear down on all subnodes */

      for (sp = np->nd_psn;sp != NULL;sp = sp->next)
        {
        sp->inuse &= ~INUSE_DOWN;
        }
      }
    }

  if (newstate & INUSE_UNKNOWN)
    {
    np->nd_state |= INUSE_UNKNOWN;
    }                                                                                    

  if (log_buffer[0] != '\0')
    {
    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  return;
  }  /* END update_node_state() */


/*
 * find_job_by_node - return a job structure by looking for a jobid in a
 * specific node struct 
 *
 * probably only useful as a test to see if a job exists on a given node
 * and it's much faster than find_job()
 */

job *find_job_by_node(

  struct pbsnode *pnode, /* I */
  char *jobid)           /* I */

  {
  struct pbssubn *np;
  struct jobinfo *jp;
  struct job     *pjob = NULL;

  char *at;

  if ((at = strchr(jobid,(int)'@')) != NULL)
    *at = '\0'; /* strip off @server_name */

  /* for each subnode on node ... */
  for (np = pnode->nd_psn;np != NULL;np = np->next)
    {

    /* for each jobinfo on subnode on node ... */
    for (jp = np->jobs;jp != NULL;jp = jp->next)
      {
      if ((jp->job != NULL) && 
          (jp->job->ji_qs.ji_jobid != NULL) && 
          (strcmp(jobid,jp->job->ji_qs.ji_jobid) == 0))
        {
        pjob=jp->job;

        break;
        }
      }
    }

  if (at)                                                                                     
    *at = '@';  /* restore @server_name */

  return(pjob);
  } /* END find_job_by_node() */


/*
 * sync_node_jobs() - determine if a MOM has a stale job and possibly delete it
 */
void sync_node_jobs(

  struct pbsnode *np,            /* I */
  char           *jobstring_in)  /* I */

  {
  char      *id = "sync_node_jobs";
  char      *joblist;
  char      *jobidstr;
  struct batch_request *preq;
  int conn;

  if ((jobstring_in == NULL) || (!isdigit(*jobstring_in)))
    return;

  joblist=strdup(jobstring_in);

  jobidstr=strtok(joblist," ");

  while((jobidstr!=NULL) && isdigit(*jobidstr))
    {
    if (strstr(jobidstr,server_name) != NULL)
      {
      if (find_job_by_node(np,jobidstr) == NULL)
        {
        sprintf(log_buffer,"stray job %s found on %s",
          jobidstr,
          np->nd_name);

        log_err(-1,id,log_buffer);

        if ((preq = alloc_br(PBS_BATCH_DeleteJob)) == NULL)
          {
          log_err(-1,id,"unable to allocate DeleteJob request - big trouble!");

          break;
          }

        conn = svr_connect(
          np->nd_addrs[0],
          pbs_mom_port,
          process_Dreply,
          ToServerDIS);

        strcpy(preq->rq_ind.rq_delete.rq_objname,jobidstr);

        issue_Drequest(conn,preq,release_req,0);

        /* release_req will free preq and close connection */
        }
      }

    jobidstr=strtok(NULL," ");
    }

  free(joblist);
  }  /* END sync_node_jobs() */



int is_stat_get(
		
  struct pbsnode *np)  /* I (modified) */

  { 
  char      *id = "is_stat_get";

  int stream = np->nd_stream;
  int        rc;
  int        count = 0;
  char      *ret_info;
  attribute  temp;
  char       date_attrib[100];
  int        msg_error = 0;

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer,"received status from node %s",
      (np != NULL) ? np->nd_name : "NULL");

    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  if (stream < 0)
    {
    return(DIS_EOF);
    }

  /*
   *  Before filling the "temp" attribute, initialize it.
   *  The second and third parameter to decode_arst are never
   *  used, so just leave them empty. (GBS)
   */

  memset(&temp,0,sizeof(temp));

  rc = DIS_SUCCESS;

  if (decode_arst(&temp,NULL,NULL,NULL))
    {
    DBPRT(("is_stat_get:  cannot initialize attribute\n"));
      rpp_eom(stream);

    return(DIS_NOCOMMIT);
    }

  while (rc != DIS_EOD)
    {
    ++count;
    funcs_dis();

    if ((ret_info = disrst(stream,&rc)) != NULL)
      {
      /* add the info to the "temp" attribute */

      if (decode_arst(&temp,NULL,NULL,ret_info))
        {
        DBPRT(("is_stat_get: cannot add attributes\n"));

        free_arst(&temp);

        free(ret_info);

        return(DIS_NOCOMMIT);
        }

      if (!strncmp(ret_info,"state",5))
        {
        /* MOM currently never sends multiple states - bad assumption for the future? */

        if (!strncmp(ret_info,"state=down",10))
          {
          update_node_state(np,INUSE_DOWN);
          }
        else if (!strncmp(ret_info,"state=busy",10))
          {
          update_node_state(np,INUSE_BUSY);
          }
        else if (!strncmp(ret_info,"state=free",10))
          {
          update_node_state(np,INUSE_FREE);
          }
        else 
          {
          sprintf(log_buffer,"unknown %s from node %s",
            ret_info,
            (np->nd_name != NULL) ? np->nd_name : "NULL");

          log_err(-1,id,log_buffer);

          update_node_state(np,INUSE_UNKNOWN);
          }                        
        }
      else if (!strncmp(ret_info,"me",2))  /* shorter str compare than "message" */
        {
        if (!strncmp(ret_info,"message=ERROR",13))
          {
          msg_error = 1;
          }
        }
      else if (server.sv_attr[(int)SRV_ATR_MomJobSync].at_val.at_long && 
               (strncmp(ret_info,"jobs=",5) == 0))
        {
        sync_node_jobs(np,ret_info+5);
        }

      free(ret_info);
      }
    }    /* END while (rc != DIS_EOD) */

  if (msg_error && server.sv_attr[(int)SRV_ATR_DownOnError].at_val.at_long)
    {
    update_node_state(np,INUSE_DOWN);
    }

  /* it's nice to know when the last update happened */

  sprintf(date_attrib,"rectime=%ld",
    (long)time_now);

  if (decode_arst(&temp,NULL,NULL,date_attrib))
    {
    DBPRT(("is_stat_get:  cannot add date_attrib\n"));

    free_arst(&temp);

    return(DIS_NOCOMMIT);
    }

  /* clear the transmission */

  rpp_eom(stream);

  /* insert the information from "temp" into np */

  if (node_status_list(&temp,np,ATR_ACTION_ALTER))
    {
    DBPRT(("is_stat_get: cannot set node status list\n"));

    return(DIS_NOCOMMIT);
    }

  /* NOTE:  node state adjusted in update_node_state() */

  return(DIS_SUCCESS);
  }  /* END is_stat_get() */




/*
**	Start a standard inter-server message.
*/

int is_compose(

  int stream,  /* I */
  int command) /* I */

  {
  int ret;

  if (stream < 0)
    {
    return(DIS_EOF);
    }

  DIS_rpp_reset();

  ret = diswsi(stream,IS_PROTOCOL);

  if (ret != DIS_SUCCESS)
    goto done;

  ret = diswsi(stream,IS_PROTOCOL_VER);

  if (ret != DIS_SUCCESS)
    goto done;

  ret = diswsi(stream,command);

  if (ret != DIS_SUCCESS)
    goto done;

  return(DIS_SUCCESS);

done:

  DBPRT(("is_compose: send error %s\n", 
    dis_emsg[ret]))

  return(ret);
  }  /* END is_compose() */




/* EOF on a stream received (either stream or addr must be specified) */
/* mark node down and remove associated streams */

void stream_eof(

  int	 stream,  /* I (optional) */
  u_long addr,    /* I (optional) */
  int	 ret)     /* I (ignored) */

  {
  static char     id[] = "stream_eof";
  struct pbsnode *np;

  rpp_close(stream);

  np = NULL;

  if (stream >= 0)
    {
    /* find who the stream belongs to and mark down */

    np = tfind((u_long)stream,&streams);
    }

  if ((np == NULL) && (addr != 0))
    {
    np = tfind((u_long)addr,&ipaddrs);
    }

  if (np == NULL)
    {
    /* cannot locate node */

    return;
    }

  sprintf(log_buffer, "connection to %s dropped (%s).  setting node state to down\n",
    dis_emsg[ret],
    np->nd_name);

  log_err(-1,id,log_buffer);

  /* mark down node and all subnodes */

  update_node_state(np,INUSE_DOWN);

  /* remove stream from list of valid connections */

  if (stream >= 0)
    {
    np->nd_stream = -1;

    tdelete((u_long)stream,&streams);
    }

  return;
  }  /* END stream_eof() */




/*
 *	Send a ping to any node that is in an unknown state.
 *	If wt_parm1 is NULL, set up a worktask to ping again.
 *
 *      This shouldn't be called anymore...
 *      This is mostly only used for opening the socket
 *      connection to the node.
*/

void ping_nodes(

  struct work_task *ptask)  /* I (optional) */

  {
  static  char	        *id = "ping_nodes";
  struct  pbsnode	*np;
  struct  sockaddr_in	*addr;
  int                    i, ret, com;
  extern  int            pbs_rm_port;

  static  int            startcount = 0;

  extern int RPPConfigure(int,int);
  extern int RPPReset(void);

  DBPRT(("%s: entered\n",
    id))

  sprintf(log_buffer,"ping attempting to contact %d nodes\n",
    (svr_totnodes - startcount > 256) ? 
      256 : 
      (svr_totnodes - startcount < 0) ? 
        svr_totnodes : 
        svr_totnodes - startcount); /* phew! */

  log_record(
    PBSEVENT_SCHED, 
    PBS_EVENTCLASS_REQUEST, 
    id,
    log_buffer);

  /* change RPP to report node state quickly */

  RPPConfigure(2,2); /* (timeout,retry) retry must be at least 2 */

  for (i = startcount;i < svr_totnodes;i++) 
    {
    if (i - startcount > 256)
      {
      /* only ping 256 nodes at a time, ping next batch later */

      break;
      }

    np = pbsndmast[i];

    if (np->nd_state & (INUSE_DELETED|INUSE_OFFLINE))
      continue;

    if (np->nd_stream < 0) 
      {
      /* open new stream */

      np->nd_stream = rpp_open(np->nd_name,pbs_rm_port,NULL);

      update_node_state(np,INUSE_DOWN);

      if (np->nd_stream == -1) 
        {
	sprintf(log_buffer,"rpp_open to %s",
	  np->nd_name);

	log_err(errno,id,log_buffer);

	continue;
	}

      tinsert((u_long)np->nd_stream,np,&streams);
      }

    DBPRT(("%s: ping %s\n",
      id,
      np->nd_name))

    /* nodes are down until proven otherwise */

    com = IS_HELLO;

    ret = is_compose(np->nd_stream,com);

    if (ret == DIS_SUCCESS) 
      {
      if (rpp_flush(np->nd_stream) == 0)
        {
        sprintf(log_buffer,"successful ping to node %s\n",
          np->nd_name);

        log_record(
          PBSEVENT_SCHED,
          PBS_EVENTCLASS_REQUEST,
          id,
          log_buffer);

        continue;
        }

      ret = DIS_NOCOMMIT;
      }

    /* ping unsuccessful, mark node down, clear stream */

    update_node_state(np,INUSE_DOWN);

    addr = rpp_getaddr(np->nd_stream);

    sprintf(log_buffer,"%s %d to %s(%s)",
      dis_emsg[ret],
      errno,
      np->nd_name,
      netaddr(addr));

    log_err(-1,id,log_buffer);

    rpp_close(np->nd_stream);

    tdelete((u_long)np->nd_stream,&streams);

    np->nd_stream = -1;
    }  /* END for (i) */

  RPPReset();

  startcount = i;

  /* only ping nodes once (disable new task) */

  if (startcount < svr_totnodes)
    {
    /* continue outstanding pings in 3 seconds */

    set_task(WORK_Timed,time_now + 3,ping_nodes,NULL); 
    }

  return;
  }  /* END ping_nodes() */




 /*
  **     Mark any nodes that haven't checked in as down.
  **     This should be used rather than the ping_nodes task.  If
  **     the node isn't down then it checks to see that the
  **     last update hasn't been too long ago.
  */

void check_nodes(

  struct work_task *ptask)  /* I (modified) */

  {
  static char     id[] = "check_nodes";
  struct pbsnode *np;
  int             i,chk_len;

  /* load min refresh interval */

  chk_len = server.sv_attr[(int)SRV_ATR_check_rate].at_val.at_long;

  if (LOGLEVEL >= 5)
    {
    sprintf(log_buffer,"verifying nodes are active (min_refresh = %d seconds)",
      chk_len);

    log_event(
      PBSEVENT_ADMIN,
      PBS_EVENTCLASS_SERVER,
      id,
      log_buffer);
    }

  /* evaluate all nodes */

  for (i = 0;i < svr_totnodes;i++) 
    {
    np = pbsndmast[i];

    if (np->nd_state & (INUSE_DELETED|INUSE_DOWN))
      continue;

    if (np->nd_lastupdate < (time_now - chk_len))
      {
      if (LOGLEVEL >= 0)
        {
        sprintf(log_buffer,"node %s not detected in %ld seconds, marking node down",
          np->nd_name,
          (int)time_now - np->nd_lastupdate);

        log_event(
          PBSEVENT_ADMIN,
          PBS_EVENTCLASS_SERVER,
          id,
          log_buffer);
        }

      update_node_state(np,(INUSE_DOWN));
      }
    }    /* END for (i = 0) */

  if (ptask->wt_parm1 == NULL) 
    {
    if (server_init_type == RECOV_HOT)
      {
      /* rapid ping rate while hot restart */

      chk_len = 15;  /* doesn't do much good, this routine only check nodes to mark them down, not up? */
      }

    set_task(
      WORK_Timed,
      time_now + chk_len, 
      check_nodes, 
      NULL);
    }

  return;
  }  /* END check_nodes() */



/* sync w/#define IS_XXX */

const char *PBSServerCmds2[] = {
  "NULL",
  "HELLO",
  "CLUSTER_ADDRS",
  "UPDATE",
  "STATUS",
  NULL };

/*
**	Input is coming from another server (MOM) over a DIS rpp stream.
**	Read the stream to get a Inter-Server request.
*/

void is_request(

  int  stream,  /* I */
  int  version, /* I */
  int *cmdp)    /* O (optional) */

  {
  static	char	id[] = "is_request";
  int		command = 0;
  int		ret = DIS_SUCCESS;
  int		i, j;

  unsigned long	ipaddr;
  struct	sockaddr_in *addr;
  struct	pbsnode	*np = NULL, *node;
  struct pbssubn *sp;

  if (cmdp != NULL)
    *cmdp = 0;

  if (LOGLEVEL >= 4)
    {
    sprintf(log_buffer,"message received from stream %d (version %d)\n",
      stream,
      version);

    log_event(
      PBSEVENT_ADMIN,
      PBS_EVENTCLASS_SERVER,
      id,
      log_buffer);
    }

  addr = rpp_getaddr(stream);

  if (version != IS_PROTOCOL_VER) 
    {
    sprintf(log_buffer,"protocol version %d unknown from %s",
      version, 
      netaddr(addr));

    log_err(-1,id,log_buffer);

    rpp_close(stream);

    return;
    }

  /* check that machine is known */

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer,"message received from stream %s\n",
      netaddr(addr));

    log_event(
      PBSEVENT_ADMIN,
      PBS_EVENTCLASS_SERVER,
      id,
      log_buffer);
    }

  if ((node = tfind((u_long)stream,&streams)) != NULL)
    goto found;

  ipaddr = ntohl(addr->sin_addr.s_addr);

  if ((node = tfind(ipaddr,&ipaddrs)) != NULL) 
    {
    if (node->nd_stream >= 0) 
      {
      if (LOGLEVEL >= 3)
        {
        sprintf(log_buffer,"stream %d from node %s already open on %d (marking node state 'unknown'",
          stream,
          node->nd_name,
          node->nd_stream);

        log_event(
          PBSEVENT_ADMIN,
          PBS_EVENTCLASS_SERVER,
          id,
          log_buffer);
        }

      rpp_close(stream);
      rpp_close(node->nd_stream);

      tdelete((u_long)node->nd_stream,&streams);

      if (node->nd_state & INUSE_OFFLINE)
        {
        node->nd_state = (INUSE_UNKNOWN|INUSE_OFFLINE);
        }
      else
        {
        node->nd_state = INUSE_UNKNOWN;
        }
 
      node->nd_stream = -1;

      /* do a ping in 5 seconds */

      /* 
      set_task(WORK_Timed,time_now + 5,
        ping_nodes, node);
      */

      return;
      }

    node->nd_stream = stream;
  
    tinsert((u_long)stream,node,&streams);

    goto found;
    }  /* END if ((node = tfind(ipaddr,&ipaddrs)) != NULL) */

  /* node not listed in trusted ipaddrs list */

  sprintf(log_buffer,"bad attempt to connect from %s (address not trusted)",
    netaddr(addr));

  if (LOGLEVEL >= 2)
    {
    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  log_err(-1,id,log_buffer);

  rpp_close(stream);

  return;

found:

  command = disrsi(stream,&ret);

  if (ret != DIS_SUCCESS)
    goto err;

  if (cmdp != NULL)
    *cmdp = command;

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer,"message %s (%d) received from mom on host %s (%s)",
      PBSServerCmds2[command],
      command,
      node->nd_name,
      netaddr(addr));

    log_event(
      PBSEVENT_ADMIN,
      PBS_EVENTCLASS_SERVER,
      id,
      log_buffer);
    }

  switch (command) 
    {
    case IS_NULL:		/* a ping from server */

      DBPRT(("%s: IS_NULL\n",
        id))

      break;

    case IS_HELLO:

      if (LOGLEVEL >= 1)
        {
        sprintf(log_buffer,"HELLO received from %s\n",
          node->nd_name);

        log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
        }

      ret = is_compose(stream,IS_CLUSTER_ADDRS);

      if (ret != DIS_SUCCESS)
        goto err;

      for (i = 0;i < svr_totnodes;i++) 
        {
        np = pbsndmast[i];

        if (np->nd_state & INUSE_DELETED)
          continue;

        if (LOGLEVEL == 7)  /* higher loglevel gets more info below */
          {
          sprintf(log_buffer,"adding node[%d] %s to hello response\n",
            i,
            np->nd_name);

          log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
          }

        for (j = 0;np->nd_addrs[j];j++) 
          {
          u_long ipaddr = np->nd_addrs[j];

          if (LOGLEVEL >= 8)
            {
            sprintf(log_buffer,"adding node[%d] interface[%d] %ld.%ld.%ld.%ld to hello response\n",
              i,
              j,
              (ipaddr & 0xff000000) >> 24,
              (ipaddr & 0x00ff0000) >> 16,
              (ipaddr & 0x0000ff00) >> 8,
              (ipaddr & 0x000000ff));

            log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
            }

          ret = diswul(stream,ipaddr);

          if (ret != DIS_SUCCESS)
            goto err;
          }  /* END for (j) */
        }    /* END for (i) */

      /* NOTE:  re-enabled rpp_flush/disabled rpp_eom (CRI) */

      rpp_flush(stream);

      if (LOGLEVEL >= 3)
        {
        sprintf(log_buffer,"sending cluster-addrs to node %s\n",
          node->nd_name);

        log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
        }

      /* rpp_eom(stream); */

      /* CLUSTER_ADDRS successful */

      node->nd_state &= ~(INUSE_NEEDS_HELLO_PING);

      break;

    case IS_UPDATE:

      DBPRT(("%s: IS_UPDATE\n", 
        id))

      i = disrui(stream,&ret);

      if (ret != DIS_SUCCESS)
        {
        if (LOGLEVEL >= 1)
          {
          sprintf(log_buffer,"IS_UPDATE error %d on node %s\n",
            ret,
            node->nd_name);
 
          log_err(ret,id,log_buffer);
          }

        goto err;
        }

      DBPRT(("%s: IS_UPDATE %s 0x%x\n",
        id,
        node->nd_name,
        i))

      update_node_state(node,i);

      break;

    case IS_STATUS:

      /* pbs_server brought up 
         pbs_mom brought up  
         they send IS_HELLO to each other
         pbs_mom sends IS_STATUS message to pbs_server (replying to IS_HELLO)
         pbs_server sends IS_CLUSTER_ADDRS message to pbs_mom  (replying to IS_HELLO)
         pbs_mom uses IS_CLUSTER_ADDRS message to authorize contacts from sisters */

      if (LOGLEVEL >= 2)
        {
        sprintf(log_buffer,"IS_STATUS received from %s\n",
          node->nd_name);

        log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
        }

      ret = is_stat_get(node);

      if (ret != DIS_SUCCESS) 
        {
        if (LOGLEVEL >= 1)
          {
          sprintf(log_buffer,"IS_STATUS error %d on node %s\n",
            ret,
            node->nd_name);

          log_err(ret,id,log_buffer);
          }
 
        goto err;
        }

      node->nd_lastupdate = time_now;

      if (LOGLEVEL >= 9)
        {
        sprintf(log_buffer,"node '%s' is at state '0x%x'\n",
          node->nd_name,
          node->nd_state);

        log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
        }

      for (sp = node->nd_psn;sp != NULL;sp = sp->next)
        {
        if (!(node->nd_state & INUSE_OFFLINE) &&
             (sp->inuse & INUSE_OFFLINE))
          {
          /* this doesn't seem to ever happen */

          if (LOGLEVEL >= 2)
            {
            sprintf(log_buffer,"sync'ing subnode state '%s' with node state on node %s\n",
              "offline",
              node->nd_name);

            log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,log_buffer);
            }

          sp->inuse &= ~INUSE_OFFLINE;
          }

        sp->inuse &= ~INUSE_DOWN;
        }

      break;

    default:

      sprintf(log_buffer, "unknown command %d sent from %s",
        command,
        node->nd_name);

      log_err(-1,id,log_buffer);

      goto err;

      break;
    }  /* END switch(command) */

  rpp_eom(stream);

  return;

err:

  /* a DIS write error has occurred */

  if (LOGLEVEL >= 1)
    {
    DBPRT(("%s: error processing node %s\n",
      id,
      node->nd_name))
    }

  sprintf(log_buffer,"%s from %s(%s)",
    dis_emsg[ret],
    node->nd_name, 
    netaddr(addr));

  log_err(-1,id,log_buffer);

  rpp_close(stream);

  update_node_state(node,INUSE_DOWN);

  return;
  }  /* END is_request() */




void write_node_state()

  {
  struct pbsnode *np;
  static char *fmt = "%s %d\n";
  int	i;

  int   savemask;

  if (LOGLEVEL >= 5)
    DBPRT(("write_node_state: entered\n"))

  /* don't store volatile states like down and unknown */

  savemask = INUSE_OFFLINE|INUSE_DELETED|INUSE_RESERVE|INUSE_JOB|INUSE_JOBSHARE;

  if (nstatef != NULL) 
    {
    fseek(nstatef, 0L, SEEK_SET);	/* rewind and clear */
    if (ftruncate(fileno(nstatef),(off_t)0) != 0)
      {
      log_err(errno,"write_node_state","could not truncate file");
      return;
      }
    } 
  else 
    {
    /* need to open for first time, temporary-move to pbsd_init */
	
    if ((nstatef = fopen(path_nodestate,"w+")) == NULL) 
      {
      log_err(
        errno,
        "write_node_state",
        "could not open file");

      return;
      }
    }

  /*
  **	The only state that carries forward is if the
  **	node has been marked offline.
  */

  for (i = 0;i < svr_totnodes;i++) 
    {
    np = pbsndmast[i];

    if (np->nd_state & INUSE_DELETED)
      continue;

    if (np->nd_state & INUSE_OFFLINE)
      {
      fprintf(nstatef,fmt,
        np->nd_name, 
        np->nd_state & savemask);
      }
    }    /* END for (i) */

  if (fflush(nstatef) != 0)
    {
    log_err(errno,"write_node_state","failed saving node state to disk");
    }

  return;
  }  /* END write_node_state() */




/*
 * free_prop - free list of prop structures created by proplist()
 */

static void free_prop(

  struct prop *prop)

  {
  struct prop *pp;

  for (pp = prop;pp != NULL;pp = prop) 
    {
    prop = pp->next;

    free(pp->name);
    free(pp);
    }  /* END for (pp) */

  return;
  }    /* END free_prop() */


	

/*
 * unreserve - unreserve nodes
 *
 *	If handle is set to a existing resource_t, then release all nodes
 *	associated with that handle, otherwise, (this is dangerous) 
 *	if handle == RESOURCE_T_ALL, release all nodes period.
 */

void node_unreserve(

  resource_t handle)

  {
  struct  pbsnode *np;
  struct  pbssubn *sp;
  int     i;

  /* clear old reserve */

  for (i = 0;i < svr_totnodes;i++) 
    {
    np = pbsndlist[i];

    for (sp = np->nd_psn;sp;sp = sp->next) 
      {
      if (sp->inuse & INUSE_RESERVE) 
        {
        if ((handle == RESOURCE_T_ALL) || (handle == sp->allocto)) 
          {
          np->nd_nsnfree++;

          sp->inuse    &= ~INUSE_RESERVE;
          np->nd_state &= ~INUSE_RESERVE;
          }
        }
      }
    }

  return;
  }  /* END node_unreserve() */




/*
**	Look through the property list and make sure that all
**	those marked are contained in the node.
*/

static int hasprop(

  struct pbsnode *pnode,
  struct prop    *props)

  {
  struct  prop    *need;

  for (need = props;need;need = need->next) 
    {
    struct prop	*pp;

    if (need->mark == 0)	/* not marked, skip */
      continue;

    for (pp = pnode->nd_first;pp != NULL;pp = pp->next) 
      {
      if (strcmp(pp->name,need->name) == 0)
        break;		/* found it */
      }

    if (pp == NULL)
      {
      return(0);
      }
    }

  return(1);
  }  /* END hasprop() */





/*
 * see if node has the number of processors required 
 *	if free == SKIP_NONE,  check against total number of processors, else
 *	if free != SKIP_NONE,  check against number free
 *
 *	Return 1 if possible, 0 if not
 */

static int hasppn(

  struct pbsnode *pnode,     /* I */
  int             node_req,  /* I */
  int             free)      /* I */

  {
  if ((free != SKIP_NONE) && (pnode->nd_nsnfree >= node_req))
    {
    return(1);
    }

  if (!free && (pnode->nd_nsn >= node_req))
    {
    return(1);
    }

  return(0);
  }  /* END hasppn() */




/*
**	Mark the properties of a node that match the marked
**	properties given.
*/

static void mark(

  struct pbsnode *pnode,
  struct prop    *props)

  {
  struct  prop    *set, *pp;

  for (pp = pnode->nd_first;pp != NULL;pp=pp->next) 
    {
    pp->mark = 0;

    for (set = props;set;set = set->next) 
      {
      if (set->mark == 0)
        continue;

      if (strcmp(pp->name,set->name) == 0) 
        {
        pp->mark = 1;

        break;
        }
      }
    }

  return;
  }




#define	RECURSIVE_LIMIT	3

/*
**	Search for a node which contains properties glorf and the requirements.
**	skip indicates which nodes to pass over for this search.
**	Don't do any recursive calls deeper than RECURSIVE_LIMIT.
**      RETURN:  0 = failure, 1 = SUCCESS
*/

static int search(

  struct prop		*glorf,		/* properties */
  int			 vpreq,		/* VPs needed */
  int			 skip,
  int			 order,
  int			 depth)

  {
  static int pass = INUSE_OFFLINE|INUSE_DOWN|INUSE_RESERVE|INUSE_UNKNOWN|INUSE_DELETED;
  struct pbsnode *pnode;
  int	found;
  int	i;

  if (++depth == RECURSIVE_LIMIT)
    {
    return(0);
    }

  /* look for nodes we haven't picked already */

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    if (pnode->nd_ntype  == NTYPE_CLUSTER) 
      {
      if (pnode->nd_flag != okay)
        continue;

      if (pnode->nd_state & pass)
        continue;

      if (!hasprop(pnode,glorf))
        continue;

      if (skip == SKIP_NONE)  
        {
        if (vpreq > pnode->nd_nsn)
          continue;
        } 
      else if ((skip == SKIP_ANYINUSE) && 
              ((pnode->nd_state & INUSE_SUBNODE_MASK) || (vpreq > pnode->nd_nsnfree))) 
        {
        continue;
        } 
      else if ((skip == SKIP_EXCLUSIVE) && 
              ((pnode->nd_state & INUSE_SUBNODE_MASK) ||
               (vpreq > (pnode->nd_nsnfree + pnode->nd_nsnshared)))) 
        {
        continue;
        }
 
      pnode->nd_flag = thinking;

      mark(pnode,glorf);

      pnode->nd_needed = vpreq;
      pnode->nd_order  = order;

      /* SUCCESS */

      return(1);
      }
    }

  if (glorf == NULL)		/* no property */
    {
    /* FAILURE */

    return(0);			/* can't retry */
    }

  /* try re-shuffling the nodes to get what we want */

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    if (pnode->nd_ntype == NTYPE_CLUSTER) 
      {
      if (pnode->nd_flag != thinking)
        continue;
    
      if (pnode->nd_state & pass)
        continue;

      if ((skip == SKIP_EXCLUSIVE) && (vpreq < pnode->nd_nsnfree))
        continue;

      if ((skip == SKIP_ANYINUSE) &&
          (vpreq < (pnode->nd_nsnfree + pnode->nd_nsnshared)))
        continue;

      if (!hasprop(pnode,glorf))
        continue;

      pnode->nd_flag = conflict;

      /* Ben Webb patch (CRI 10/06/03) */

      found = search(
        pnode->nd_first, 
        pnode->nd_needed, 
        skip,
        pnode->nd_order, 
        depth);

      pnode->nd_flag = thinking;

      if (found) 
        {
        mark(pnode,glorf);

        pnode->nd_needed = vpreq;
        pnode->nd_order  = order;
		
        /* SUCCESS */
	
        return(1);
        }
      }
    }    /* END for (i) */

  /* FAILURE */

  return(0);	/* not found */
  }  /* END search() */





/*
**	Parse a number in a spec.
**	Return 0 if okay, 1 if no number exists, -1 on error
*/

static int number(

  char **ptr,
  int   *num)

  {
  char	holder[80];
  int	i = 0;
  char	*str = *ptr;

  while (isdigit(*str))
    holder[i++] = *str++;

  if (i == 0)
    {
    return(1);
    }

  holder[i] = '\0';

  if ((i = atoi(holder)) == 0) 
    {
    sprintf(log_buffer,"zero illegal");

    return(-1);
    }

  *ptr = str;
  *num = i;

  return(0);
  }  /* END number() */




/*
**	Check string to see if it is a legal property name.
**	If not, return 1.
**	*prop set to static char array containing the properity,
**	must be copied.
*/

static int property(

  char	**ptr,
  char	**prop)

  {
  static	char	name[80];
  char*	str = *ptr;
  int	i = 0;

  if (!isalpha(*str)) 
    {
    sprintf(log_buffer,
      "first character of property (%s) not a letter", 
      str);

    return(1);
    }

  while (isalnum(*str) || *str == '-' || *str == '.' || *str == '=' || *str == '_')
    name[i++] = *str++;

  name[i] = '\0';

  *prop = (i == 0) ? NULL : name;

  /* skip over "/vp_number" */

  if (*str == '/') 
    {
    do 
      {
      str++;
      } while (isdigit(*str));
    }

  *ptr = str;

  return(0);
  }  /* END property() */





/*
**	Create a property list from a string.
**	Return 0 if all is well, 1 otherwise.
*/

static int proplist(

  char		**str,
  struct prop	**plist,
  int		 *node_req)

  {
  struct prop	*pp;
  char		*pname;
  char		*pequal;

  *node_req = 1;	/* default to 1 processor per node */

  for (;;) 
    {
    if (property(str, &pname))
      {
      return(1);
      }

    if (pname == NULL)
      break;

    if ((pequal = strchr(pname,(int)'=')) != NULL) 
      {  
      /* special property */
			
      /* identify the special property and place its value */
      /* into node_req 					 */

      *pequal = '\0';

      if (strcmp(pname,"ppn") == 0) 
        {
        pequal++;

        if ((number(&pequal, node_req) != 0) || (*pequal != '\0'))
          {
          return(1);
          }
        } 
      else 
        {
        return(1);	/* not recognized - error */
        }
      } 
    else 
      {
      pp = (struct prop *)malloc(sizeof(struct prop));

      pp->mark = 1;
      pp->name = strdup(pname);
      pp->next = *plist;

      *plist = pp;
      }

    if (**str != ':')
      break;

    (*str)++;
    }  /* END for(;;) */

  return 0;
  }  /* END proplist() */




/*
 *	Evaluate one element in a node spec.
 *
 *	Return 1 if it can be satisfied
 *	       0 if it cannot be completly satisfied. (not used now)
 *	      -1 if error - can never be satisfied.
 */

static int listelem(

  char **str,
  int    order)

  {
  int	num = 1;
  int	i, hit;
  int	ret = -1;
  struct prop	*prop = NULL;
  struct pbsnode *pnode;
  int	node_req = 1;

  if ((i = number(str, &num)) == -1)	/* get number */
    {
    return(ret);
    }

  if (i == 0) 
    {				/* number exists */
    if (**str == ':') 
      {		/* there are properties */
      (*str)++;

      if (proplist(str,&prop,&node_req))
        {
        return(ret);
        }
      } 
    }
  else 
    {					/* no number */
    if (proplist(str,&prop,&node_req))
      {
      /* must be a prop list with no number in front */
      
      return(ret);
      }
    }

  /* count number of nodes with the requested property */

  hit = 0;

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    if (pnode->nd_ntype == NTYPE_CLUSTER) 
      {
      if (hasprop(pnode,prop) && hasppn(pnode,node_req,SKIP_NONE))
        hit++;

      if (hit == num) 
        {
        break;		/* found enough  */
        }
      }
    }

  if (hit < num)			/* can never be satisfied */
    goto done;

  /*
  ** Find an initial set of nodes to satisfy the request.
  ** Go ahead and use any nodes no mater what state they are in.
  */

  for (i = 0;i < num;i++) 
    {
    if (search(prop,node_req,SKIP_NONE,order,0))
      continue;

    goto done;		/* can never be satisfied */
    }

  ret = 1;

done:

  free_prop(prop);

  return(ret);
  }  /* END listelem() */





/*
**	Add the "global" spec to every sub-spec in "spec".
**      RETURNS:  allocated string buffer (must be freed externally) 
*/

static char *mod_spec(

  char *spec,    /* I */
  char *global)  /* I */

  {
  char  *line;
  char	*cp;
  int    len;
  int    nsubspec;

  nsubspec = 1;

  for (cp = spec;*cp != '\0';cp++)
    {
    if (*cp == '+') 
      {
       nsubspec++;
      }
    }

  len = strlen(global);

  line = malloc(nsubspec * (len + 1) + strlen(spec) + 1);

  cp = line;

  while (*spec) 
    {
    if (*spec == '+') 
      {
      *cp++ = ':';

      strcpy(cp,global);

      cp += len;
      }

    *cp++ = *spec++;
    }

  *cp++ = ':';

  strcpy(cp,global);

  return(line);
  }  /* END mod_spec() */




/* cntjons - count jobs on (shared) nodes */

static int cntjons(

  struct pbsnode *pn)

  {
  struct pbssubn *psn;
  int ct = 0;
  int n;
  struct jobinfo *pj;

  psn = pn->nd_psn;

  for (n = 0;n < pn->nd_nsn;++n) 
    {
    pj = psn->jobs;

    while (pj) 
      {
      ++ct;

      pj = pj->next;
      }

    psn = psn->next;
    }

  return(ct);
  }




/*
 * nodecmp - compare two nodes for sorting 
 *	For "exclusive", depending on setting of node_order attribute:
 *	    pack:    put free node with fewest non-zero free VPs in node first
 *	    scatter: put free node with most fre VPs first
 *	For "shared", put current shared with fewest jobs first,
 *		then free nodes, and others last
 */


#define BIG_NUM 32768 /* used only in nodecmp() */

static int nodecmp(

  const	void *aa,
  const	void *bb)

  {
  struct pbsnode	*a = *(struct pbsnode **)aa;
  struct pbsnode	*b = *(struct pbsnode **)bb;
  int	aprim, bprim;

  /* exclusive is global */

  if (exclusive) 
    {	
    /* best is free */

    if (server.sv_attr[(int)SRV_ATR_NodePack].at_val.at_long) 
      {
      /* pack - fill up nodes first */

      aprim = (a->nd_nsnfree > 0) ? a->nd_nsnfree : BIG_NUM;

      bprim = (b->nd_nsnfree > 0) ? b->nd_nsnfree : BIG_NUM;
      } 
    else 
      {
      /* scatter - spread amoung nodes first */

      aprim = a->nd_nsn - a->nd_nsnfree;
      bprim = b->nd_nsn - b->nd_nsnfree;
      }
    } 
  else 
    {		
    /* best is shared with fewest jobs */

    aprim = (a->nd_state == INUSE_JOBSHARE) ? 
      cntjons(a) :
      ((a->nd_state == INUSE_FREE) ? 5 : 1000);

    bprim = (b->nd_state == INUSE_JOBSHARE) ? 
      cntjons(b) :
      ((b->nd_state == INUSE_FREE) ? 5 : 1000);
    }

  if (aprim == bprim) 
    {
    return (a->nd_nprops - b->nd_nprops);
    } 

  return (aprim - bprim);
  }  /* END nodecmp() */





/*
 *	Test a node specification.  
 *
 *	Return >0 - number of nodes counted in the spec if it works,
 *	        0 - if it cannot be satisfied,
 *	       -1 - if it can never be satisfied.
 *	Okay to bail early if "early" is true.
 *	VPs selected are marked "thinking"
 */

static int node_spec(

  char	*spec,       /* I */
  int	 early,      /* I (boolean) */
  int    exactmatch) /* I (boolean) */

  {
  static char	id[] = "node_spec";

  struct pbsnode *pnode;
  struct pbssubn *snp;
  char	*str, *globs, *cp, *hold;
  int	i, num;
  int	rv;
  static char shared[] = "shared";

  if (LOGLEVEL >= 6)
    {
    DBPRT(("%s: entered spec=%s\n", 
      id, 
      spec))
    }

  exclusive = 1;	/* by default, nodes (VPs) are requested exclusively */

  spec = strdup(spec);

  if ((globs = strchr(spec,'#')) != NULL) 
    {
    *globs++ = '\0';

    globs = strdup(globs);

    while ((cp = strrchr(globs,'#')) != NULL) 
      {
      *cp++ = '\0';

      if (strcmp(cp,shared) != 0) 
        {
        hold = mod_spec(spec,cp);

        free(spec);

        spec = hold;
        } 
      else 
        {
        exclusive = 0;
        }
      }

    if (strcmp(globs,shared) != 0) 
      {
      hold = mod_spec(spec,globs);

      free(spec);

      spec = hold;
      } 
    else 
      {
      exclusive = 0;
      }

    free(globs);
    }

  str = spec;
		
  num = ctnodes(str);

  if (num > svr_clnodes) 
    {
    free(spec);

    if (LOGLEVEL >= 6)
      {
      DBPRT(("%s: requested nodes exceeds available cluster nodes (%d > %d)\n",
        id,
        num,
        svr_clnodes))
      }

    return(-1);
    }

  /*
   * if SRV_ATR_NodePack set (true or false), then
   * sort nodes by state, number of VPs and number of attributes;
   * otherwise, leave unsorted
   */

  if (server.sv_attr[(int)SRV_ATR_NodePack].at_flags & ATR_VFLAG_SET) 
    {
    qsort(pbsndlist,svr_totnodes,sizeof(struct pbsnode *),nodecmp);
    }

  /* reset subnodes (VPs) to okay */

  svr_numnodes = 0;

  for (i=0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    if (LOGLEVEL >= 6)
      {
      DBPRT(("%s: %s nsn %d, nsnfree %d\n", 
        id, 
        pnode->nd_name,
        pnode->nd_nsn, 
        pnode->nd_nsnfree))
      }

    pnode->nd_flag   = okay;
    pnode->nd_needed = 0;

    for (snp = pnode->nd_psn;snp != NULL;snp = snp->next) 
      {
      snp->flag = okay;

      if (LOGLEVEL >= 6)
        {
        DBPRT(("%s: %s/%d inuse 0x%x nprops %d\n", 
          id,
          pnode->nd_name,
          snp->index,
          snp->inuse,
          pnode->nd_nprops))
        }
      }

    if ((pnode->nd_ntype == NTYPE_CLUSTER) &&
       ((pnode->nd_state & (INUSE_OFFLINE|INUSE_DOWN|INUSE_RESERVE|INUSE_JOB)) == 0))
      {
      svr_numnodes++;
      }
    }    /* END for (i = 0) */

  /*
   * Make first pass at finding nodes to allocate.
   * process each subspec (piece between '+'s)
   */

  for (i = 1;;i++) 
    {
    if ((rv = listelem(&str,i)) <= 0) 
      {
      free(spec);

      return(rv);
      }

    if (*str != '+')
      break;

    str++;
    }

  i = (int)*str;

  free(spec);

  if (i != 0)					/* garbled list */
    {
    /* failure */

    if (LOGLEVEL >= 6)
      {
      DBPRT(("%s: request is corrupt\n",
        id))
      }

    return(-1);
    }

  if ((num > svr_numnodes) && early)	/* temp fail, not available */
    {
    if (LOGLEVEL >= 6)
      {
      DBPRT(("%s: inadequate nodes currently in cluster (%d > %d)\n",
        id,
        num,
        svr_numnodes))
      }

    return(0);
    }

  /*
   * 	At this point we know the spec is legal.
   *	Here we find a replacement for any nodes chosen above
   *	that are already inuse.
   */

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    if (pnode->nd_ntype != NTYPE_CLUSTER)
      continue;	

    if (pnode->nd_flag != thinking)  /* thinking is global */
      continue;

    if (pnode->nd_state == INUSE_FREE)  
      {
      if (pnode->nd_needed <= pnode->nd_nsnfree) 
        continue;

      if (!exclusive && 
         (pnode->nd_needed < pnode->nd_nsnfree + pnode->nd_nsnshared))
        continue;		/* shared node */
      } 
    else 
      {
      if (!exclusive && 
         (pnode->nd_needed <= pnode->nd_nsnfree + pnode->nd_nsnshared))
        continue;		/* shared node */
      }

    /* otherwise find replacement node */

    /* Ben Webb search patch applied (CRI 10/03/03) */

    pnode->nd_flag = okay;

    if (search(
        pnode->nd_first,
        pnode->nd_needed, 
        (exclusive != 0) ? SKIP_ANYINUSE : SKIP_EXCLUSIVE,
        pnode->nd_order,0)) 
      {
      continue;
      }

    if (early != 0)
      {
      /* specified node not available and replacement cannot be located */

      if (LOGLEVEL >= 6)
        {
        DBPRT(("%s: cannot locate requested resource '%s'\n",
          id,
          pnode->nd_name))
        }

      return(0);
      }

    num = 0;
    }  /* END for (i) */

  return(num);	/* spec ok */
  }  /* END node_spec() */





/*
 *	set_nodes() - Call node_spec() to allocate nodes then set them inuse.
 *	Build list of allocated nodes to pass back in rtnlist.
 *      Return: PBS error code 
 */

int set_nodes(

  job   *pjob,    /* I */
  char	*spec,    /* I */
  char **rtnlist) /* O */

  {
  struct howl {
    char *name;
    int	  order;
    int	  index;
    struct howl *next;
    } *hp, *hlist, *curr, *prev, *nxt;

  int     i;
  short   newstate;

  int     NCount;

  static char	*id = "set_nodes";

  struct pbsnode *pnode;
  struct pbssubn *snp;
  char	*nodelist;

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer,"allocating nodes for job %s with node expression '%s'",
      pjob->ji_qs.ji_jobid,
      spec);

    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  /* allocate nodes */

  if ((i = node_spec(spec,1,1)) == 0)	/* check spec */
    {
    /* no resources located, request failed */

    return(PBSE_RESCUNAV);
    }

  if (i < 0)
    {
    /* request failed, corrupt request */

    return(PBSE_UNKNODE);
    }

  /* i indicates number of matching nodes */

  if (exclusive)        /* exclusive is global */
    svr_numnodes -= i;

  hlist = NULL;

  newstate = exclusive ? INUSE_JOB : INUSE_JOBSHARE;

  for (i = 0;i < svr_totnodes;i++) 
    {
    struct jobinfo *jp;

    pnode = pbsndlist[i];

    if (pnode->nd_flag != thinking)
      continue;			/* skip this one */

    /* within the node, check each subnode */

    for (snp = pnode->nd_psn;snp && pnode->nd_needed;snp = snp->next) 
      {
      if (exclusive) 
        {
        if (snp->inuse != INUSE_FREE)
          continue;
        } 
      else 
        {
        if ((snp->inuse != INUSE_FREE) && (snp->inuse != INUSE_JOBSHARE))
          continue;
        }

      /* Mark subnode as being IN USE */

      if (LOGLEVEL >= 5)
        {
        sprintf(log_buffer,"allocated node %s/%d to job %s",
          pnode->nd_name, 
          snp->index,
          pjob->ji_qs.ji_jobid);

        log_record(
          PBSEVENT_SCHED,
          PBS_EVENTCLASS_REQUEST,
          id,
          log_buffer);
        }

      if (snp->inuse == INUSE_FREE) 
        {
        snp->inuse = newstate;

        pnode->nd_nsnfree--;		/* reduce free count */

        if (!exclusive)
          pnode->nd_nsnshared++;
        }

      jp  = (struct jobinfo *)malloc(sizeof(struct jobinfo));

      jp->next = snp->jobs;

      snp->jobs = jp;

      jp->job = pjob;

      /* build list of nodes ordered to match request */

      curr = (struct howl *)malloc(sizeof(struct howl));
      curr->order = pnode->nd_order;
      curr->name  = pnode->nd_name;
      curr->index = snp->index;

      for (prev = NULL,hp = hlist;hp;prev = hp,hp = hp->next) 
        {
        if (curr->order <= hp->order)
          break;
        }

      curr->next = hp;

      if (prev == NULL)
        hlist = curr;
      else
        prev->next = curr;

      --pnode->nd_needed;
      }  /* END for (snp) */

    if (pnode->nd_nsnfree == 0)	    /* if no free VPs, set node state */
      pnode->nd_state = newstate;
    }    /* END for (i) */

  if (hlist == NULL)
    {
    if (LOGLEVEL >= 1)
      {
      sprintf(log_buffer,"ALERT:  no nodes can be allocated to job %s",
        pjob->ji_qs.ji_jobid);

      log_record(
        PBSEVENT_SCHED,
        PBS_EVENTCLASS_REQUEST,
        id,
        log_buffer);
      }

    return(PBSE_RESCUNAV);
    }

  pjob->ji_qs.ji_svrflags |= JOB_SVFLG_HasNodes;  /* indicate has nodes */

  /* build list of allocated nodes */

  i = 1;  /* first, size list */

  for (hp = hlist;hp != NULL;hp = hp->next) 
    {
    i += (strlen(hp->name) + 6);
    }

  nodelist = malloc(++i);

  *nodelist = '\0';

  /* now copy in name+name+... */

  NCount = 0;

  for (hp = hlist;hp;hp = nxt) 
    {
    NCount++;

    sprintf(nodelist + strlen(nodelist),"%s/%d+",
      hp->name, 
      hp->index);

    nxt = hp->next;
   
    free(hp);
    }

  *(nodelist + strlen(nodelist) - 1) = '\0';	/* strip trailing + */

  *rtnlist = nodelist;

  if (LOGLEVEL >= 3)
    {
    snprintf(log_buffer,sizeof(log_buffer),"job %s allocated %d nodes (nodelist=%s)",
      pjob->ji_qs.ji_jobid,
      NCount,
      nodelist);

    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  /* SUCCESS */

  return(PBSE_NONE);
  }  /* END set_nodes() */





/*
 * node_avail_complex - 
 *		*navail is set to number available
 *		*nalloc is set to number allocated
 *		*nresvd is set to number reserved 
 *		*ndown  is set to number down/offline
 */

int node_avail_complex(

  char	*spec,		/* I - node spec */
  int 	*navail,	/* O - number available */
  int	*nalloc,	/* O - number allocated */
  int	*nresvd,	/* O - number reserved  */
  int	*ndown)		/* O - number down      */

  {
  int	holdnum;
  int	ret;

  holdnum = svr_numnodes;

  ret = node_spec(spec,1,0);

  svr_numnodes = holdnum;

  *navail = ret;
  *nalloc = 0;
  *nresvd = 0;
  *ndown  = 0;

  return(ret);
  }  /* END node_avail_complex() */





/*
 * node_avail - report if nodes requested are available
 *	Does NOT even consider Time Shared Nodes 
 *
 *	Return 0 when no error in request and
 *		*navail is set to number available
 *		*nalloc is set to number allocated
 *		*nresvd is set to number reserved 
 *		*ndown  is set to number down/offline
 *	     !=0 error number when error in request
 */

int node_avail(

  char	*spec,		/* I  - node spec */
  int 	*navail,	/* O - number available */
  int	*nalloc,	/* O - number allocated */
  int	*nresvd,	/* O - number reserved  */
  int	*ndown)		/* O - number down      */

  {
  char	*id = "node_avail";
  int	i;
  int	j;
  int	holdnum;
  struct	pbsnode	*pn;
  struct  pbssubn *psn;
  char    *pc;
  struct prop *prop = NULL;
  register int xavail;
  register int xalloc;
  register int xresvd;
  register int xdown;
  int          node_req = 1;

  if (spec == NULL) 
    {
    log_event(PBSEVENT_ADMIN,PBS_EVENTCLASS_SERVER,id,"no spec");

    return(RM_ERR_NOPARAM);
    }

  pc = spec;

  if ((strchr(spec,(int)'+') == NULL) && (number(&pc,&holdnum) == 1)) 
    {
    /* A simple node spec - reply with numbers of avaiable,	*/
    /* allocated, reserved, and down nodes that match the	*/
    /* the spec, null or simple number means all		*/

    xavail = 0;
    xalloc = 0;
    xresvd = 0;
    xdown  = 0;

    /* find number of a specific type of node */

    if (*pc)
      {
      if (proplist(&pc,&prop,&node_req))
        {
        return(RM_ERR_BADPARAM);
        }
      }

    for (i = 0;i < svr_totnodes;i++) 
      {
      pn = pbsndlist[i];

      if ((pn->nd_ntype == NTYPE_CLUSTER) && hasprop(pn,prop)) 
        {
        if (pn->nd_state & (INUSE_OFFLINE|INUSE_DOWN))
          ++xdown;
        else if (hasppn(pn,node_req,SKIP_ANYINUSE))
          ++xavail;
        else if (hasppn(pn,node_req,SKIP_NONE)) 
          {
          /* node has enough processors, are they busy or reserved? */

          j = 0;

          for (psn = pn->nd_psn;psn;psn = psn->next) 
            {
            if (psn->inuse & INUSE_RESERVE)
              j++;
            }

          if (j >= node_req)
            ++xresvd;
          else
            ++xalloc;
          }
        }
      }    /* END for (i) */

    free_prop(prop);

    *navail = xavail;
    *nalloc = xalloc;
    *nresvd = xresvd;
    *ndown  = xdown;

    return(0);
    }
  else if (number(&pc,&holdnum) == -1) 
    {
    /* invalid spec */

    return(RM_ERR_BADPARAM);
    }

  /* not a simple spec - determine if supplied complex	*/
  /* node spec can be satisified from avail nodes	*/
  /* navail set to >0 if can be satified now		*/
  /*		  0 if not now but possible		*/
  /*		 -l if never possible			*/

  node_avail_complex(spec,navail,nalloc,nresvd,ndown);

  return(0);
  }  /* END node_avail() */




/*
 * node_reserve - Reserve nodes 
 *	Cannot reserve Time Shared Nodes
 *
 *	Returns: >0 - reservation succeeded, number of nodes reserved
 *		  0 - None or partial reservation
 *		 -1 - requested reservation impossible
 */

int node_reserve(

  char       *nspec, /* In     - a node specification */
  resource_t  tag)   /* In/Out - tag for resource if reserved */

  {
  static	char	id[] = "node_reserve";
  int		 nrd;
  struct  pbsnode *pnode;
  struct	pbssubn	*snp;
  int		 ret_val;
  int		 i;

  DBPRT(("%s: entered\n", 
    id))

  if ((nspec == NULL) || (*nspec == '\0')) 
    {
    log_event(
      PBSEVENT_ADMIN, 
      PBS_EVENTCLASS_SERVER, 
      id, 
      "no spec");

    return(-1);
    }

  if ((ret_val = node_spec(nspec,0,0)) >= 0) 
    {
    /*
    ** Zero or more of the needed Nodes are available to be 
    ** reserved.
    */

    for (i = 0;i < svr_totnodes;i++) 
      {
      pnode = pbsndlist[i];

      if (pnode->nd_flag != thinking)
        continue;			/* skip this one */

      nrd = 0;

      for (snp = pnode->nd_psn;snp && pnode->nd_needed;snp = snp->next) 
        {
        if (snp->inuse == INUSE_FREE) 
          {
          DBPRT(("hold %s/%d\n", 
            pnode->nd_name, 
            snp->index))

          snp->inuse |= INUSE_RESERVE;
          snp->allocto = tag;
 
          pnode->nd_nsnfree--;

          --pnode->nd_needed;

          ++nrd;
          }
        }

      if (nrd == pnode->nd_nsn)
        pnode->nd_state = INUSE_RESERVE;
      }
    } 
  else 
    {
    /* could never satisfy the reservation */

    sprintf(log_buffer,"can never reserve %s",
      nspec);

    log_record(
      PBSEVENT_SCHED, 
      PBS_EVENTCLASS_REQUEST, 
      id,
      log_buffer);
    }

  return(ret_val);
  }  /* END node_reserve() */






/*
 * is_ts_node - does the nodestr specify a single time share node?
 *	0 - yes
 *	1 - no, not a ts node or more than one node (name will not match)
 */

int is_ts_node(

  char *nodestr)

  {
  struct pbsnode *np;
  int		i;

  for (i = 0;i < svr_totnodes;i++) 
    {
    np = pbsndmast[i];

    if (((np->nd_state & INUSE_DELETED) == 0) &&
         (np->nd_ntype == NTYPE_TIMESHARED)) 
      {
      if (!strcmp(nodestr,np->nd_name))
        {
        return(0);
        }
      }
    }

  return(1);
  }  /* END is_ts_node() */





/*
 * find_ts_node - find first up time-shared node
 *
 *	returns name of node or null
 */

char *find_ts_node()

  {
  struct pbsnode *np;
  int             i;

  for (i = 0;i < svr_totnodes;i++) 
    {
    np = pbsndmast[i];

    if ((np->nd_ntype == NTYPE_TIMESHARED) &&
       ((np->nd_state & (INUSE_DOWN|INUSE_DELETED|INUSE_OFFLINE)) == 0)) 
      {
      return(np->nd_name);
      }
    }

  return(NULL);
  }





/*
 * free_nodes - free nodes allocated to a job
 */

void free_nodes(

  job *pjob)  /* I (modified) */

  {
  static char	id[] = "free_nodes";
  struct pbssubn *np;
  struct pbssubn *xnp;
  struct pbsnode *pnode;
  struct jobinfo *jp, *prev;
  int             i;

  if (LOGLEVEL >= 3)
    {
    sprintf(log_buffer,"freeing nodes for job %s",
      pjob->ji_qs.ji_jobid);

    log_record(
      PBSEVENT_SCHED,
      PBS_EVENTCLASS_REQUEST,
      id,
      log_buffer);
    }

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndlist[i];

    for (np = pnode->nd_psn;np != NULL;np = np->next) 
      {
      for (prev = NULL,jp = np->jobs;jp != NULL;prev = jp,jp = jp->next) 
        {
        if (jp->job != pjob)
          continue;

        if (LOGLEVEL >= 4)
          {
          sprintf(log_buffer,"freeing node %s/%d for job %s",
            pnode->nd_name,
            np->index,
            pjob->ji_qs.ji_jobid);

          log_record(
            PBSEVENT_SCHED,
            PBS_EVENTCLASS_REQUEST,
            id,
            log_buffer);
          }

        if (prev == NULL)
          np->jobs = jp->next;
        else
          prev->next = jp->next;

        free(jp);

        if (np->jobs == NULL) 
          {
          pnode->nd_nsnfree++;	/* up count of free */

          if (LOGLEVEL >= 6)
            {
            DBPRT(("%s: upping free count to %d\n", 
              id, 
              pnode->nd_nsnfree))
            }

          if (np->inuse & INUSE_JOBSHARE)
            pnode->nd_nsnshared--;

          np->inuse &= ~(INUSE_JOB|INUSE_JOBSHARE);

          for (xnp = pnode->nd_psn;xnp;xnp = xnp->next) 
            {
            if (xnp->inuse & (INUSE_JOB|INUSE_JOBSHARE))
              break;
            }

          pnode->nd_state &= ~(INUSE_JOB|INUSE_JOBSHARE);
          }

        break;
        }
      }
    }    /* END for (i) */

  pjob->ji_qs.ji_svrflags &= ~JOB_SVFLG_HasNodes;

  return;
  }  /* END free_nodes() */





/*
 * set_one_old - set a named node as allocated to a job
 */

static void set_one_old(

  char *name,
  job  *pjob,
  int   shared)	/* how used flag, either INUSE_JOB or INUSE_JOBSHARE */

  {
  int		i;
  int		index;
  struct pbsnode *pnode;
  struct pbssubn *snp;
  struct jobinfo *jp;
  char	       *pc;
	
  if ((pc = strchr(name,(int)'/'))) 
    {
    index = atoi(pc + 1);

    *pc = '\0';
    } 
  else
    {
    index = 0;
    }

  for (i = 0;i < svr_totnodes;i++) 
    {
    pnode = pbsndmast[i];

    if (strcmp(name,pnode->nd_name) == 0) 
      {
      /* Mark node as being IN USE ...  */

      if (pnode->nd_ntype == NTYPE_CLUSTER) 
        {
        for (snp = pnode->nd_psn;snp;snp = snp->next) 
          {
          if (snp->index == index) 
            {
            snp->inuse = shared;

            jp = (struct jobinfo *)malloc(sizeof(struct jobinfo));

            jp->next = snp->jobs;

            snp->jobs = jp;

            jp->job = pjob;

            if (--pnode->nd_nsnfree == 0)
              pnode->nd_state = shared;

            return;
            }
          }
        }
      }
    }

  return;
  }




	
/*
 * set_old_nodes - set "old" nodes as in use - called from pbsd_init()
 *	when recovering a job in the running state.
 */

void set_old_nodes(

  job *pjob)  /* I (modified) */

  {
  char *old;
  char *po;
  resource *presc;
  int   shared = INUSE_JOB;

  if ((pbsndmast != NULL) && 
      (pjob->ji_wattr[(int)JOB_ATR_exec_host].at_flags & ATR_VFLAG_SET)) 
    {
    /* are the nodes being used shared? Look in "neednodes" */

    presc = find_resc_entry(
      &pjob->ji_wattr[(int)JOB_ATR_resource],
      find_resc_def(svr_resc_def,"neednodes",svr_resc_size));

    if ((presc != NULL) && (presc->rs_value.at_flags & ATR_VFLAG_SET)) 
      {
      if ((po = strchr(presc->rs_value.at_val.at_str,'#'))) 
        {
        if (strstr(++po,"shared") != NULL) 
          shared = INUSE_JOBSHARE;
        }
      }

    old = strdup(pjob->ji_wattr[(int)JOB_ATR_exec_host].at_val.at_str);

    while ((po = strrchr(old,(int)'+')) != NULL) 
      {
      *po++ = '\0';

      set_one_old(po,pjob,shared);
      }

    set_one_old(old,pjob,shared);

    free(old);
    }

  return;
  }  /* END set_old_nodes() */


/* END node_manager.c */

