/*
*         OpenPBS (Portable Batch System) v2.3 Software License
* 
* Copyright (c) 1999-2000 Veridian Information Solutions, Inc.
* All rights reserved.
* 
* ---------------------------------------------------------------------------
* For a license to use or redistribute the OpenPBS software under conditions
* other than those described below, or to purchase support for this software,
* please contact Veridian Systems, PBS Products Department ("Licensor") at:
* 
*    www.OpenPBS.org  +1 650 967-4675                  sales@OpenPBS.org
*                        877 902-4PBS (US toll-free)
* ---------------------------------------------------------------------------
* 
* This license covers use of the OpenPBS v2.3 software (the "Software") at
* your site or location, and, for certain users, redistribution of the
* Software to other sites and locations.  Use and redistribution of
* OpenPBS v2.3 in source and binary forms, with or without modification,
* are permitted provided that all of the following conditions are met.
* After December 31, 2001, only conditions 3-6 must be met:
* 
* 1. Commercial and/or non-commercial use of the Software is permitted
*    provided a current software registration is on file at www.OpenPBS.org.
*    If use of this software contributes to a publication, product, or
*    service, proper attribution must be given; see www.OpenPBS.org/credit.html
* 
* 2. Redistribution in any form is only permitted for non-commercial,
*    non-profit purposes.  There can be no charge for the Software or any
*    software incorporating the Software.  Further, there can be no
*    expectation of revenue generated as a consequence of redistributing
*    the Software.
* 
* 3. Any Redistribution of source code must retain the above copyright notice
*    and the acknowledgment contained in paragraph 6, this list of conditions
*    and the disclaimer contained in paragraph 7.
* 
* 4. Any Redistribution in binary form must reproduce the above copyright
*    notice and the acknowledgment contained in paragraph 6, this list of
*    conditions and the disclaimer contained in paragraph 7 in the
*    documentation and/or other materials provided with the distribution.
* 
* 5. Redistributions in any form must be accompanied by information on how to
*    obtain complete source code for the OpenPBS software and any
*    modifications and/or additions to the OpenPBS software.  The source code
*    must either be included in the distribution or be available for no more
*    than the cost of distribution plus a nominal fee, and all modifications
*    and additions to the Software must be freely redistributable by any party
*    (including Licensor) without restriction.
* 
* 6. All advertising materials mentioning features or use of the Software must
*    display the following acknowledgment:
* 
*     "This product includes software developed by NASA Ames Research Center,
*     Lawrence Livermore National Laboratory, and Veridian Information 
*     Solutions, Inc.
*     Visit www.OpenPBS.org for OpenPBS software support,
*     products, and information."
* 
* 7. DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND. ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT
* ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT SHALL VERIDIAN CORPORATION, ITS AFFILIATED COMPANIES, OR THE
* U.S. GOVERNMENT OR ANY OF ITS AGENCIES BE LIABLE FOR ANY DIRECT OR INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
* OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
* EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <stddef.h>
#include <string.h>

#include "Long.h"
#include "Long_.h"

#undef strTouL

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

static unsigned x_val;
static const char letters[] = "abcdefghijklmnopqrstuvwxyz";
static char table[UCHAR_MAX + 1];

/*
 * strTouL - returns the unsigned Long value representing the string whose
 * 	     first character is *nptr, when interpreted as an integer in base,
 *	     base.
 *
 * 	If base is zero, the base of the integer is determined by the way the
 * 	string starts.  The string is interpreted as decimal if the first
 * 	character after leading white space and an optional sign is a digit
 * 	between 1 and 9, inclusive.  The string is interpreted as octal if the
 * 	first character after leading white space and an optional sign is the
 * 	digit "0" and the next character is not an "X" (either upper or lower
 * 	case).  The string is interpreted as hexidecimal if the first character
 * 	after leading white space and an optional sign is the digit "0",
 * 	followed by an "X" (either upper or lower case).
 *
 * 	If base is greater than 1 and less than the number of characters in the
 *	Long_dig array, it represents the base in which the number will be
 *	interpreted.  Characters for digits beyond 9 are represented by the
 *	letters of the alphabet, either upper case or lower case.
 */

u_Long strTouL(nptr, endptr, base)
    const char		*nptr;
    char		**endptr;
    int			base;
{
	unsigned	digit;
	u_Long		limit = 0, value;
	enum {
	    unknown1,
	    unknown2,
	    hex1,
	    hex2,
	    hex3,
	    known,
	    working,
	    overflow
	}		state;

	if (table[(unsigned char)'1'] != 1) {
		int	i;		/* Initialize conversion table */

		(void)memset(table, CHAR_MAX, sizeof(table));
		for (i = (int)strlen(Long_dig) - 1; i >= 0; i--)
			table[(unsigned char)Long_dig[i]] = i;
		for (i = (int)strlen(letters) - 1; i >= 0; i--)
			table[(unsigned char)letters[i]] = i + 10;
		x_val = table[(unsigned char)'x'];
	}
	if (nptr == NULL) {
		if (endptr != NULL)
		        *endptr = (char *)nptr;
	        return (0);
	}
	if (base < 0 || base == 1 || (size_t)base > strlen(Long_dig)) {
	        errno = EDOM;
		if (endptr != NULL)
		        *endptr = (char *)nptr;
	        return (0);
	}
	switch (base) {
	    case 0:
		state = unknown1;
		break;
	    case 16:
		state = hex1;
		break;
	    default:
		state = known;
	}
	while (isspace(*nptr++));
	Long_neg = FALSE;
	switch (*--nptr) {
	    case '-':
		Long_neg = TRUE;
	    case '+':
		nptr++;
	}
        value = 0;
	while ((digit = table[(unsigned char)*nptr++]) != CHAR_MAX) {
		switch (state) {
		    case unknown1:
			if (digit >= 10)
				goto done;
			if (digit == 0) {
				state = unknown2;
				break;
			}
			base = 10;
			state = working;
			limit = UlONG_MAX / 10;
			value = digit;
			break;
		    case unknown2:
			if (digit >= 8) {
				if (digit != x_val)
					goto done;
				base = 16;
				state = hex3;
				break;
			}
			base = 8;
			state = working;
			limit = UlONG_MAX / 8;
			value = digit;
			break;
		    case hex1:
			if ((int)digit >= base)
			        goto done;
			if (digit == 0) {
				state = hex2;
				break;
			}
			state = working;
			limit = UlONG_MAX / 16;
			value = digit;
			break;
		    case hex2:
			if (digit == x_val) {
				state = hex3;
				break;
			}
		    case hex3:
		    case known:
			if ((int)digit >= base)
				goto done;
			state = working;
			limit = UlONG_MAX / base;
			value = digit;
			break;
		    case working:
			if ((int)digit >= base)
			        goto done;
			if (value < limit) {
				value = value * base + digit;
				break;
			}
			if (value > limit ||
			    UlONG_MAX - (value *= base) < digit) {
				state = overflow;
				value = UlONG_MAX;
				errno = ERANGE;
				break;
			}
			value += digit;
			break;
		    case overflow:
			if ((int)digit >= base)
			        goto done;
		}
	}
    done:
	if (endptr != NULL) {
		if (state == hex3)
		        nptr--;
	        *endptr = (char *)--nptr;
	}
        if (Long_neg)
                return (-value);
        return (value);
}
